package com.kdgc.framework.dao.jpa;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import javax.persistence.LockModeType;

import com.kdgc.framework.dao.support.Filter;
import com.kdgc.framework.dao.support.Order;
import com.kdgc.framework.dao.support.Pageable;
import com.kdgc.framework.dao.support.Pagination;

/**
 * DAO 接口定义
 * @author BridgeBai
 *
 * @param <T>
 * @param <ID>
 */
public abstract interface IBaseDao<T, ID extends Serializable>
{
	
	/**
	 * 查找实体对象
	 * 
	 * @param id
	 *            ID
	 * @return 实体对象的代理，若不存在则返回null
	 */
	T getReference(ID id);
	
	/**
	 * 查找实体对象
	 * 
	 * @param id
	 *            ID
	 * @return 实体对象，若不存在则返回null
	 */
	T find(ID id);

	/**
	 * 查找实体对象
	 * 
	 * @param id
	 *            ID
	 * @param lockModeType
	 *            锁定方式
	 * @return 实体对象，若不存在则返回null
	 */
	T find(ID id, LockModeType lockModeType);

	/**
	 * 查找实体对象集合
	 * 
	 * @param first
	 *            起始记录
	 * @param count
	 *            数量
	 * @param filters
	 *            筛选
	 * @param orders
	 *            排序
	 * @return 实体对象集合
	 */
	List<T> findList(Integer first, Integer count, List<Filter> filters, List<Order> orders);

	/**
	 * 查找实体对象分页
	 * 
	 * @param pageable
	 *            分页信息
	 * @return 实体对象分页
	 */
	Pagination<T> findPage(Pageable pageable);

	/**
	 * 查询实体对象数量
	 * 
	 * @param filters
	 *            筛选
	 * @return 实体对象数量
	 */
	long count(Filter... filters);

	/**
	 * 持久化实体对象
	 * 
	 * @param entity
	 *            实体对象
	 */
	void persist(T entity);

	/**
	 * 合并实体对象
	 * 
	 * @param entity
	 *            实体对象
	 * @return 实体对象
	 */
	T merge(T entity);

	/**
	 * 批量持久化实体
	 * @param entitys
	 */
	void batchPersist(List<T> entitys);
	/**
	 * 批量更新实体
	 * @param entitys
	 */
	void batchMerge(List<T> entitys);
	/**
	 * 移除实体对象
	 * 
	 * @param entity
	 *            实体对象
	 */
	void remove(T entity);

	/**
	 * 刷新实体对象
	 * 
	 * @param entity
	 *            实体对象
	 */
	void refresh(T entity);

	/**
	 * 刷新实体对象
	 * 
	 * @param entity
	 *            实体对象
	 * @param lockModeType
	 *            锁定方式
	 */
	void refresh(T entity, LockModeType lockModeType);

	/**
	 * 获取实体对象ID
	 * 
	 * @param entity
	 *            实体对象
	 * @return 实体对象ID
	 */
	ID getIdentifier(T entity);

	/**
	 * 判断是否为托管状态
	 * 
	 * @param entity
	 *            实体对象
	 * @return 是否为托管状态
	 */
	boolean isManaged(T entity);

	/**
	 * 设置为游离状态
	 * 
	 * @param entity
	 *            实体对象
	 */
	void detach(T entity);

	/**
	 * 锁定实体对象
	 * 
	 * @param entity
	 *            实体对象
	 * @param lockModeType
	 *            锁定方式
	 */
	void lock(T entity, LockModeType lockModeType);

	/**
	 * 清除缓存
	 */
	void clear();

	/**
	 * 同步数据
	 */
	void flush();
	
	/**
	 * 执行JPQl语句查询
	 * @param queryString JPQL语句
	 * @return
	 */
	public List<T> findByJPQL(final String queryString);

	/**
	 * 执行JPQl语句查询
	 * @param queryString JPQL语句
	 * @param value  参数值
	 * @return
	 */
	public List<T> findByJPQL(final String queryString, final Object value);

	/**
	 * 执行JPQl语句查询
	 * @param queryString JPQL语句
	 * @param values  参数值
	 * @return
	 */
	public List<T> findByJPQL(final String queryString, final Object[] values);
	/**
	 * 执行JPQL更新或删除语句
	 * @param queryString JPQL语句
	 * @return 执行语句的行数
	 */
	public int bulkUpdate(String queryString);

	/**
	 * 执行JPQL更新或删除语句
	 * @param queryString JPQL语句
	 * @param value  参数值
	 * @return 执行语句的行数
	 */
	public int bulkUpdate(String queryString, Object value);
	
	/**
	 * 执行JPQL更新或删除语句
	 * @param queryString JPQL语句
	 * @param values  参数值
	 * @return 执行语句的行数
	 */
	public int bulkUpdate(final String queryString, final Object[] values);
	/**
	 * 执行JPQL查询语句 返回分页对象
	 * @param queryString JPQL语句
	 * @param params  参数值
	 * @param pageable 分页对象
	 * @return 执行语句的行数
	 */
	public  Pagination<T> pagerJPQLList(String jpql, Object[] params,Pageable pageable);
	/**
	 * 执行JPQL查询语句 返回分页对象
	 * @param queryString JPQL语句
	 * @param params  参数值
	 * @param pageable 分页对象
	 * @return 执行语句的行数
	 */
	public  Pagination<T> pagerJPQLList(String jpql, Map<String,?> params,Pageable pageable);
	/**
	 * 功能：执行JPQL查询语句 返回分页对象
	 * @author BridgeBai
	 * @date   2015年11月30日下午4:53:30
	 * @param jpql
	 * @param clazz
	 * @param params
	 * @param pageable
	 * @return
	 */
	public <E> Pagination<E> pagerJPQLList(String jpql,Class<E> clazz, Map<String,?> params,Pageable pageable);
}
