package com.kdgc.framework.core.support;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.UUID;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.handler.HandlerInterceptorAdapter;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.kdgc.framework.web.common.ResponseData;


/**
 * 重复请求token处理
 * @author BridgeBai
 * @date   2016年5月16日下午2:02:51
 */
public class TokenInterceptor extends HandlerInterceptorAdapter {

	private static Logger logger=LoggerFactory.getLogger(TokenInterceptor.class);

	/** 
	 * 保存token值的默认命名空间 
	*/  
	private static final String TOKEN_NAMESPACE = "kdgc.tokens";  
	      
	/** 
	* 持有token名称的字段名 
	*/  
	private static final String TOKEN_NAME_FIELD = "kdgc.token.name";  

	protected final ObjectMapper mapper;
	/** Cache */
	protected final Ehcache cache ;
	
	public TokenInterceptor() {
		mapper = new ObjectMapper();
		cache = CacheManager.create().getEhcache(TOKEN_NAMESPACE);
	}
	@Override
	public boolean preHandle(HttpServletRequest request,HttpServletResponse response, Object handler) throws Exception {
		 HandlerMethod handlerMethod = (HandlerMethod) handler;
         Method method = handlerMethod.getMethod();
         TokenAnnotation annotation = method.getAnnotation(TokenAnnotation.class);
         if (annotation != null) {
             boolean saveToken = annotation.saveToken();
             if (saveToken) {
            	 setToken(request);  
            	 return true;  
             }
             boolean removeToken = annotation.removeToken();
             if (removeToken) {
            	 return handleToken(request, response, handler);
             }
         }
         return true;
	}
	
	protected boolean handleToken(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception  
	{  
		synchronized (this) {
			if (!validToken(request)) {
				writeMessageUtf8(response,ResponseData.warn("请不要重复提交!"));
				return false;
			}
		}
		return true;
	 }  

	protected void writeMessageUtf8(HttpServletResponse response,ResponseData responseData) throws IOException {
		try {
			String json = mapper.writeValueAsString(responseData);
			response.setContentType("text/html;charset=UTF-8");
			response.getWriter().write(json);
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		}
	}

	/** 
	* 使用随机字串作为token名字保存token 
	*  
	* @param request 
	* @return token 
	*/  
	protected String setToken(HttpServletRequest request)  
	{  
	  return setToken(request, UUID.randomUUID().toString());  
	}  

	protected String setToken(HttpServletRequest request, String tokenName)  
	{  
		 String token = UUID.randomUUID().toString();
		 setCacheToken(request, tokenName, token);  
		 return token;  
	}  

	
	protected void setCacheToken(HttpServletRequest request,String tokenName, String token) {
		cache.put(new net.sf.ehcache.Element(tokenName, token));
		request.setAttribute(TOKEN_NAME_FIELD, tokenName);
		request.setAttribute(tokenName, token);
	}

	/** 
	* 从请求域中获取给定token名字的token值 
	*  
	* @param tokenName 
	* @return the token String or null, if the token could not be found 
	*/  
	protected String getToken(HttpServletRequest request, String tokenName)  
	{  
		if (tokenName == null) {
			return null;
		}
		Map<String,String[]> params = request.getParameterMap();
		String[] tokens = params.get(tokenName);
		String token;
		if ((tokens == null) || (tokens.length < 1)) {
			logger.warn("Could not find token mapped to token name "+ tokenName);
			return null;
		}
		token = tokens[0];
		return token;
	}  

	/** 
	* 从请求参数中获取token名字 
	*  
	* @return the token name found in the params, or null if it could not be found 
	*/  
	protected String getTokenName(HttpServletRequest request)  
	{  
		Map<String,String[]> params = request.getParameterMap();
		if (!params.containsKey(TOKEN_NAME_FIELD)) {
			logger.warn("Could not find token name in params.");
			return null;
		}
		String[] tokenNames = params.get(TOKEN_NAME_FIELD);
		String tokenName;
		if ((tokenNames == null) || (tokenNames.length < 1)) {
			logger.warn("Got a null or empty token name.");
			return null;
		}
		tokenName = tokenNames[0];
		return tokenName;
	}

	/** 
	* 验证当前请求参数中的token是否合法，如果合法的token出现就会删除它，它不会再次成功合法的token 
	* @return 验证结果 
	*/  
	protected boolean validToken(HttpServletRequest request)  
	{  
		String tokenName = getTokenName(request);
		if (tokenName == null) {
			logger.debug("no token name found -> Invalid token ");
			return false;
		}
		String token = getToken(request, tokenName);
		if (token == null) {
			return false;
		}
		net.sf.ehcache.Element cacheElement = cache.get(tokenName);
		String cacheToken=null;
		if (cacheElement != null) {
			cacheToken = (String) cacheElement.getObjectValue();
		}
		if (!token.equals(cacheToken)) {
			logger.warn("xxx.internal.invalid.token Form token " + token
					+ " does not match the session token " + cacheToken + ".");
			return false;
		}
		cache.remove(tokenName);
		return true;
	}  

}
