/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.metrics;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonNode;
import java.io.IOException;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import org.apache.iceberg.metrics.TimerResult;
import org.apache.iceberg.relocated.com.google.common.annotations.VisibleForTesting;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.util.JsonUtil;

class TimerResultParser {
    private static final String MISSING_FIELD_ERROR_MSG = "Cannot parse timer from '%s': Missing field '%s'";
    private static final String TIME_UNIT = "time-unit";
    private static final String COUNT = "count";
    private static final String TOTAL_DURATION = "total-duration";

    private TimerResultParser() {
    }

    static String toJson(TimerResult timer) {
        return TimerResultParser.toJson(timer, false);
    }

    static String toJson(TimerResult timer, boolean pretty) {
        return JsonUtil.generate(gen -> TimerResultParser.toJson(timer, gen), pretty);
    }

    static void toJson(TimerResult timer, JsonGenerator gen) throws IOException {
        Preconditions.checkArgument(null != timer, "Invalid timer: null");
        gen.writeStartObject();
        gen.writeNumberField(COUNT, timer.count());
        gen.writeStringField(TIME_UNIT, timer.timeUnit().name().toLowerCase(Locale.ENGLISH));
        gen.writeNumberField(TOTAL_DURATION, TimerResultParser.fromDuration(timer.totalDuration(), timer.timeUnit()));
        gen.writeEndObject();
    }

    static TimerResult fromJson(String json) {
        return JsonUtil.parse(json, TimerResultParser::fromJson);
    }

    static TimerResult fromJson(JsonNode json) {
        Preconditions.checkArgument(null != json, "Cannot parse timer from null object");
        Preconditions.checkArgument(json.isObject(), "Cannot parse timer from non-object: %s", (Object)json);
        long count = JsonUtil.getLong(COUNT, json);
        TimeUnit unit = TimerResultParser.toTimeUnit(JsonUtil.getString(TIME_UNIT, json));
        long duration = JsonUtil.getLong(TOTAL_DURATION, json);
        return TimerResult.of(unit, TimerResultParser.toDuration(duration, unit), count);
    }

    static TimerResult fromJson(String timerName, JsonNode json) {
        Preconditions.checkArgument(null != json, "Cannot parse timer from null object");
        Preconditions.checkArgument(json.isObject(), "Cannot parse timer from non-object: %s", (Object)json);
        if (!json.has(timerName)) {
            return null;
        }
        JsonNode timer = json.get(timerName);
        Preconditions.checkArgument(timer.has(COUNT), MISSING_FIELD_ERROR_MSG, (Object)timerName, (Object)COUNT);
        Preconditions.checkArgument(timer.has(TIME_UNIT), MISSING_FIELD_ERROR_MSG, (Object)timerName, (Object)TIME_UNIT);
        Preconditions.checkArgument(timer.has(TOTAL_DURATION), MISSING_FIELD_ERROR_MSG, (Object)timerName, (Object)TOTAL_DURATION);
        long count = JsonUtil.getLong(COUNT, timer);
        TimeUnit unit = TimerResultParser.toTimeUnit(JsonUtil.getString(TIME_UNIT, timer));
        long duration = JsonUtil.getLong(TOTAL_DURATION, timer);
        return TimerResult.of(unit, TimerResultParser.toDuration(duration, unit), count);
    }

    @VisibleForTesting
    static long fromDuration(Duration duration, TimeUnit unit) {
        return unit.convert(duration.toNanos(), TimeUnit.NANOSECONDS);
    }

    @VisibleForTesting
    static Duration toDuration(long val, TimeUnit unit) {
        return Duration.of(val, TimerResultParser.toChronoUnit(unit));
    }

    private static TimeUnit toTimeUnit(String timeUnit) {
        try {
            return TimeUnit.valueOf(timeUnit.toUpperCase(Locale.ENGLISH));
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(String.format("Invalid time unit: %s", timeUnit), e);
        }
    }

    private static ChronoUnit toChronoUnit(TimeUnit unit) {
        switch (unit) {
            case NANOSECONDS: {
                return ChronoUnit.NANOS;
            }
            case MICROSECONDS: {
                return ChronoUnit.MICROS;
            }
            case MILLISECONDS: {
                return ChronoUnit.MILLIS;
            }
            case SECONDS: {
                return ChronoUnit.SECONDS;
            }
            case MINUTES: {
                return ChronoUnit.MINUTES;
            }
            case HOURS: {
                return ChronoUnit.HOURS;
            }
            case DAYS: {
                return ChronoUnit.DAYS;
            }
        }
        throw new IllegalArgumentException("Cannot determine chrono unit from time unit: " + (Object)((Object)unit));
    }
}

