/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.orc;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.iceberg.Schema;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.orc.ApplyNameMapping;
import org.apache.iceberg.orc.HasIds;
import org.apache.iceberg.orc.IdToOrcName;
import org.apache.iceberg.orc.OrcSchemaVisitor;
import org.apache.iceberg.orc.OrcToIcebergVisitor;
import org.apache.iceberg.orc.RemoveIds;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMultimap;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.shaded.org.apache.orc.TypeDescription;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.TypeUtil;
import org.apache.iceberg.types.Types;

public final class ORCSchemaUtil {
    static final String ICEBERG_ID_ATTRIBUTE = "iceberg.id";
    static final String ICEBERG_REQUIRED_ATTRIBUTE = "iceberg.required";
    public static final String ICEBERG_BINARY_TYPE_ATTRIBUTE = "iceberg.binary-type";
    public static final String ICEBERG_LONG_TYPE_ATTRIBUTE = "iceberg.long-type";
    static final String ICEBERG_FIELD_LENGTH = "iceberg.length";
    private static final ImmutableMultimap<Type.TypeID, TypeDescription.Category> TYPE_MAPPING = ImmutableMultimap.builder().put(Type.TypeID.BOOLEAN, TypeDescription.Category.BOOLEAN).put(Type.TypeID.INTEGER, TypeDescription.Category.BYTE).put(Type.TypeID.INTEGER, TypeDescription.Category.SHORT).put(Type.TypeID.INTEGER, TypeDescription.Category.INT).put(Type.TypeID.LONG, TypeDescription.Category.LONG).put(Type.TypeID.TIME, TypeDescription.Category.LONG).put(Type.TypeID.FLOAT, TypeDescription.Category.FLOAT).put(Type.TypeID.DOUBLE, TypeDescription.Category.DOUBLE).put(Type.TypeID.DATE, TypeDescription.Category.DATE).put(Type.TypeID.STRING, TypeDescription.Category.CHAR).put(Type.TypeID.STRING, TypeDescription.Category.VARCHAR).put(Type.TypeID.STRING, TypeDescription.Category.STRING).put(Type.TypeID.UUID, TypeDescription.Category.BINARY).put(Type.TypeID.FIXED, TypeDescription.Category.BINARY).put(Type.TypeID.BINARY, TypeDescription.Category.BINARY).put(Type.TypeID.DECIMAL, TypeDescription.Category.DECIMAL).build();

    private ORCSchemaUtil() {
    }

    public static TypeDescription convert(Schema schema) {
        TypeDescription root = TypeDescription.createStruct();
        Types.StructType schemaRoot = schema.asStruct();
        for (Types.NestedField field : schemaRoot.asStructType().fields()) {
            TypeDescription orcColumType = ORCSchemaUtil.convert(field.fieldId(), field.type(), field.isRequired());
            root.addField(field.name(), orcColumType);
        }
        return root;
    }

    private static TypeDescription convert(Integer fieldId, Type type, boolean isRequired) {
        TypeDescription orcType;
        switch (type.typeId()) {
            case BOOLEAN: {
                orcType = TypeDescription.createBoolean();
                break;
            }
            case INTEGER: {
                orcType = TypeDescription.createInt();
                break;
            }
            case TIME: {
                orcType = TypeDescription.createLong();
                orcType.setAttribute(ICEBERG_LONG_TYPE_ATTRIBUTE, LongType.TIME.toString());
                break;
            }
            case LONG: {
                orcType = TypeDescription.createLong();
                orcType.setAttribute(ICEBERG_LONG_TYPE_ATTRIBUTE, LongType.LONG.toString());
                break;
            }
            case FLOAT: {
                orcType = TypeDescription.createFloat();
                break;
            }
            case DOUBLE: {
                orcType = TypeDescription.createDouble();
                break;
            }
            case DATE: {
                orcType = TypeDescription.createDate();
                break;
            }
            case TIMESTAMP: {
                Types.TimestampType tsType = (Types.TimestampType)type;
                if (tsType.shouldAdjustToUTC()) {
                    orcType = TypeDescription.createTimestampInstant();
                    break;
                }
                orcType = TypeDescription.createTimestamp();
                break;
            }
            case STRING: {
                orcType = TypeDescription.createString();
                break;
            }
            case UUID: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.UUID.toString());
                break;
            }
            case FIXED: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.FIXED.toString());
                orcType.setAttribute(ICEBERG_FIELD_LENGTH, Integer.toString(((Types.FixedType)type).length()));
                break;
            }
            case BINARY: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.BINARY.toString());
                break;
            }
            case DECIMAL: {
                Types.DecimalType decimal = (Types.DecimalType)type;
                orcType = TypeDescription.createDecimal().withScale(decimal.scale()).withPrecision(decimal.precision());
                break;
            }
            case STRUCT: {
                orcType = TypeDescription.createStruct();
                for (Types.NestedField field : type.asStructType().fields()) {
                    TypeDescription childType = ORCSchemaUtil.convert(field.fieldId(), field.type(), field.isRequired());
                    orcType.addField(field.name(), childType);
                }
                break;
            }
            case LIST: {
                Types.ListType list = (Types.ListType)type;
                TypeDescription elementType = ORCSchemaUtil.convert(list.elementId(), list.elementType(), list.isElementRequired());
                orcType = TypeDescription.createList(elementType);
                break;
            }
            case MAP: {
                Types.MapType map = (Types.MapType)type;
                TypeDescription keyType = ORCSchemaUtil.convert(map.keyId(), map.keyType(), true);
                TypeDescription valueType = ORCSchemaUtil.convert(map.valueId(), map.valueType(), map.isValueRequired());
                orcType = TypeDescription.createMap(keyType, valueType);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unhandled type " + (Object)((Object)type.typeId()));
            }
        }
        orcType.setAttribute(ICEBERG_ID_ATTRIBUTE, String.valueOf(fieldId));
        orcType.setAttribute(ICEBERG_REQUIRED_ATTRIBUTE, String.valueOf(isRequired));
        return orcType;
    }

    public static Schema convert(TypeDescription orcSchema) {
        List<TypeDescription> children = orcSchema.getChildren();
        List<String> childrenNames = orcSchema.getFieldNames();
        Preconditions.checkState(children.size() == childrenNames.size(), "Error in ORC file, children fields and names do not match.");
        OrcToIcebergVisitor schemaConverter = new OrcToIcebergVisitor();
        List<Types.NestedField> fields = OrcToIcebergVisitor.visitSchema(orcSchema, schemaConverter).stream().filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        if (fields.size() == 0) {
            throw new IllegalArgumentException("ORC schema does not contain Iceberg IDs");
        }
        return new Schema(fields);
    }

    public static TypeDescription buildOrcProjection(Schema schema, TypeDescription originalOrcSchema) {
        Map<Integer, OrcField> icebergToOrc = ORCSchemaUtil.icebergToOrcMapping("root", originalOrcSchema);
        return ORCSchemaUtil.buildOrcProjection(Integer.MIN_VALUE, schema.asStruct(), true, icebergToOrc);
    }

    private static TypeDescription buildOrcProjection(Integer fieldId, Type type, boolean isRequired, Map<Integer, OrcField> mapping) {
        TypeDescription orcType;
        switch (type.typeId()) {
            case STRUCT: {
                orcType = TypeDescription.createStruct();
                for (Types.NestedField nestedField : type.asStructType().fields()) {
                    String name = Optional.ofNullable(mapping.get(nestedField.fieldId())).map(OrcField::name).orElseGet(() -> nestedField.name() + "_r" + nestedField.fieldId());
                    TypeDescription childType = ORCSchemaUtil.buildOrcProjection(nestedField.fieldId(), nestedField.type(), isRequired && nestedField.isRequired(), mapping);
                    orcType.addField(name, childType);
                }
                break;
            }
            case LIST: {
                Types.ListType list = (Types.ListType)type;
                TypeDescription elementType = ORCSchemaUtil.buildOrcProjection(list.elementId(), list.elementType(), isRequired && list.isElementRequired(), mapping);
                orcType = TypeDescription.createList(elementType);
                break;
            }
            case MAP: {
                Types.MapType map = (Types.MapType)type;
                TypeDescription keyType = ORCSchemaUtil.buildOrcProjection(map.keyId(), map.keyType(), isRequired, mapping);
                TypeDescription valueType = ORCSchemaUtil.buildOrcProjection(map.valueId(), map.valueType(), isRequired && map.isValueRequired(), mapping);
                orcType = TypeDescription.createMap(keyType, valueType);
                break;
            }
            default: {
                if (mapping.containsKey(fieldId)) {
                    TypeDescription originalType = mapping.get(fieldId).type();
                    Optional<TypeDescription> promotedType = ORCSchemaUtil.getPromotedType(type, originalType);
                    if (promotedType.isPresent()) {
                        orcType = promotedType.get();
                        break;
                    }
                    Preconditions.checkArgument(ORCSchemaUtil.isSameType(originalType, type), "Can not promote %s type to %s", (Object)originalType.getCategory(), (Object)type.typeId().name());
                    orcType = originalType.clone();
                    break;
                }
                if (isRequired) {
                    throw new IllegalArgumentException(String.format("Field %d of type %s is required and was not found.", fieldId, type.toString()));
                }
                orcType = ORCSchemaUtil.convert(fieldId, type, false);
            }
        }
        orcType.setAttribute(ICEBERG_ID_ATTRIBUTE, fieldId.toString());
        return orcType;
    }

    private static Map<Integer, OrcField> icebergToOrcMapping(String name, TypeDescription orcType) {
        HashMap<Integer, OrcField> icebergToOrc = Maps.newHashMap();
        switch (orcType.getCategory()) {
            case STRUCT: {
                List<String> childrenNames = orcType.getFieldNames();
                List<TypeDescription> children = orcType.getChildren();
                for (int i = 0; i < children.size(); ++i) {
                    icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping(childrenNames.get(i), children.get(i)));
                }
                break;
            }
            case LIST: {
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping("element", orcType.getChildren().get(0)));
                break;
            }
            case MAP: {
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping("key", orcType.getChildren().get(0)));
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping("value", orcType.getChildren().get(1)));
            }
        }
        if (orcType.getId() > 0) {
            ORCSchemaUtil.icebergID(orcType).ifPresent(integer -> icebergToOrc.put((Integer)integer, new OrcField(name, orcType)));
        }
        return icebergToOrc;
    }

    private static Optional<TypeDescription> getPromotedType(Type icebergType, TypeDescription originalOrcType) {
        Types.DecimalType newDecimal;
        TypeDescription promotedOrcType = null;
        if (Type.TypeID.LONG.equals((Object)icebergType.typeId()) && TypeDescription.Category.INT.equals((Object)originalOrcType.getCategory())) {
            promotedOrcType = TypeDescription.createLong();
        } else if (Type.TypeID.DOUBLE.equals((Object)icebergType.typeId()) && TypeDescription.Category.FLOAT.equals((Object)originalOrcType.getCategory())) {
            promotedOrcType = TypeDescription.createDouble();
        } else if (Type.TypeID.DECIMAL.equals((Object)icebergType.typeId()) && TypeDescription.Category.DECIMAL.equals((Object)originalOrcType.getCategory()) && (newDecimal = (Types.DecimalType)icebergType).scale() == originalOrcType.getScale() && newDecimal.precision() > originalOrcType.getPrecision()) {
            promotedOrcType = TypeDescription.createDecimal().withScale(newDecimal.scale()).withPrecision(newDecimal.precision());
        }
        return Optional.ofNullable(promotedOrcType);
    }

    private static boolean isSameType(TypeDescription orcType, Type icebergType) {
        if (icebergType.typeId() == Type.TypeID.TIMESTAMP) {
            Types.TimestampType tsType = (Types.TimestampType)icebergType;
            return Objects.equals((Object)(tsType.shouldAdjustToUTC() ? TypeDescription.Category.TIMESTAMP_INSTANT : TypeDescription.Category.TIMESTAMP), (Object)orcType.getCategory());
        }
        return TYPE_MAPPING.containsEntry((Object)icebergType.typeId(), (Object)orcType.getCategory());
    }

    static Optional<Integer> icebergID(TypeDescription orcType) {
        return Optional.ofNullable(orcType.getAttributeValue(ICEBERG_ID_ATTRIBUTE)).map(Integer::parseInt);
    }

    public static int fieldId(TypeDescription orcType) {
        String idStr = orcType.getAttributeValue(ICEBERG_ID_ATTRIBUTE);
        Preconditions.checkNotNull(idStr, "Missing expected '%s' property", (Object)ICEBERG_ID_ATTRIBUTE);
        return Integer.parseInt(idStr);
    }

    static boolean isOptional(TypeDescription orcType) {
        String isRequiredStr = orcType.getAttributeValue(ICEBERG_REQUIRED_ATTRIBUTE);
        if (isRequiredStr != null) {
            return !Boolean.parseBoolean(isRequiredStr);
        }
        return true;
    }

    static TypeDescription removeIds(TypeDescription type) {
        return OrcSchemaVisitor.visit(type, new RemoveIds());
    }

    static boolean hasIds(TypeDescription orcSchema) {
        return OrcSchemaVisitor.visit(orcSchema, new HasIds());
    }

    static TypeDescription applyNameMapping(TypeDescription orcSchema, NameMapping nameMapping) {
        return OrcSchemaVisitor.visit(orcSchema, new ApplyNameMapping(nameMapping));
    }

    public static Map<Integer, String> idToOrcName(Schema schema) {
        return TypeUtil.visit(schema, new IdToOrcName());
    }

    private static class OrcField {
        private final String name;
        private final TypeDescription type;

        OrcField(String name, TypeDescription type) {
            this.name = name;
            this.type = type;
        }

        public String name() {
            return this.name;
        }

        public TypeDescription type() {
            return this.type;
        }
    }

    public static enum LongType {
        TIME,
        LONG;

    }

    public static enum BinaryType {
        UUID,
        FIXED,
        BINARY;

    }
}

