/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.jdbc;

import java.io.Closeable;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.SQLNonTransientConnectionException;
import java.sql.SQLTimeoutException;
import java.sql.SQLTransientConnectionException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.iceberg.BaseMetastoreCatalog;
import org.apache.iceberg.CatalogUtil;
import org.apache.iceberg.TableMetadata;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.SupportsNamespaces;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.exceptions.NamespaceNotEmptyException;
import org.apache.iceberg.exceptions.NoSuchNamespaceException;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.hadoop.Configurable;
import org.apache.iceberg.io.FileIO;
import org.apache.iceberg.jdbc.JdbcClientPool;
import org.apache.iceberg.jdbc.JdbcTableOperations;
import org.apache.iceberg.jdbc.JdbcUtil;
import org.apache.iceberg.jdbc.UncheckedInterruptedException;
import org.apache.iceberg.jdbc.UncheckedSQLException;
import org.apache.iceberg.relocated.com.google.common.base.Joiner;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcCatalog
extends BaseMetastoreCatalog
implements Configurable<Configuration>,
SupportsNamespaces,
Closeable {
    public static final String PROPERTY_PREFIX = "jdbc.";
    private static final Logger LOG = LoggerFactory.getLogger(JdbcCatalog.class);
    private static final Joiner SLASH = Joiner.on("/");
    private FileIO io;
    private String catalogName = "jdbc";
    private String warehouseLocation;
    private Object conf;
    private JdbcClientPool connections;

    @Override
    public void initialize(String name, Map<String, String> properties) {
        String uri = properties.get("uri");
        Preconditions.checkNotNull(uri, "JDBC connection URI is required");
        String warehouse = properties.get("warehouse");
        Preconditions.checkNotNull(warehouse, "JDBC warehouse location is required");
        this.warehouseLocation = warehouse.replaceAll("/*$", "");
        if (name != null) {
            this.catalogName = name;
        }
        String fileIOImpl = properties.getOrDefault("io-impl", "org.apache.iceberg.hadoop.HadoopFileIO");
        this.io = CatalogUtil.loadFileIO(fileIOImpl, properties, this.conf);
        try {
            LOG.debug("Connecting to JDBC database {}", (Object)properties.get("uri"));
            this.connections = new JdbcClientPool(uri, properties);
            this.initializeCatalogTables();
        }
        catch (SQLTimeoutException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Query timed out", new Object[0]);
        }
        catch (SQLNonTransientConnectionException | SQLTransientConnectionException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog: Connection failed", new Object[0]);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Cannot initialize JDBC catalog", new Object[0]);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to initialize", new Object[0]);
        }
    }

    private void initializeCatalogTables() throws InterruptedException, SQLException {
        LOG.trace("Creating database tables (if missing) to store iceberg catalog");
        this.connections.run(conn -> {
            DatabaseMetaData dbMeta = conn.getMetaData();
            ResultSet tableExists = dbMeta.getTables(null, null, "iceberg_tables", null);
            if (tableExists.next()) {
                return true;
            }
            LOG.debug("Creating table {} to store iceberg catalog", (Object)"iceberg_tables");
            return conn.prepareStatement("CREATE TABLE iceberg_tables(catalog_name VARCHAR(255) NOT NULL,table_namespace VARCHAR(255) NOT NULL,table_name VARCHAR(255) NOT NULL,metadata_location VARCHAR(5500),previous_metadata_location VARCHAR(5500),PRIMARY KEY (catalog_name, table_namespace, table_name))").execute();
        });
    }

    @Override
    protected TableOperations newTableOps(TableIdentifier tableIdentifier) {
        return new JdbcTableOperations(this.connections, this.io, this.catalogName, tableIdentifier);
    }

    @Override
    protected String defaultWarehouseLocation(TableIdentifier table) {
        return SLASH.join(this.defaultNamespaceLocation(table.namespace()), table.name(), new Object[0]);
    }

    @Override
    public boolean dropTable(TableIdentifier identifier, boolean purge) {
        int deletedRecords;
        TableOperations ops = this.newTableOps(identifier);
        TableMetadata lastMetadata = purge && ops.current() != null ? ops.current() : null;
        try {
            deletedRecords = this.connections.run(conn -> {
                PreparedStatement sql = conn.prepareStatement("DELETE FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? ");
                Throwable throwable = null;
                try {
                    sql.setString(1, this.catalogName);
                    sql.setString(2, JdbcUtil.namespaceToString(identifier.namespace()));
                    sql.setString(3, identifier.name());
                    Integer n = sql.executeUpdate();
                    return n;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (sql != null) {
                        JdbcCatalog.$closeResource(throwable, sql);
                    }
                }
            });
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to drop %s", identifier);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to dropTable", new Object[0]);
        }
        if (deletedRecords == 0) {
            LOG.info("Skipping drop, table does not exist: {}", (Object)identifier);
            return false;
        }
        if (purge && lastMetadata != null) {
            CatalogUtil.dropTableData(ops.io(), lastMetadata);
        }
        LOG.info("Dropped table: {}", (Object)identifier);
        return true;
    }

    @Override
    public List<TableIdentifier> listTables(Namespace namespace) {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        try {
            return this.connections.run(conn -> {
                ArrayList<TableIdentifier> results = Lists.newArrayList();
                PreparedStatement sql = conn.prepareStatement("SELECT * FROM iceberg_tables WHERE catalog_name = ? AND table_namespace = ?");
                Throwable throwable = null;
                try {
                    sql.setString(1, this.catalogName);
                    sql.setString(2, JdbcUtil.namespaceToString(namespace));
                    ResultSet rs = sql.executeQuery();
                    while (rs.next()) {
                        results.add(JdbcUtil.stringToTableIdentifier(rs.getString("table_namespace"), rs.getString("table_name")));
                    }
                    ArrayList<TableIdentifier> arrayList = results;
                    return arrayList;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (sql != null) {
                        JdbcCatalog.$closeResource(throwable, sql);
                    }
                }
            });
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to list tables in namespace: %s", namespace);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted during JDBC operation", new Object[0]);
        }
    }

    @Override
    public void renameTable(TableIdentifier from, TableIdentifier to) {
        try {
            int updatedRecords = this.connections.run(conn -> {
                PreparedStatement sql = conn.prepareStatement("UPDATE iceberg_tables SET table_namespace = ? , table_name = ?  WHERE catalog_name = ? AND table_namespace = ? AND table_name = ? ");
                Throwable throwable = null;
                try {
                    sql.setString(1, JdbcUtil.namespaceToString(to.namespace()));
                    sql.setString(2, to.name());
                    sql.setString(3, this.catalogName);
                    sql.setString(4, JdbcUtil.namespaceToString(from.namespace()));
                    sql.setString(5, from.name());
                    Integer n = sql.executeUpdate();
                    return n;
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (sql != null) {
                        JdbcCatalog.$closeResource(throwable, sql);
                    }
                }
            });
            if (updatedRecords == 1) {
                LOG.info("Renamed table from {}, to {}", (Object)from, (Object)to);
            } else {
                if (updatedRecords == 0) {
                    throw new NoSuchTableException("Table does not exist: %s", from);
                }
                LOG.warn("Rename operation affected {} rows: the catalog table's primary key assumption has been violated", (Object)updatedRecords);
            }
        }
        catch (SQLIntegrityConstraintViolationException e) {
            throw new AlreadyExistsException("Table already exists: %s", to);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to rename %s to %s", from, to);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to rename", new Object[0]);
        }
    }

    @Override
    public String name() {
        return this.catalogName;
    }

    @Override
    public void setConf(Configuration conf) {
        this.conf = conf;
    }

    @Override
    public void createNamespace(Namespace namespace, Map<String, String> metadata) {
        throw new UnsupportedOperationException("Cannot create namespace " + namespace + ": createNamespace is not supported");
    }

    @Override
    public List<Namespace> listNamespaces(Namespace namespace) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        try {
            List<Namespace> namespaces = this.connections.run(conn -> {
                ArrayList<Namespace> result = Lists.newArrayList();
                PreparedStatement sql = conn.prepareStatement("SELECT DISTINCT table_namespace FROM iceberg_tables WHERE catalog_name = ? AND table_namespace LIKE ?");
                Throwable throwable = null;
                try {
                    sql.setString(1, this.catalogName);
                    sql.setString(2, JdbcUtil.namespaceToString(namespace) + "%");
                    ResultSet rs = sql.executeQuery();
                    while (rs.next()) {
                        result.add(JdbcUtil.stringToNamespace(rs.getString("table_namespace")));
                    }
                    rs.close();
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (sql != null) {
                        JdbcCatalog.$closeResource(throwable, sql);
                    }
                }
                return result;
            });
            int subNamespaceLevelLength = namespace.levels().length + 1;
            namespaces = namespaces.stream().filter(n -> !n.equals(namespace)).filter(n -> n.levels().length >= subNamespaceLevelLength).map(n -> Namespace.of((String[])Arrays.stream(n.levels()).limit(subNamespaceLevelLength).toArray(String[]::new))).distinct().collect(Collectors.toList());
            return namespaces;
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to listNamespaces(namespace) Namespace: %s", namespace);
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to list all namespace: %s in catalog", namespace);
        }
    }

    @Override
    public Map<String, String> loadNamespaceMetadata(Namespace namespace) throws NoSuchNamespaceException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        return ImmutableMap.of("location", this.defaultNamespaceLocation(namespace));
    }

    private String defaultNamespaceLocation(Namespace namespace) {
        if (namespace.isEmpty()) {
            return this.warehouseLocation;
        }
        return SLASH.join(this.warehouseLocation, SLASH.join(namespace.levels()), new Object[0]);
    }

    @Override
    public boolean dropNamespace(Namespace namespace) throws NamespaceNotEmptyException {
        if (!this.namespaceExists(namespace)) {
            throw new NoSuchNamespaceException("Namespace does not exist: %s", namespace);
        }
        List<TableIdentifier> tableIdentifiers = this.listTables(namespace);
        if (tableIdentifiers != null && !tableIdentifiers.isEmpty()) {
            throw new NamespaceNotEmptyException("Namespace %s is not empty. %s tables exist.", namespace, tableIdentifiers.size());
        }
        return false;
    }

    @Override
    public boolean setProperties(Namespace namespace, Map<String, String> properties) throws NoSuchNamespaceException {
        throw new UnsupportedOperationException("Cannot set properties " + namespace + " : setProperties is not supported");
    }

    @Override
    public boolean removeProperties(Namespace namespace, Set<String> properties) throws NoSuchNamespaceException {
        throw new UnsupportedOperationException("Cannot remove properties " + namespace + " : removeProperties is not supported");
    }

    @Override
    public void close() {
        this.connections.close();
    }

    @Override
    public boolean namespaceExists(Namespace namespace) {
        try {
            return this.connections.run(conn -> {
                boolean exists = false;
                PreparedStatement sql = conn.prepareStatement("SELECT table_namespace FROM iceberg_tables WHERE catalog_name = ? AND table_namespace LIKE ? LIMIT 1");
                Throwable throwable = null;
                try {
                    sql.setString(1, this.catalogName);
                    sql.setString(2, JdbcUtil.namespaceToString(namespace) + "%");
                    ResultSet rs = sql.executeQuery();
                    if (rs.next()) {
                        exists = true;
                    }
                    rs.close();
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (sql != null) {
                        JdbcCatalog.$closeResource(throwable, sql);
                    }
                }
                return exists;
            });
        }
        catch (SQLException e) {
            throw new UncheckedSQLException(e, "Failed to get namespace %s", namespace);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new UncheckedInterruptedException(e, "Interrupted in call to namespaceExists(namespace)", new Object[0]);
        }
    }

    private static /* synthetic */ /* end resource */ void $closeResource(Throwable x0, AutoCloseable x1) {
        if (x0 != null) {
            try {
                x1.close();
            }
            catch (Throwable throwable) {
                x0.addSuppressed(throwable);
            }
        } else {
            x1.close();
        }
    }
}

