/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hudi.common.util;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.hudi.common.util.StringUtils;
import org.apache.hudi.common.util.ValidationUtils;

public class DateTimeUtils {
    private static final Map<String, ChronoUnit> LABEL_TO_UNIT_MAP = Collections.unmodifiableMap(DateTimeUtils.initMap());

    public static Instant microsToInstant(long microsFromEpoch) {
        long epochSeconds = microsFromEpoch / 1000000L;
        long nanoAdjustment = microsFromEpoch % 1000000L * 1000L;
        return Instant.ofEpochSecond(epochSeconds, nanoAdjustment);
    }

    public static long instantToMicros(Instant instant) {
        long seconds = instant.getEpochSecond();
        int nanos = instant.getNano();
        if (seconds < 0L && nanos > 0) {
            long micros = Math.multiplyExact(seconds + 1L, 1000000L);
            long adjustment = (long)nanos / 1000L - 1000000L;
            return Math.addExact(micros, adjustment);
        }
        long micros = Math.multiplyExact(seconds, 1000000L);
        return Math.addExact(micros, (long)nanos / 1000L);
    }

    public static Instant parseDateTime(String s) throws DateTimeParseException {
        ValidationUtils.checkArgument((boolean)Objects.nonNull(s), (String)"Input String cannot be null.");
        try {
            return Instant.ofEpochMilli(Long.parseLong(s));
        }
        catch (NumberFormatException e) {
            return Instant.parse(s);
        }
    }

    public static Duration parseDuration(String text) {
        long value;
        char current;
        int pos;
        ValidationUtils.checkArgument((!StringUtils.isNullOrEmpty((String)text) ? 1 : 0) != 0);
        String trimmed = text.trim();
        ValidationUtils.checkArgument((!trimmed.isEmpty() ? 1 : 0) != 0, (String)"argument is an empty- or whitespace-only string");
        int len = trimmed.length();
        for (pos = 0; pos < len && (current = trimmed.charAt(pos)) >= '0' && current <= '9'; ++pos) {
        }
        String number = trimmed.substring(0, pos);
        String unitLabel = trimmed.substring(pos).trim().toLowerCase(Locale.US);
        if (number.isEmpty()) {
            throw new NumberFormatException("text does not start with a number");
        }
        try {
            value = Long.parseLong(number);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("The value '" + number + "' cannot be re represented as 64bit number (numeric overflow).");
        }
        if (unitLabel.isEmpty()) {
            return Duration.of(value, ChronoUnit.MILLIS);
        }
        ChronoUnit unit = LABEL_TO_UNIT_MAP.get(unitLabel);
        if (unit != null) {
            return Duration.of(value, unit);
        }
        throw new IllegalArgumentException("Time interval unit label '" + unitLabel + "' does not match any of the recognized units: " + TimeUnit.getAllUnits());
    }

    private static Map<String, ChronoUnit> initMap() {
        HashMap<String, ChronoUnit> labelToUnit = new HashMap<String, ChronoUnit>();
        for (TimeUnit timeUnit : TimeUnit.values()) {
            for (String label : timeUnit.getLabels()) {
                labelToUnit.put(label, timeUnit.getUnit());
            }
        }
        return labelToUnit;
    }

    public static String formatUnixTimestamp(long unixTimestamp, String timeFormat) {
        ValidationUtils.checkArgument((!StringUtils.isNullOrEmpty((String)timeFormat) ? 1 : 0) != 0);
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern(timeFormat);
        return LocalDateTime.ofInstant(Instant.ofEpochSecond(unixTimestamp), ZoneId.systemDefault()).format(dtf);
    }

    private static enum TimeUnit {
        DAYS(ChronoUnit.DAYS, TimeUnit.singular("d"), TimeUnit.plural("day")),
        HOURS(ChronoUnit.HOURS, TimeUnit.singular("h"), TimeUnit.plural("hour")),
        MINUTES(ChronoUnit.MINUTES, TimeUnit.singular("min"), TimeUnit.plural("minute")),
        SECONDS(ChronoUnit.SECONDS, TimeUnit.singular("s"), TimeUnit.plural("sec"), TimeUnit.plural("second")),
        MILLISECONDS(ChronoUnit.MILLIS, TimeUnit.singular("ms"), TimeUnit.plural("milli"), TimeUnit.plural("millisecond")),
        MICROSECONDS(ChronoUnit.MICROS, TimeUnit.singular("\u00b5s"), TimeUnit.plural("micro"), TimeUnit.plural("microsecond")),
        NANOSECONDS(ChronoUnit.NANOS, TimeUnit.singular("ns"), TimeUnit.plural("nano"), TimeUnit.plural("nanosecond"));

        private static final String PLURAL_SUFFIX = "s";
        private final List<String> labels;
        private final ChronoUnit unit;

        private TimeUnit(ChronoUnit unit, String[] ... labels) {
            this.unit = unit;
            this.labels = Arrays.stream(labels).flatMap(Arrays::stream).collect(Collectors.toList());
        }

        private static String[] singular(String label) {
            return new String[]{label};
        }

        private static String[] plural(String label) {
            return new String[]{label, label + PLURAL_SUFFIX};
        }

        public List<String> getLabels() {
            return this.labels;
        }

        public ChronoUnit getUnit() {
            return this.unit;
        }

        public static String getAllUnits() {
            return Arrays.stream(TimeUnit.values()).map(TimeUnit::createTimeUnitString).collect(Collectors.joining(", "));
        }

        private static String createTimeUnitString(TimeUnit timeUnit) {
            return timeUnit.name() + ": (" + String.join((CharSequence)" | ", timeUnit.getLabels()) + ")";
        }
    }
}

