/*
 * Copyright (C) 2005-2015 Schlichtherle IT Services.
 * All rights reserved. Use is subject to license terms.
 */
package com.nexes.wizard;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * The controller for the wizard component, which is responsible for reacting to
 * events generated by pushing any of the three buttons, 'Next', 'Previous', and
 * 'Cancel.' Based on what button is pressed, the controller will update the
 * model to show a new panel and reset the state of the buttons as necessary.
 * 
 * @author Robert Eckstein (original code)
 * @author Christian Schlichtherle (revision)
 */
public class WizardController implements ActionListener {

    private final Wizard wizard;

    /**
     * This constructor accepts a reference to the Wizard component that created
     * it, which it uses to update the button components and access the
     * WizardModel.
     *
     * @param wizard A callback to the Wizard component that created this controller.
     */
    public WizardController(final Wizard wizard) {
        this.wizard = wizard;
    }

    /**
     * Calling method for the action listener interface. This class listens for
     * actions performed by the buttons in the Wizard class, and calls methods
     * below to determine the correct course of action.
     *
     * @param evt The ActionEvent that occurred.
     */
    public void actionPerformed(ActionEvent evt) {
        final String command = evt.getActionCommand();
        if (command.equals(Wizard.BACK_BUTTON_ACTION_COMMAND))
            backButtonPressed();
        else if (command.equals(Wizard.NEXT_BUTTON_ACTION_COMMAND))
            nextButtonPressed();
        else if (command.equals(Wizard.CANCEL_BUTTON_ACTION_COMMAND))
            cancelButtonPressed();
    }

    private void backButtonPressed() {
        final WizardModel model = wizard.getModel();
        final WizardPanelDescriptor current = model.getCurrentPanelDescriptor();

        //  Get the descriptor that the current panel identifies as the previous
        //  panel, and display it.
        final Object back = current.getBackPanelDescriptor();
        wizard.setCurrentPanel(back);
    }

    private void nextButtonPressed() {
        final WizardModel model = wizard.getModel();
        final WizardPanelDescriptor current = model.getCurrentPanelDescriptor();

        //  If it is a finishable panel, close down the dialog. Otherwise,
        //  get the ID that the current panel identifies as the next panel,
        //  and display it.
        final Object next = current.getNextPanelDescriptor();
        if (next instanceof WizardPanelDescriptor.FinishIdentifier)
            wizard.close(Wizard.FINISH_RETURN_CODE);
        else
            wizard.setCurrentPanel(next);
    }

    private void cancelButtonPressed() {
        wizard.close(Wizard.CANCEL_RETURN_CODE);
    }

    void resetButtonsToPanelRules() {
        //  Reset the buttons to support the original panel rules,
        //  including whether the next or back buttons are enabled or
        //  disabled, or if the panel is finishable.
        final WizardModel model = wizard.getModel();
        final WizardPanelDescriptor current = model.getCurrentPanelDescriptor();

        //  If the panel in question has another panel behind it, enable
        //  the back button. Otherwise, disable it.
        model.setBackButtonText(Wizard.DEFAULT_BACK_BUTTON_TEXT);
        if (current.getBackPanelDescriptor() != null)
            model.setBackButtonEnabled(Boolean.TRUE);
        else
            model.setBackButtonEnabled(Boolean.FALSE);

        //  If the panel in question has one or more panels in front of it,
        //  enable the next button. Otherwise, disable it.
        //model.setNextButtonText(Wizard.DEFAULT_NEXT_BUTTON_TEXT);
        if (current.getNextPanelDescriptor() != null)
            model.setNextButtonEnabled(Boolean.TRUE);
        else
            model.setNextButtonEnabled(Boolean.FALSE);

        //  If the panel in question is the last panel in the series, change
        //  the Next button to Finish and enable it. Otherwise, set the text
        //  back to Next.
        if (current.getNextPanelDescriptor() instanceof WizardPanelDescriptor.FinishIdentifier) {
            model.setNextButtonText(Wizard.DEFAULT_FINISH_BUTTON_TEXT);
            model.setNextButtonEnabled(Boolean.TRUE);
        } else {
            model.setNextButtonText(Wizard.DEFAULT_NEXT_BUTTON_TEXT);
        }
    }
}
