package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.utils.BeanConvertUtils;
import com.gccloud.starter.config.GlobalConfig;
import com.gccloud.starter.core.dao.SysUserDao;
import com.gccloud.starter.core.dto.*;
import com.gccloud.starter.core.entity.SysTenantEntity;
import com.gccloud.starter.core.entity.SysUserEntity;
import com.gccloud.starter.core.event.SysUserEvent;
import com.gccloud.starter.core.service.*;
import com.gccloud.starter.core.vo.SysUserVO;
import com.gccloud.starter.mybatis.page.PageVO;
import com.gccloud.starter.mybatis.utils.QueryWrapperUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.crypto.hash.Sha256Hash;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * 用户接口业务类
 *
 * @Author tang.jiawen
 * @Date 2020/6/16
 * @Version 1.0.0
 */
@Service
@Slf4j
public class SysUserServiceImpl extends ServiceImpl<SysUserDao, SysUserEntity> implements ISysUserService {

    @Autowired
    @Lazy
    private ISysOrgService orgService;
    @Autowired
    private ISysUserRoleService userRoleService;
    @Resource
    private ISysTenantService tenantService;
    @Autowired
    private ISysDataPermissionService dataPermissionService;
    @Autowired
    private ApplicationContext context;
    @Autowired
    private ISysMenuService menuService;
    @Autowired
    private GlobalConfig globalConfig;

    @Override
    public void importExcel(List<SysUserDTO> userList) {
        for (SysUserDTO user : userList) {
            String pwd = new Sha256Hash(new Sha256Hash(user.getPassword()).toHex() + user.getPassword()).toHex();
            user.setPassword(pwd);
            SysUserEntity userEntity = BeanConvertUtils.convert(user, SysUserEntity.class);
            this.save(userEntity);
            publishEvent(user.getId(), GlobalConst.User.EventType.IMPORT);
        }
    }

    @Override
    public PageVO<SysUserEntity> getPage(SysUserSearchDTO searchDTO) {
        LambdaQueryWrapper<SysUserEntity> queryWrapper = QueryWrapperUtils.wrapperLike(new LambdaQueryWrapper<>(), searchDTO.getSearchKey(), SysUserEntity::getUsername, SysUserEntity::getRealName, SysUserEntity::getOrgId);
        if (StringUtils.isNotBlank(searchDTO.getOrgId())) {
            Set<String> orgIdSet = orgService.getAllChildrenIds(searchDTO.getOrgId());
            HashSet<String> copyOrgIdSet = Sets.newHashSet(orgIdSet);
            copyOrgIdSet.add(searchDTO.getOrgId());
            queryWrapper.in(SysUserEntity::getOrgId, copyOrgIdSet);
        }
        // 按照修改日期进行降序
        queryWrapper.orderByDesc(SysUserEntity::getUpdateDate);
        return page(searchDTO, queryWrapper);
    }

    @Override
    public PageVO<SysUserEntity> getTenantManagerPage(SysTenantManagerSearchDTO searchDTO) {
        if (StringUtils.isBlank(searchDTO.getTenantId())) {
            throw new GlobalException("租户信息不能为空");
        }
        LambdaQueryWrapper<SysUserEntity> queryWrapper = QueryWrapperUtils.wrapperLike(new LambdaQueryWrapper<>(), searchDTO.getSearchKey(), SysUserEntity::getUsername, SysUserEntity::getEmail, SysUserEntity::getPhone, SysUserEntity::getRealName);
        queryWrapper
                .eq(SysUserEntity::getTenantId, searchDTO.getTenantId())
                .eq(SysUserEntity::getTenantManager, true)
                .orderByAsc(SysUserEntity::getCreateDate);
        return page(searchDTO, queryWrapper);
    }

    @Override
    public List<String> getMenuIdList(String userId) {
        return baseMapper.getAllMenuId(userId);
    }

    @Override
    public Set<String> getPermission(String userId) {
        List<String> permsList = getBaseMapper().getAllPerms(userId);
        // 用户权限列表
        Set<String> permsSet = new HashSet<>();
        for (String perms : permsList) {
            if (StringUtils.isBlank(perms)) {
                continue;
            }
            permsSet.addAll(Arrays.asList(perms.trim().split(",")));
        }
        return permsSet;
    }

    @Override
    public SysUserEntity getByCount(String count) {
        if (StringUtils.isBlank(count)) {
            return null;
        }
        LambdaQueryWrapper<SysUserEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.
                eq(SysUserEntity::getUsername, count).or().
                eq(SysUserEntity::getEmail, count).or().
                eq(SysUserEntity::getPhone, count);
        return this.getOne(queryWrapper);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(SysUserDTO userDTO) {
        SysUserEntity entity = BeanConvertUtils.convert(userDTO, SysUserEntity.class);
        check(entity);
        // 密码过期时间
        entity.setPwdExpireDate(DateTime.now().plusDays(globalConfig.getPassword().getExpireDate()).toDate());
        this.save(entity);
        userDTO.setId(entity.getId());
        if (entity.getDataPermission() == GlobalConst.User.DataPermission.ORG_CUSTOM) {
            dataPermissionService.add(entity.getId(), userDTO.getDataPermissionIdList());
        }
        SysUserRoleDTO userRoleDTO = new SysUserRoleDTO(userDTO.getTenantId(), entity.getId());
        userRoleService.add(entity.getId(), userDTO.getRoleIdList(), userRoleDTO);
        // 事件通知
        publishEvent(entity.getId(), GlobalConst.User.EventType.ADD);
    }

    @Override
    public void updateStatus(SysUserStatusDTO userStatusDTO) {
        if (userStatusDTO == null) {
            return;
        }
        if (StringUtils.isBlank(userStatusDTO.getId())) {
            return;
        }
        if (userStatusDTO.getStatus() == null) {
            return;
        }
        LambdaUpdateWrapper<SysUserEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper.eq(SysUserEntity::getId, userStatusDTO.getId());
        updateWrapper.set(SysUserEntity::getStatus, userStatusDTO.getStatus());
        boolean result = update(updateWrapper);
        if (result) {
            // 通知
            publishEvent(userStatusDTO.getId(), GlobalConst.User.EventType.UPDATE);
        }
    }

    @Override
    public void update(SysUserDTO userDTO) {
        SysUserEntity entity = BeanConvertUtils.convert(userDTO, SysUserEntity.class);
        check(entity);
        if (StringUtils.isNotBlank(userDTO.getPassword())) {
            entity.setPwdExpireDate(DateTime.now().plusDays(globalConfig.getPassword().getExpireDate()).toDate());
        }
        this.updateById(entity);
        if (entity.getDataPermission() == GlobalConst.User.DataPermission.ORG_CUSTOM) {
            dataPermissionService.add(entity.getId(), userDTO.getDataPermissionIdList());
        }
        userRoleService.add(entity.getId(), userDTO.getRoleIdList(), null);
        // 事件通知
        publishEvent(entity.getId(), GlobalConst.User.EventType.UPDATE);
    }

    @Override
    public void updateInfo(SysUserEntity user) {
        check(user);
        // 仅更新指定的几个字段
        LambdaUpdateWrapper<SysUserEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper
                .eq(SysUserEntity::getId, user.getId())
                .set(StringUtils.isNotBlank(user.getPhone()), SysUserEntity::getPhone, user.getPhone())
                .set(StringUtils.isNotBlank(user.getEmail()), SysUserEntity::getEmail, user.getEmail())
                .set(StringUtils.isNotBlank(user.getPassword()), SysUserEntity::getPassword, user.getPassword())
                .set(StringUtils.isNotBlank(user.getPassword()), SysUserEntity::getPwdExpireDate, DateTime.now().plusDays(globalConfig.getPassword().getExpireDate()).toDate());
        this.update(updateWrapper);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteUserIds(String[] userIds) {
        if (ArrayUtils.isEmpty(userIds)) {
            return;
        }
        LambdaQueryWrapper<SysUserEntity> queryWrapper = Wrappers.lambdaQuery();
        queryWrapper.in(SysUserEntity::getId, userIds);
        this.delete(queryWrapper);
        userRoleService.deleteByUserIds(userIds);
        publishEvent(userIds, GlobalConst.User.EventType.DELETE);
    }

    @Override
    public List<SysUserVO> getTenantManager(String tenantId) {
        if (StringUtils.isBlank(tenantId)) {
            throw new GlobalException("租户id不能为空");
        }
        LambdaQueryWrapper<SysUserEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysUserEntity::getTenantId, tenantId)
                .eq(SysUserEntity::getTenantManager, true);
        List<SysUserEntity> list = this.list(queryWrapper);
        SysTenantEntity tenantEntity = tenantService.getById(tenantId);
        List<SysUserVO> userVos = Lists.newArrayList();
        for (SysUserEntity userEntity : list) {
            SysUserVO userVo = BeanConvertUtils.convert(userEntity, SysUserVO.class);
            userVo.setTenantName(tenantEntity.getName());
            userVos.add(userVo);
        }
        return userVos;
    }

    @Override
    public boolean existByOrgId(String orgId) {
        LambdaQueryWrapper<SysUserEntity> queryWrapper = new LambdaQueryWrapper<SysUserEntity>()
                .select(SysUserEntity::getId)
                .in(SysUserEntity::getOrgId, orgId);
        int count = count(queryWrapper);
        return count > 0 ? true : false;
    }

    /**
     * 事件通知
     *
     * @param userId
     * @param eventType
     */
    private void publishEvent(String userId, int eventType) {
        context.publishEvent(new SysUserEvent(this, userId, eventType));
    }

    private void publishEvent(String[] userIds, int eventType) {
        for (String userId : userIds) {
            publishEvent(userId, eventType);
        }
    }

    private void checkEmail(SysUserEntity user) {
        if (StringUtils.isBlank(user.getEmail())) {
            return;
        }
        // 检查邮箱是否重复
        boolean repeat = repeatGlobal(SysUserEntity::getId, user.getId(), SysUserEntity::getEmail, user.getEmail());
        if (repeat) {
            throw new GlobalException(String.format("邮箱: %s 已被占用，换一个吧", user.getEmail()));
        }
    }

    private void checkPhone(SysUserEntity user) {
        if (StringUtils.isBlank(user.getPhone())) {
            return;
        }
        // 检查手机是否重复
        boolean repeat = repeatGlobal(SysUserEntity::getId, user.getId(), SysUserEntity::getPhone, user.getPhone());
        if (repeat) {
            throw new GlobalException(String.format("手机号: %s 已被占用，换一个吧", user.getPhone()));
        }
    }

    private void checkUsername(SysUserEntity user) {
        if (StringUtils.isBlank(user.getUsername())) {
            return;
        }
        // 检查手机是否重复
        boolean repeat = repeatGlobal(SysUserEntity::getId, user.getId(), SysUserEntity::getUsername, user.getUsername());
        if (repeat) {
            throw new GlobalException(String.format("账号: %s 已被占用，换一个吧", user.getUsername()));
        }
    }

    private void check(SysUserEntity user) {
        checkUsername(user);
        checkPhone(user);
        checkEmail(user);
    }
}
