package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.dto.SearchDTO;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.utils.BeanConvertUtils;
import com.gccloud.starter.core.dao.SysRoleDao;
import com.gccloud.starter.core.dto.SysRoleDTO;
import com.gccloud.starter.core.entity.SysRoleEntity;
import com.gccloud.starter.core.entity.SysRoleMenuEntity;
import com.gccloud.starter.core.service.ISysRoleMenuService;
import com.gccloud.starter.core.service.ISysRoleService;
import com.gccloud.starter.core.service.ISysUserRoleService;
import com.gccloud.starter.mybatis.page.PageVO;
import com.gccloud.starter.mybatis.utils.QueryWrapperUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @Author tang.jiawen
 * @Date 2020/6/18
 * @Version 1.0.0
 */

@Service
@Slf4j
public class SysRoleServiceImpl extends ServiceImpl<SysRoleDao, SysRoleEntity> implements ISysRoleService {

    @Autowired
    private ISysRoleMenuService roleMenuService;
    @Autowired
    private ISysUserRoleService userRoleService;

    @Override
    public PageVO getPage(SearchDTO searchDTO) {
        LambdaQueryWrapper<SysRoleEntity> qw = QueryWrapperUtils.wrapperLike(new LambdaQueryWrapper(), searchDTO.getSearchKey(), SysRoleEntity::getName, SysRoleEntity::getRemark);
        return this.page(searchDTO, qw.orderByDesc(SysRoleEntity::getUpdateDate));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(SysRoleDTO role) {
        SysRoleEntity entity = BeanConvertUtils.convert(role, SysRoleEntity.class);
        this.checkRoleName(entity);
        this.checkRoleCode(entity);
        this.save(entity);
        // 保存角色与菜单关系
        roleMenuService.addOrUpdate(entity.getId(), role.getMenuIdList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SysRoleDTO role) {
        SysRoleEntity roleEntity = BeanConvertUtils.convert(role, SysRoleEntity.class);
        this.checkRoleName(roleEntity);
        this.updateById(roleEntity);
        // 更新角色与菜单关系
        roleMenuService.addOrUpdate(role.getId(), role.getMenuIdList());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteRoleIds(String[] roleIds) {
        if (ArrayUtils.isEmpty(roleIds)) {
            return;
        }
        // 删除角色
        for (String roleId : roleIds) {
            this.deleteById(roleId);
        }
        // 删除角色与菜单关联
        roleMenuService.deleteByRoleIds(roleIds);
        // 删除角色与用户关联
        userRoleService.deleteByRoleIds(roleIds);
    }

    @Override
    public List<SysRoleEntity> getRoleList(String userId) {
        return this.baseMapper.getRoleList(userId);
    }

    @Override
    public Set<String> getMenuIds(String roleId) {
        LambdaQueryWrapper<SysRoleMenuEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .select(SysRoleMenuEntity::getMenuId)
                .eq(SysRoleMenuEntity::getRoleId, roleId);
        List<SysRoleMenuEntity> list = roleMenuService.list(queryWrapper);
        Set<String> menuIdSet = list.stream().map(m -> m.getMenuId()).collect(Collectors.toSet());
        return menuIdSet;
    }


    private void checkRoleName(SysRoleEntity entity) {
        if (StringUtils.isBlank(entity.getName())) {
            return;
        }
        // 检查邮箱是否重复
        boolean repeat = repeatGlobal(SysRoleEntity::getId, entity.getId(), SysRoleEntity::getName, entity.getName());
        if (repeat) {
            throw new GlobalException(String.format("角色名称: %s 已被占用，换一个吧", entity.getName()));
        }
    }

    private void checkRoleCode(SysRoleEntity entity) {
        if (StringUtils.isBlank(entity.getCode())) {
            return;
        }
        // 检查邮箱是否重复
        boolean repeat = repeatGlobal(SysRoleEntity::getId, entity.getId(), SysRoleEntity::getCode, entity.getCode());
        if (repeat) {
            throw new GlobalException(String.format("角色编码: %s 已被占用，换一个吧", entity.getCode()));
        }
    }

}
