package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.utils.SpringContextUtils;
import com.gccloud.starter.core.dao.SysOrgDao;
import com.gccloud.starter.core.entity.SysOrgEntity;
import com.gccloud.starter.core.service.ISysOrgService;
import com.gccloud.starter.core.service.ISysUserService;
import com.gccloud.starter.core.shiro.UserUtils;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @Author tang.jiawen
 * @Date 2020/6/16
 * @Version 1.0.0
 */
@Service
public class SysOrgServiceImpl extends ServiceImpl<SysOrgDao, SysOrgEntity> implements ISysOrgService {

    @Autowired
    private ISysUserService userService;
    /**
     * 缓存
     */
    private static LoadingCache<String, Set<String>> DIRECT_CHILDREN_IDS_CACHE = CacheBuilder.newBuilder().expireAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<String, Set<String>>() {
        @Override
        public Set<String> load(String parentOrgId) {
            ISysOrgService orgService = SpringContextUtils.getBean(ISysOrgService.class);
            List<SysOrgEntity> children = orgService.getDirectChildren(UserUtils.getTenantId(), parentOrgId);
            Set<String> idSet = children.stream().map(o -> o.getId()).collect(Collectors.toSet());
            return idSet;
        }
    });

    private static LoadingCache<String, Set<String>> ALL_CHILDREN_IDS_CACHE = CacheBuilder.newBuilder().expireAfterWrite(10, TimeUnit.MINUTES).build(new CacheLoader<String, Set<String>>() {
        @Override
        public Set<String> load(String parentOrgId) {
            ISysOrgService orgService = SpringContextUtils.getBean(ISysOrgService.class);
            List<SysOrgEntity> allChildren = orgService.getAllChildren(UserUtils.getTenantId(), parentOrgId);
            Set<String> idSet = allChildren.stream().map(o -> o.getId()).collect(Collectors.toSet());
            return idSet;
        }
    });

    @Override
    public void add(SysOrgEntity org) {
        if (StringUtils.isBlank(org.getParentId())) {
            org.setParentId(GlobalConst.SUPER_PARENT_ID);
        }
        if (repeat(org)) {
            throw new GlobalException(MessageFormat.format("机构名称: {0} 已存在", org.getName()));
        }
        String parentIds = GlobalConst.Category.DEFAULT_PID + "";
        if (!org.getParentId().equals(GlobalConst.SUPER_PARENT_ID)) {
            SysOrgEntity parentOrg = getById(org.getParentId());
            if (parentOrg == null) {
                throw new GlobalException("父机构不存在");
            }
            parentIds = parentOrg.getParentIds() + "," + org.getParentId();
        }
        org.setParentIds(parentIds);
        if (org.getOrderNum() == null) {
            org.setOrderNum(0);
        }
        this.save(org);
        DIRECT_CHILDREN_IDS_CACHE.invalidateAll();
        ALL_CHILDREN_IDS_CACHE.invalidateAll();
    }

    @Override
    public void update(SysOrgEntity org) {
        if (StringUtils.isBlank(org.getParentId())) {
            org.setParentId(GlobalConst.SUPER_PARENT_ID);
        }
        if (org.getParentId().equals(org.getId())) {
            throw new GlobalException("不允许选择自己为父机构");
        }
        if (repeat(org)) {
            throw new GlobalException(MessageFormat.format("机构名称: {0} 已存在", org.getName()));
        }
        SysOrgEntity oldOrg = getById(org.getId());
        String newParentIds = GlobalConst.SUPER_PARENT_ID + "";
        if (!org.getParentId().equals(GlobalConst.SUPER_PARENT_ID)) {
            SysOrgEntity parentOrg = getById(org.getParentId());
            if (parentOrg == null) {
                throw new GlobalException("父机构不存在");
            }
            List<SysOrgEntity> children = getAllChildren(UserUtils.getTenantId(), oldOrg.getId());
            children.forEach(o -> {
                if (o.getId().equals(org.getParentId())) {
                    throw new GlobalException("不允许选择子机构为机构");
                }
            });
            newParentIds = parentOrg.getParentIds() + "," + org.getParentId();
        }
        this.updateById(org);
        // 替换原来机构下的所有子机构的parentIds
        String oldParentIds = oldOrg.getParentIds();
        List<SysOrgEntity> childList = getAllChildren(UserUtils.getTenantId(), oldOrg.getId());
        for (SysOrgEntity childCategory : childList) {
            // 更新所有子节点的parentIds
            LambdaUpdateWrapper<SysOrgEntity> updateWrapper = new LambdaUpdateWrapper<>();
            String pids = childCategory.getParentIds().replace(oldParentIds, newParentIds);
            updateWrapper.set(SysOrgEntity::getParentIds, pids);
            updateWrapper.eq(SysOrgEntity::getId, childCategory.getId());
            this.update(updateWrapper);
        }
        DIRECT_CHILDREN_IDS_CACHE.invalidateAll();
        ALL_CHILDREN_IDS_CACHE.invalidateAll();
    }

    @Override
    public void deleteById(String id) {
        LambdaQueryWrapper<SysOrgEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysOrgEntity::getParentId, id)
                .eq(SysOrgEntity::getTenantId, UserUtils.getTenantId());
        int count = this.count(queryWrapper);
        if (count > 0) {
            throw new GlobalException("存在子机构，不允许删除");
        }
        // 存在用户不给删除
        if (userService.existByOrgId(id)) {
            throw new GlobalException("机构下存在用户，不允许删除");
        }
        this.removeById(id);
        DIRECT_CHILDREN_IDS_CACHE.invalidateAll();
        ALL_CHILDREN_IDS_CACHE.invalidateAll();
    }

    @Override
    public boolean repeat(SysOrgEntity org) {
        LambdaQueryWrapper<SysOrgEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(SysOrgEntity::getName, org.getName());
        if (StringUtils.isNotBlank(org.getId())) {
            queryWrapper.ne(SysOrgEntity::getId, org.getId());
        }
        queryWrapper.eq(SysOrgEntity::getTenantId, UserUtils.tryGetTenantId());
        queryWrapper.eq(SysOrgEntity::getParentId, org.getParentId());
        int count = count(queryWrapper);
        return count > 0;
    }

    @Override
    public List<SysOrgEntity> getAllChildren(String tenantId, String parentOrgId) {
        String parentIds = GlobalConst.SUPER_PARENT_ID + "";
        if (!parentOrgId.equals(GlobalConst.SUPER_PARENT_ID)) {
            SysOrgEntity org = getEntityByFieldTenant(SysOrgEntity::getId, parentOrgId);
            if (org == null) {
                throw new GlobalException("机构不存在");
            }
            parentIds = org.getParentIds() + "," + org.getId();
        }
        LambdaQueryWrapper<SysOrgEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysOrgEntity::getParentIds, parentIds)
                .or()
                .likeRight(SysOrgEntity::getParentIds, parentIds + ",");
        List<SysOrgEntity> list = list(queryWrapper);
        return list;
    }

    @Override
    public List<SysOrgEntity> getDirectChildren(String tenantId, String parentOrgId) {
        LambdaQueryWrapper<SysOrgEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(SysOrgEntity::getParentId, parentOrgId);
        queryWrapper.eq(SysOrgEntity::getTenantId, tenantId);
        List<SysOrgEntity> list = list(queryWrapper);
        return list;
    }

    @Override
    public Set<String> getAllChildrenIds(String parentOrgId) {
        try {
            Set<String> ids = ALL_CHILDREN_IDS_CACHE.get(parentOrgId);
            return ids;
        } catch (ExecutionException e) {
            log.error(ExceptionUtils.getStackTrace(e));
        }
        return Sets.newHashSet();
    }

    @Override
    public Set<String> getDirectChildrenIds(String parentOrgId) {
        try {
            Set<String> ids = DIRECT_CHILDREN_IDS_CACHE.get(parentOrgId);
            return ids;
        } catch (ExecutionException e) {
            log.error(ExceptionUtils.getStackTrace(e));
        }
        return Sets.newHashSet();
    }


}
