package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.common.utils.BeanConvertUtils;
import com.gccloud.starter.core.dao.SysMenuDao;
import com.gccloud.starter.core.entity.SysMenuEntity;
import com.gccloud.starter.core.entity.SysRoleEntity;
import com.gccloud.starter.core.entity.SysRoleMenuEntity;
import com.gccloud.starter.core.service.*;
import com.gccloud.starter.core.shiro.UserUtils;
import com.gccloud.starter.core.vo.SysMenuVO;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @Author tang.jiawen
 * @Date 2020/6/23
 * @Version 1.0.0
 */

@Service
@Slf4j
public class SysMenuServiceImpl extends ServiceImpl<SysMenuDao, SysMenuEntity> implements ISysMenuService {

    @Autowired
    private ISysRoleMenuService roleMenuService;
    @Autowired
    private ISysRoleService roleService;
    @Autowired
    private ITreeService treeService;

    @Override
    public void add(SysMenuEntity menu) {
        if (repeat(menu)) {
            throw new GlobalException(MessageFormat.format("名称{0}已存在", menu.getName()));
        }
        this.save(menu);
        // 设置超级管理员角色
        SysRoleMenuEntity roleMenuEntity = new SysRoleMenuEntity();
        roleMenuEntity.setRoleId(GlobalConst.Role.SUPER_ADMIN_ROLE_ID);
        roleMenuEntity.setMenuId(menu.getId());
        roleMenuService.save(roleMenuEntity);
        // 设置租户管理员角色
        roleMenuEntity = new SysRoleMenuEntity();
        roleMenuEntity.setRoleId(GlobalConst.Role.TENANT_ADMIN_ROLE_ID);
        roleMenuEntity.setMenuId(menu.getId());
        roleMenuService.save(roleMenuEntity);
    }

    @Override
    public void update(SysMenuEntity menu) {
        this.updateById(menu);
    }

    @Override
    public List<SysMenuEntity> getByParentId(String parentId) {
        LambdaQueryWrapper<SysMenuEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysMenuEntity::getParentId, parentId)
                .orderByAsc(SysMenuEntity::getOrderNum);
        return this.list(queryWrapper);
    }

    @Override
    public List<SysMenuEntity> getMenuByType(Integer... type) {
        LambdaQueryWrapper<SysMenuEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .in(SysMenuEntity::getType, type)
                .orderByAsc(SysMenuEntity::getOrderNum);
        return list(queryWrapper);
    }

    @Override
    public List<SysMenuEntity> getMenuList() {
        QueryWrapper<SysMenuEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.select("distinct gc_menu.*");
        queryWrapper.orderByAsc("order_num");
        List<SysMenuEntity> list = this.list(queryWrapper);
        return list;
    }

    @Override
    public List<SysMenuEntity> getNavMenuList() {
        // 根据角色配置的
        List<SysRoleEntity> roleList = roleService.getRoleList(UserUtils.getCurrentUserId());
        List<String> roleIdList = roleList.stream().map(role -> role.getId()).collect(Collectors.toList());
        if (roleIdList == null || roleIdList.size() == 0) {
            throw new GlobalException("您没有访问任何页面的权限，请联系管理员！");
        }
        List<SysMenuEntity> list = baseMapper.getNavMenuList(roleIdList);
        return list;
    }

    @Override
    public void deleteById(String menuId) {
        this.removeById(menuId);
        roleMenuService.deleteByMenuId(menuId);
    }

    @Override
    public List<SysMenuVO> transToTree(List<SysMenuEntity> list) {
        List<SysMenuVO> rootVoList = Lists.newArrayList();
        List<SysMenuVO> voList = Lists.newArrayList();
        Map<String, SysMenuVO> menuMap = Maps.newHashMap();
        for (SysMenuEntity menu : list) {
            SysMenuVO vo = BeanConvertUtils.convert(menu, SysMenuVO.class);
            if (GlobalConst.SUPER_PARENT_ID.equals(menu.getParentId())) {
                rootVoList.add(vo);
            }
            menuMap.put(vo.getId(), vo);
            voList.add(vo);
        }
        treeService.transToTree(voList);
        return rootVoList;
    }

    public boolean repeat(SysMenuEntity menuEntity) {
        LambdaQueryWrapper<SysMenuEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(SysMenuEntity::getName, menuEntity.getName());
        if (StringUtils.isNotBlank(menuEntity.getId())) {
            queryWrapper.ne(SysMenuEntity::getId, menuEntity.getId());
        }
        if (StringUtils.isBlank(menuEntity.getParentId())) {
            menuEntity.setParentId(GlobalConst.SUPER_PARENT_ID);
        }
        queryWrapper.eq(SysMenuEntity::getParentId, menuEntity.getParentId());
        int count = count(queryWrapper);
        return count > 0;
    }

}
