package com.gccloud.starter.authority.service.service.impl;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.config.GlobalConfig;
import com.gccloud.starter.core.dao.SysDictItemDao;
import com.gccloud.starter.core.entity.SysDictEntity;
import com.gccloud.starter.core.entity.SysDictItemEntity;
import com.gccloud.starter.core.dto.SysLoggerDTO;
import com.gccloud.starter.core.service.ISysDictItemService;
import com.gccloud.starter.core.service.ISysDictService;
import com.gccloud.starter.core.service.ISysLoggerService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.List;

/**
 * 业务字典
 *
 * @Author maoshufeng
 * @Date 2020-06-16
 * @Version 1.0.0
 */
@Service
@Slf4j
public class SysLoggerServiceImpl extends ServiceImpl<SysDictItemDao, SysDictItemEntity> implements ISysLoggerService {

    @Autowired
    private ISysDictItemService dictItemService;
    @Autowired
    private ISysDictService dictService;
    @Autowired
    private GlobalConfig globalConfig;

    @Override
    public List<SysLoggerDTO> getList() {
        List<SysLoggerDTO> dtoList = Lists.newArrayList();
        List<SysDictItemEntity> itemList = getByDictCode(GlobalConst.Logger.LOGGER_DICT_CODE);
        for (SysDictItemEntity item : itemList) {
            SysLoggerDTO loggerDTO = new SysLoggerDTO(item.getName(), item.getValue());
            loggerDTO.setId(item.getId());
            loggerDTO.setLevel(item.getValue());
            loggerDTO.setPackagePreffix(item.getName());
            dtoList.add(loggerDTO);
        }
        return dtoList;
    }

    @Override
    @PostConstruct
    public void init() {
        log.info("开始初始化日志级别");
        if (globalConfig.getLogger().getEnable()) {
            this.updateServerLogger(this.getList());
            log.info("初始化日志级别完成");
            return;
        }
        log.info("未启用日志级别初始化，但不影响使用，如果需要配置，请配置 gc.starter.logger.enable=true即可");
    }

    @Override
    public void update(List<SysLoggerDTO> logList) {
        LambdaQueryWrapper<SysDictEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysDictEntity::getCode, GlobalConst.Logger.LOGGER_DICT_CODE)
                .eq(SysDictEntity::getTenantId, GlobalConst.Tenant.SUPER_TENANTID);
        SysDictEntity dictEntity = dictService.getOne(queryWrapper);
        // 删除之前关联的
        deleteAll();
        for (SysLoggerDTO loggerDTO : logList) {
            if (StringUtils.isBlank(loggerDTO.getPackagePreffix())) {
                log.error("包前缀不能为空");
                throw new GlobalException("包前缀不能为空");
            }
            SysDictItemEntity dictItemEntity = new SysDictItemEntity(dictEntity.getCode(), loggerDTO.getPackagePreffix(), loggerDTO.getLevel(), null, 999, GlobalConst.Logger.Status.NORMAL);
            dictItemEntity.setTenantId(GlobalConst.Tenant.SUPER_TENANTID);
            dictItemService.add(dictItemEntity);
        }
        init();
    }

    private void updateServerLogger(List<SysLoggerDTO> logList) {
        if (logList == null || logList.size() == 0) {
            return;
        }
        LoggerContext logContext = (LoggerContext) LoggerFactory.getILoggerFactory();
        for (SysLoggerDTO log : logList) {
            logContext.getLogger(log.getPackagePreffix()).setLevel(Level.toLevel(log.getLevel()));
        }
    }

    private void deleteAll() {
        List<SysDictItemEntity> list = getByDictCode(GlobalConst.Logger.LOGGER_DICT_CODE);
        if (list == null || list.size() == 0) {
            return;
        }
        LoggerContext logContext = (LoggerContext) LoggerFactory.getILoggerFactory();
        for (SysDictItemEntity itemEntity : list) {
            if (GlobalConst.Logger.DEFAULT_ROOT_PACKAGE_PREFFIX.equalsIgnoreCase(itemEntity.getName())) {
                continue;
            }
            logContext.getLogger(itemEntity.getName()).setLevel(null);
            dictItemService.getBaseDao().deleteById(itemEntity.getId());
        }
    }

    private List<SysDictItemEntity> getByDictCode(String dictCode) {
        LambdaQueryWrapper<SysDictItemEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysDictItemEntity::getStatus, GlobalConst.DictItem.Status.NORMAL)
                .eq(SysDictItemEntity::getDictCode, dictCode)
                .eq(SysDictItemEntity::getTenantId, GlobalConst.Tenant.SUPER_TENANTID);
        return list(queryWrapper);
    }
}
