package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.dto.SearchDTO;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.core.dao.SysDictDao;
import com.gccloud.starter.core.entity.SysDictEntity;
import com.gccloud.starter.core.entity.SysDictItemEntity;
import com.gccloud.starter.core.service.ISysDictItemService;
import com.gccloud.starter.core.service.ISysDictService;
import com.gccloud.starter.mybatis.page.PageVO;
import com.gccloud.starter.mybatis.utils.QueryWrapperUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.text.MessageFormat;
import java.util.List;

/**
 * 业务字典
 *
 * @Author maoshufeng
 * @Date 2020-06-16
 * @Version 1.0.0
 */
@Service
@Slf4j
public class SysDictServiceImpl extends ServiceImpl<SysDictDao, SysDictEntity> implements ISysDictService {

    @Autowired
    private ISysDictItemService dictItemService;

    @Override
    public PageVO<SysDictEntity> getPage(SearchDTO searchDTO) {
        LambdaQueryWrapper<SysDictEntity> queryWrapper = QueryWrapperUtils.wrapperLike(new LambdaQueryWrapper<>(), searchDTO.getSearchKey(), SysDictEntity::getName, SysDictEntity::getCode, SysDictEntity::getRemark);
        queryWrapper.orderByDesc(SysDictEntity::getUpdateDate);
        return page(searchDTO, queryWrapper);
    }

    @Override
    public void add(SysDictEntity sysDictEntity) {
        checkRepeat(sysDictEntity);
        this.save(sysDictEntity);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteById(String id) {
        // 根据id获取字典，判断业务字典是否存在
        SysDictEntity dictEntity = this.getById(id);
        if (dictEntity == null) {
            throw new GlobalException(("没有该业务字典"));
        }
        String dictCode = dictEntity.getCode();
        // 根据dictCode获取业务字典项列表数组
        List<SysDictItemEntity> itemList = dictItemService.getByDictCode(dictCode);
        // 删除字典项数组
        for (SysDictItemEntity item : itemList) {
            dictItemService.deleteById(item.getId());
        }
        // 删除业务字典
        this.removeById(id);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SysDictEntity dict) {
        checkRepeat(dict);
        this.updateById(dict);
    }

    @Override
    public SysDictEntity getByCode(String code) {
        LambdaQueryWrapper<SysDictEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(SysDictEntity::getCode, code);
        return getOne(queryWrapper);
    }

    private void checkRepeat(SysDictEntity sysDictEntity) {
        // 根据name查重
        if (repeatGlobal(SysDictEntity::getId, sysDictEntity.getId(), SysDictEntity::getName, sysDictEntity.getName())) {
            throw new GlobalException(MessageFormat.format("字典名称：{0} 已存在", sysDictEntity.getName()));
        }
        // 根据code查重
        if (repeatGlobal(SysDictEntity::getId, sysDictEntity.getId(), SysDictEntity::getCode, sysDictEntity.getCode())) {
            throw new GlobalException(MessageFormat.format("字典编码：{0} 已存在", sysDictEntity.getCode()));
        }
    }
}
