package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.core.dao.SysDictItemDao;
import com.gccloud.starter.core.dto.SysDictiItemSearchDTO;
import com.gccloud.starter.core.entity.SysDictItemEntity;
import com.gccloud.starter.core.service.ISysDictItemService;
import com.gccloud.starter.mybatis.page.PageVO;
import com.gccloud.starter.mybatis.utils.QueryWrapperUtils;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.List;

/**
 * 业务字典项业务实现类
 *
 * @Author maoshufeng
 * @Date 2020-06-16
 * @Version 1.0.0
 */
@Service
@Slf4j
public class SysDictItemServiceImpl extends ServiceImpl<SysDictItemDao, SysDictItemEntity> implements ISysDictItemService {

    @Override
    public PageVO<SysDictItemEntity> getPageByDictCode(SysDictiItemSearchDTO sysDictiItemSearchDTO) {
        LambdaQueryWrapper<SysDictItemEntity> queryWrapper = QueryWrapperUtils.wrapperLike(new LambdaQueryWrapper<>(), sysDictiItemSearchDTO.getSearchKey(), SysDictItemEntity::getName, SysDictItemEntity::getValue, SysDictItemEntity::getRemark);
        queryWrapper
                .eq(SysDictItemEntity::getDictCode, sysDictiItemSearchDTO.getDictCode())
                .orderByAsc(SysDictItemEntity::getOrderNum);
        return page(sysDictiItemSearchDTO, queryWrapper);
    }

    @Override
    public void add(SysDictItemEntity dictItemEntity) {
        if (checkRepeat(dictItemEntity)) {
            throw new GlobalException(MessageFormat.format("字典编码：{0} 中的字典项名称：{1} 已被使用", dictItemEntity.getDictCode(), dictItemEntity.getName()));
        }
        // 状态设置默认值1
        if (dictItemEntity.getStatus() == null) {
            dictItemEntity.setStatus(GlobalConst.DictItem.Status.NORMAL);
        }
        // 排序设置默认值0
        if (dictItemEntity.getOrderNum() == null) {
            dictItemEntity.setOrderNum(GlobalConst.DictItem.DEFAULT_ORDER_NUM);
        }
        this.save(dictItemEntity);
    }

    @Override
    public void update(SysDictItemEntity dictItemEntity) {
        if (checkRepeat(dictItemEntity)) {
            throw new GlobalException(MessageFormat.format("字典编码：{0} 中的字典项名称：{1} 已被使用", dictItemEntity.getDictCode(), dictItemEntity.getName()));
        }
        this.updateById(dictItemEntity);
    }

    @Override
    public List<SysDictItemEntity> getByDictCode(String dictCode) {
        LambdaQueryWrapper<SysDictItemEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysDictItemEntity::getStatus, GlobalConst.DictItem.Status.NORMAL)
                .eq(SysDictItemEntity::getDictCode, dictCode);
        List<SysDictItemEntity> list = this.list(queryWrapper);
        return list;
    }

    @Override
    public <T> List<T> getByDictCode(String dictCode, Class<T> resultClass) {
        List<SysDictItemEntity> dictItemEntities = this.getByDictCode(dictCode);
        List<T> list = Lists.newArrayList();
        if (dictItemEntities != null && dictItemEntities.size() > 0) {
            for (SysDictItemEntity dictEntity : dictItemEntities) {
                if (StringUtils.isNotBlank(dictEntity.getValue())) {
                    try {
                        list.add(new Gson().fromJson(dictEntity.getValue(), resultClass));
                    } catch (Exception e) {
                        log.error(ExceptionUtils.getStackTrace(e));
                    }
                }
            }
        }
        return list;
    }

    @Override
    public void changeStatusById(String id) {
        SysDictItemEntity dictItemEntity = this.getById(id);
        if (dictItemEntity == null) {
            throw new GlobalException("没有该业务字典项");
        }
        Integer status = dictItemEntity.getStatus();
        if (status == null) {
            status = GlobalConst.DictItem.Status.FORBIDDEN;
        }
        status = status.equals(GlobalConst.DictItem.Status.NORMAL) ? GlobalConst.DictItem.Status.FORBIDDEN : GlobalConst.DictItem.Status.NORMAL;
        LambdaUpdateWrapper<SysDictItemEntity> updateWrapper = new LambdaUpdateWrapper<>();
        updateWrapper
                .eq(SysDictItemEntity::getId, id)
                .set(SysDictItemEntity::getStatus, status);
        this.update(updateWrapper);
    }

    /**
     * 根据名称和dictCode查重
     *
     * @param dictItemEntity
     */
    @Override
    public boolean checkRepeat(SysDictItemEntity dictItemEntity) {
        String id = dictItemEntity.getId();
        LambdaQueryWrapper<SysDictItemEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysDictItemEntity::getName, dictItemEntity.getName())
                .eq(SysDictItemEntity::getDictCode, dictItemEntity.getDictCode());
        if (StringUtils.isNotBlank(id)) {
            queryWrapper.ne(SysDictItemEntity::getId, dictItemEntity.getId());
        }
        int count = count(queryWrapper);
        return count > 0;
    }
}
