package com.gccloud.starter.authority.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.gccloud.starter.common.constant.GlobalConst;
import com.gccloud.starter.common.dto.SearchDTO;
import com.gccloud.starter.common.exception.GlobalException;
import com.gccloud.starter.core.dao.SysCategoryDao;
import com.gccloud.starter.core.entity.SysCategoryEntity;
import com.gccloud.starter.core.service.ISysCategoryService;
import com.gccloud.starter.mybatis.page.PageVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import java.text.MessageFormat;
import java.util.List;

/**
 * @Author maoshufeng
 * @Date 2020-06-18
 * @Version 1.0.0
 */
@Service
@Slf4j
public class SysCategoryServiceImpl extends ServiceImpl<SysCategoryDao, SysCategoryEntity> implements ISysCategoryService {

    @Override
    public PageVO<SysCategoryEntity> getPage(SearchDTO searchDTO) {
        LambdaQueryWrapper<SysCategoryEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysCategoryEntity::getParentId, GlobalConst.Category.DEFAULT_PID)
                .orderByDesc(SysCategoryEntity::getUpdateDate);
        PageVO<SysCategoryEntity> page = page(searchDTO, queryWrapper);
        return page;
    }

    @Override
    @CacheEvict(value = "SysCategoryEntity", beforeInvocation = true, allEntries = true)
    public void add(SysCategoryEntity category) {
        // 字典值全局唯一
        if (repeatGlobal(SysCategoryEntity::getCode, category.getCode())) {
            throw new GlobalException(MessageFormat.format("字典编码：{0} 已存在，建议前面加上父字典的编码作为前缀", category.getCode()));
        }
        if (repeatName(category)) {
            throw new GlobalException(MessageFormat.format("字典名称：{0} 已存在", category.getName()));
        }
        // 不传父id时设置为一级字典
        if (StringUtils.isBlank(category.getParentId())) {
            category.setParentId(GlobalConst.Category.DEFAULT_PID);
        }
        String parentIds = GlobalConst.Category.DEFAULT_PID + "";
        // 父字典如果状态禁用，不允许新增
        if (!category.getParentId().equals(GlobalConst.Category.DEFAULT_PID)) {
            SysCategoryEntity parentCategory = getById(category.getParentId());
            if (parentCategory == null) {
                throw new GlobalException("父字典不存在");
            }
            if (GlobalConst.Category.Status.FORBIDDEN == parentCategory.getStatus()) {
                throw new GlobalException(String.format("父字典：%s 下不允许创建子字典", parentCategory.getName()));
            }
            parentIds = parentCategory.getParentIds() + "," + category.getParentId();
        }
        category.setParentIds(parentIds);
        // 设置默认排序为0
        if (category.getOrderNum() == null) {
            category.setOrderNum(GlobalConst.Category.DEFAULT_ORDER_NUM);
        }
        // 设置默认状态为1
        if (category.getStatus() == null) {
            category.setStatus(GlobalConst.Category.Status.NORMAL);
        }
        this.save(category);
    }

    @Override
    @CacheEvict(value = "SysCategoryEntity", beforeInvocation = true, allEntries = true)
    public void deleteById(String id) {
        LambdaQueryWrapper<SysCategoryEntity> countWrapper = new LambdaQueryWrapper<>();
        countWrapper
                .select(SysCategoryEntity::getId)
                .in(SysCategoryEntity::getParentId, id);
        int count = count(countWrapper);
        if (count > 0) {
            throw new GlobalException("存在子字典,不允许删除");
        }
        this.removeById(id);
    }

    @Override
    @Cacheable(value = "SysCategoryEntity", key = "#code", unless = "#result!=null&&#result.size()!=0")
    public List<SysCategoryEntity> getChildren(String code) {
        SysCategoryEntity category = getEntityByField(SysCategoryEntity::getCode, code);
        if (category == null) {
            throw new GlobalException("字典不存在");
        }
        String parentIds = category.getParentIds() + "," + category.getId();
        LambdaQueryWrapper<SysCategoryEntity> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper
                .eq(SysCategoryEntity::getParentIds, parentIds)
                .or()
                .likeRight(SysCategoryEntity::getParentIds, parentIds + ",");
        List<SysCategoryEntity> list = list(queryWrapper);
        return list;
    }

    @Override
    public List<SysCategoryEntity> getChildren(List<String> parentIdsList) {
        LambdaQueryWrapper<SysCategoryEntity> queryWrapper = new LambdaQueryWrapper<>();
        for (String parentIds : parentIdsList) {
            queryWrapper.or().likeRight(SysCategoryEntity::getParentIds, parentIds);
        }
        return list(queryWrapper);
    }

    @Override
    public boolean repeatName(SysCategoryEntity category) {
        LambdaQueryWrapper<SysCategoryEntity> countWrapper = new LambdaQueryWrapper<>();
        countWrapper
                .eq(SysCategoryEntity::getName, category.getName())
                .eq(SysCategoryEntity::getParentId, category.getParentId());
        if (StringUtils.isBlank(category.getId())) {
            countWrapper.ne(SysCategoryEntity::getId, category.getId());
        }
        return count(countWrapper) > 0;
    }


    @Override
    @CacheEvict(value = "SysCategoryEntity", beforeInvocation = true, allEntries = true)
    public void update(SysCategoryEntity category) {
        // 不传父id时设置为一级字典
        if (StringUtils.isBlank(category.getParentId())) {
            category.setParentId(GlobalConst.Category.DEFAULT_PID);
        }
        if (category.getParentId().equals(category.getId())) {
            throw new GlobalException("不允许选择自己为父字典");
        }
        String code = category.getCode();
        if (repeatGlobal(SysCategoryEntity::getId, category.getId(), SysCategoryEntity::getCode, code)) {
            throw new GlobalException(MessageFormat.format("字典编码：{0} 已存在，建议前面加上父字典的编码作为前缀", code));
        }
        if (repeatName(category)) {
            throw new GlobalException(MessageFormat.format("字典名称：{0} 已存在", category.getName()));
        }
        SysCategoryEntity oldCategory = getById(category.getId());
        String newParentIds = GlobalConst.Category.DEFAULT_PID + "";
        if (!category.getParentId().equals(GlobalConst.Category.DEFAULT_PID)) {
            SysCategoryEntity parentCategory = getById(category.getParentId());
            if (parentCategory == null) {
                throw new GlobalException("父字典不存在");
            }
            if (GlobalConst.Category.Status.FORBIDDEN == parentCategory.getStatus()) {
                throw new GlobalException(String.format("父字典：{} 下不允许修改子字典", parentCategory.getName()));
            }
            // 不允许选择子节点为父节点
            List<SysCategoryEntity> children = getChildren(oldCategory.getCode());
            children.forEach(c -> {
                if (category.getParentId().equals(c.getId())) {
                    throw new GlobalException("不允许选择子字典为父字典");
                }
            });
            newParentIds = parentCategory.getParentIds() + "," + category.getParentId();
        }
        category.setParentIds(newParentIds);
        this.updateById(category);
        String oldParentIds = oldCategory.getParentIds() + "," + oldCategory.getId();
        LambdaQueryWrapper<SysCategoryEntity> childQuery = new LambdaQueryWrapper<>();
        childQuery.likeRight(SysCategoryEntity::getParentIds, oldParentIds);
        List<SysCategoryEntity> childList = this.list(childQuery);
        newParentIds = newParentIds + "," + category.getId();
        for (SysCategoryEntity childCategory : childList) {
            // 更新所有子节点的parentIds
            LambdaUpdateWrapper<SysCategoryEntity> updateWrapper = new LambdaUpdateWrapper<>();
            String pids = childCategory.getParentIds().replace(oldParentIds, newParentIds);
            updateWrapper.set(SysCategoryEntity::getParentIds, pids);
            updateWrapper.eq(SysCategoryEntity::getId, childCategory.getId());
            this.update(updateWrapper);
        }
    }
}
