/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses for this work are available. These replace the above
 * ASL 2.0 and offer limited warranties, support, maintenance, and commercial
 * database integrations.
 *
 * For more information, please visit: http://www.jooq.org/licenses
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package org.jooq.tools.jdbc;

import static java.util.Collections.nCopies;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.CallableStatement;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.Date;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.SQLXML;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Map;

import org.jooq.Record;

/**
 * A mock statement.
 * <p>
 * This statement can be used to mock any of these:
 * <ul>
 * <li> {@link Statement}</li>
 * <li> {@link PreparedStatement}</li>
 * <li> {@link CallableStatement}</li>
 * </ul>
 *
 * @author Lukas Eder
 * @see MockConnection
 */
public class MockStatement extends JDBC41Statement implements CallableStatement {

    private final MockConnection        connection;

    private final MockDataProvider      data;
    private final List<String>          sql;
    private final List<List<Object>>    bindings;
    private final List<Integer>         outParameterTypes;
    private MockResult[]                result;
    private int                         resultIndex;
    private boolean                     resultWasNull;
    private boolean                     isClosed;

    // Execution parameters
    int                                 resultSetType        = ResultSet.TYPE_FORWARD_ONLY;
    int                                 resultSetConcurrency = ResultSet.CONCUR_READ_ONLY;
    int                                 resultSetHoldability = ResultSet.CLOSE_CURSORS_AT_COMMIT;
    int                                 autoGeneratedKeys    = Statement.NO_GENERATED_KEYS;
    int[]                               columnIndexes;
    String[]                            columnNames;

    // Statement properties
    private int                         queryTimeout;
    private int                         maxRows;

    public MockStatement(MockConnection connection, MockDataProvider data) {
        this(connection, data, null);
    }

    public MockStatement(MockConnection connection, MockDataProvider data, String sql) {
        this.connection = connection;
        this.data = data;
        this.sql = new ArrayList<String>();
        this.bindings = new ArrayList<List<Object>>();
        this.outParameterTypes = new ArrayList<Integer>();

        if (sql != null) {
            this.sql.add(sql);
        }

        this.bindings.add(new ArrayList<Object>());
    }

    // -------------------------------------------------------------------------
    // XXX: Utilities
    // -------------------------------------------------------------------------

    private List<Object> bindings() {
        return bindings.get(bindings.size() - 1);
    }

    private void ensureBindingsCapacity(int index) {
        List<Object> b = bindings();

        if (b.size() < index) {
            b.addAll(nCopies(index - b.size(), null));
        }
    }

    private void ensureOutParameterTypesCapacity(int index) {
        if (outParameterTypes.size() < index) {
            outParameterTypes.addAll(nCopies(index - outParameterTypes.size(), (Integer) null));
        }
    }

    private void checkNotClosed() throws SQLException {
        if (isClosed) {
            throw new SQLException("Connection is already closed");
        }
    }

    @Override
    public Connection getConnection() throws SQLException {
        return connection;
    }

    // -------------------------------------------------------------------------
    // XXX: Executing queries
    // -------------------------------------------------------------------------

    @SuppressWarnings("unused")
    private boolean execute0(
            String localSql,
            int localResultSetType,
            int localResultSetConcurrency,
            int localResultSetHoldability,
            int localAutoGeneratedKeys,
            int[] localColumnIndexes,
            String[] localColumnNames) throws SQLException {

        checkNotClosed();

        MockExecuteContext context = new MockExecuteContext(
            new String[] { localSql },
            new Object[][] { bindings().toArray() },
            localAutoGeneratedKeys,
            localColumnIndexes,
            localColumnNames,
            unbox(outParameterTypes)
        );

        result = data.execute(context);
        return result != null && result.length > 0 && result[resultIndex].data != null;
    }

    private static final int[] unbox(List<Integer> list) {
        int[] array = new int[list.size()];

        for (int i = 0; i < array.length; i++) {
            Integer value = list.get(i);
            array[i] = value == null ? 0 : value;
        }

        return array;
    }

    @Override
    public ResultSet getGeneratedKeys() throws SQLException {
        return getResultSet();
    }

    @Override
    public boolean getMoreResults() throws SQLException {
        return getMoreResults(CLOSE_CURRENT_RESULT);
    }

    @Override
    public boolean getMoreResults(int current) throws SQLException {
        return (result != null && ++resultIndex < result.length);
    }

    @SuppressWarnings("resource")
    @Override
    public ResultSet getResultSet() throws SQLException {
        checkNotClosed();
        return (result != null && resultIndex < result.length && result[resultIndex].data != null)
            ? new MockResultSet(result[resultIndex].data, maxRows)
            : null;
    }

    @Override
    public int getUpdateCount() throws SQLException {
        checkNotClosed();
        return (result != null && resultIndex < result.length) && result[resultIndex].data == null ? result[resultIndex].rows : -1;
    }

    @Override
    public ResultSet executeQuery() throws SQLException {
        return executeQuery(sql.get(0));
    }

    @Override
    public ResultSet executeQuery(String localSql) throws SQLException {
        execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, autoGeneratedKeys, columnIndexes, columnNames);
        return getResultSet();
    }

    @Override
    public boolean execute() throws SQLException {
        return execute(sql.get(0));
    }

    @Override
    public boolean execute(String localSql) throws SQLException {
        return execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, autoGeneratedKeys, columnIndexes, columnNames);
    }

    @Override
    public boolean execute(String localSql, int localAutoGeneratedKeys) throws SQLException {
        return execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, localAutoGeneratedKeys, null, null);
    }

    @Override
    public boolean execute(String localSql, int[] localColumnIndexes) throws SQLException {
        return execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, Statement.RETURN_GENERATED_KEYS, localColumnIndexes, null);
    }

    @Override
    public boolean execute(String localSql, String[] localColumnNames) throws SQLException {
        return execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, Statement.RETURN_GENERATED_KEYS, null, localColumnNames);
    }

    @Override
    public int executeUpdate() throws SQLException {
        return executeUpdate(sql.get(0));
    }

    @Override
    public int executeUpdate(String localSql) throws SQLException {
        execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, autoGeneratedKeys, columnIndexes, columnNames);
        return getUpdateCount();
    }

    @Override
    public int executeUpdate(String localSql, int localAutoGeneratedKeys) throws SQLException {
        execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, localAutoGeneratedKeys, null, null);
        return getUpdateCount();
    }

    @Override
    public int executeUpdate(String localSql, int[] localColumnIndexes) throws SQLException {
        execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, Statement.RETURN_GENERATED_KEYS, localColumnIndexes, null);
        return getUpdateCount();
    }

    @Override
    public int executeUpdate(String localSql, String[] localColumnNames) throws SQLException {
        execute0(localSql, resultSetType, resultSetConcurrency, resultSetHoldability, Statement.RETURN_GENERATED_KEYS, null, localColumnNames);
        return getUpdateCount();
    }

    // -------------------------------------------------------------------------
    // XXX: Batch processing
    // -------------------------------------------------------------------------

    @Override
    public void addBatch() throws SQLException {
        checkNotClosed();
        bindings.add(new ArrayList<Object>());
    }

    @Override
    public void addBatch(String localSql) throws SQLException {
        checkNotClosed();
        sql.add(localSql);
    }

    @Override
    public void clearBatch() throws SQLException {
        checkNotClosed();
        sql.clear();
        bindings.clear();
        bindings.add(new ArrayList<Object>());
    }

    @Override
    public int[] executeBatch() throws SQLException {
        checkNotClosed();

        Object[][] matrix = new Object[bindings.size() - 1][];
        for (int i = 0; i < bindings.size() - 1; i++)
            matrix[i] = bindings.get(i).toArray();

        result = data.execute(new MockExecuteContext(sql.toArray(new String[0]), matrix));

        int[] rows = new int[result.length];
        for (int i = 0; i < result.length; i++)
            rows[i] = result[i].rows;

        return rows;
    }

    // -------------------------------------------------------------------------
    // XXX: Bind variables
    // -------------------------------------------------------------------------

    @Override
    public void clearParameters() throws SQLException {
        checkNotClosed();
        List<Object> b = bindings();

        for (int i = 0; i < b.size(); i++)
            b.set(i, null);
    }

    @Override
    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, null);
    }

    @Override
    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, null);
    }

    @Override
    public void setBoolean(int parameterIndex, boolean x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setByte(int parameterIndex, byte x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setShort(int parameterIndex, short x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setInt(int parameterIndex, int x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setLong(int parameterIndex, long x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setFloat(int parameterIndex, float x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setDouble(int parameterIndex, double x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setString(int parameterIndex, String x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setNString(int parameterIndex, String value) throws SQLException {
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, value);
    }

    @Override
    public void setBytes(int parameterIndex, byte[] x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setDate(int parameterIndex, Date x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setTime(int parameterIndex, Time x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setObject(int parameterIndex, Object x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setObject(int parameterIndex, Object x, int targetSqlType, int scaleOrLength) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setURL(int parameterIndex, URL x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x);
    }

    @Override
    public void setArray(int parameterIndex, Array x) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        bindings().set(parameterIndex - 1, x.getArray());
    }

    // -------------------------------------------------------------------------
    // XXX: Bind variables from CallableStatement
    // -------------------------------------------------------------------------

    private Record outParameters() throws SQLException {
        if (result == null || result.length == 0 || result[0].data == null || result[0].data.size() == 0)
            throw new SQLException("No OUT Parameters available");

        return result[0].data.get(0);
    }

    private int translate(int parameterIndex) throws SQLException {
        if (parameterIndex > outParameterTypes.size())
            throw new SQLException("OUT parameter index too high: " + parameterIndex);

        int index = -1;

        for (int i = 0; i < parameterIndex; i++)
            if (outParameterTypes.get(i) != null)
                index++;

        return index;
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType) throws SQLException {
        checkNotClosed();
        ensureBindingsCapacity(parameterIndex);
        ensureOutParameterTypesCapacity(parameterIndex);
        outParameterTypes.set(parameterIndex - 1, sqlType);
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType, int scale) throws SQLException {
        registerOutParameter(parameterIndex, sqlType);
    }

    @Override
    public void registerOutParameter(int parameterIndex, int sqlType, String typeName) throws SQLException {
        registerOutParameter(parameterIndex, sqlType);
    }

    @Override
    public boolean wasNull() throws SQLException {
        return resultWasNull;
    }

    @Override
    public String getString(int parameterIndex) throws SQLException {
        String value = outParameters().get(translate(parameterIndex), String.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public String getNString(int parameterIndex) throws SQLException {
        String value = outParameters().get(translate(parameterIndex), String.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public boolean getBoolean(int parameterIndex) throws SQLException {
        Boolean value = outParameters().get(translate(parameterIndex), Boolean.class);
        resultWasNull = value == null;
        return value == null ? false : value;
    }

    @Override
    public byte getByte(int parameterIndex) throws SQLException {
        Byte value = outParameters().get(translate(parameterIndex), Byte.class);
        resultWasNull = value == null;
        return value == null ? (byte) 0 : value;
    }

    @Override
    public short getShort(int parameterIndex) throws SQLException {
        Short value = outParameters().get(translate(parameterIndex), Short.class);
        resultWasNull = value == null;
        return value == null ? (short) 0 : value;
    }

    @Override
    public int getInt(int parameterIndex) throws SQLException {
        Integer value = outParameters().get(translate(parameterIndex), Integer.class);
        resultWasNull = value == null;
        return value == null ? 0 : value;
    }

    @Override
    public long getLong(int parameterIndex) throws SQLException {
        Long value = outParameters().get(translate(parameterIndex), Long.class);
        resultWasNull = value == null;
        return value == null ? 0L : value;
    }

    @Override
    public float getFloat(int parameterIndex) throws SQLException {
        Float value = outParameters().get(translate(parameterIndex), Float.class);
        resultWasNull = value == null;
        return value == null ? 0.0f : value;
    }

    @Override
    public double getDouble(int parameterIndex) throws SQLException {
        Double value = outParameters().get(translate(parameterIndex), Double.class);
        resultWasNull = value == null;
        return value == null ? 0.0 : value;
    }

    @Override
    public BigDecimal getBigDecimal(int parameterIndex, int scale) throws SQLException {
        BigDecimal value = outParameters().get(translate(parameterIndex), BigDecimal.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public BigDecimal getBigDecimal(int parameterIndex) throws SQLException {
        BigDecimal value = outParameters().get(translate(parameterIndex), BigDecimal.class);
        resultWasNull = value == null;
        return value;
    }
    @Override
    public byte[] getBytes(int parameterIndex) throws SQLException {
        byte[] value = outParameters().get(translate(parameterIndex), byte[].class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Date getDate(int parameterIndex) throws SQLException {
        Date value = outParameters().get(translate(parameterIndex), Date.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Date getDate(int parameterIndex, Calendar cal) throws SQLException {
        Date value = outParameters().get(translate(parameterIndex), Date.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Time getTime(int parameterIndex) throws SQLException {
        Time value = outParameters().get(translate(parameterIndex), Time.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Time getTime(int parameterIndex, Calendar cal) throws SQLException {
        Time value = outParameters().get(translate(parameterIndex), Time.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Timestamp getTimestamp(int parameterIndex) throws SQLException {
        Timestamp value = outParameters().get(translate(parameterIndex), Timestamp.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Timestamp getTimestamp(int parameterIndex, Calendar cal) throws SQLException {
        Timestamp value = outParameters().get(translate(parameterIndex), Timestamp.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Object getObject(int parameterIndex) throws SQLException {
        Object value = outParameters().get(translate(parameterIndex));
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Object getObject(int parameterIndex, Map<String, Class<?>> map) throws SQLException {
        Object value = outParameters().get(translate(parameterIndex));
        resultWasNull = value == null;
        return value;
    }

    @Override
    public Array getArray(int parameterIndex) throws SQLException {
        Array value = outParameters().get(translate(parameterIndex), Array.class);
        resultWasNull = value == null;
        return value;
    }

    @Override
    public URL getURL(int parameterIndex) throws SQLException {
        URL value = outParameters().get(translate(parameterIndex), URL.class);
        resultWasNull = value == null;
        return value;
    }

    // -------------------------------------------------------------------------
    // XXX: Ignored operations
    // -------------------------------------------------------------------------

    @Override
    public boolean isClosed() throws SQLException {
        return isClosed;
    }

    @Override
    public void close() throws SQLException {
        isClosed = true;
    }

    @Override
    public void cancel() throws SQLException {
        isClosed = true;
    }

    @Override
    public int getMaxFieldSize() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public void setMaxFieldSize(int max) throws SQLException {
        checkNotClosed();
    }

    @Override
    public int getMaxRows() throws SQLException {
        checkNotClosed();
        return maxRows;
    }

    @Override
    public void setMaxRows(int max) throws SQLException {
        checkNotClosed();
        this.maxRows = max;
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        checkNotClosed();
    }

    @Override
    public int getQueryTimeout() throws SQLException {
        checkNotClosed();
        return queryTimeout;
    }

    @Override
    public void setQueryTimeout(int seconds) throws SQLException {
        checkNotClosed();
        this.queryTimeout = seconds;
    }

    @Override
    public void setFetchDirection(int direction) throws SQLException {
        checkNotClosed();
    }

    @Override
    public int getFetchDirection() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public void setFetchSize(int rows) throws SQLException {
        checkNotClosed();
    }

    @Override
    public int getFetchSize() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public int getResultSetConcurrency() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public int getResultSetType() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public int getResultSetHoldability() throws SQLException {
        checkNotClosed();
        return 0;
    }

    @Override
    public void setPoolable(boolean poolable) throws SQLException {
        checkNotClosed();
    }

    @Override
    public boolean isPoolable() throws SQLException {
        checkNotClosed();
        return false;
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        checkNotClosed();
        return null;
    }

    @Override
    public void clearWarnings() throws SQLException {
        checkNotClosed();
    }

    @Override
    public void setCursorName(String name) throws SQLException {
        checkNotClosed();
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        checkNotClosed();
        throw new SQLException("Can this be ignored?");
    }

    // -------------------------------------------------------------------------
    // XXX: Unsupported operations
    // -------------------------------------------------------------------------

    @Override
    public <T> T unwrap(Class<T> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public boolean isWrapperFor(Class<?> iface) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setAsciiStream(String parameterName, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBinaryStream(String parameterName, InputStream x, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setCharacterStream(String parameterName, Reader reader, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNCharacterStream(String parameterName, Reader value) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNCharacterStream(String parameterName, Reader value, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setRef(int parameterIndex, Ref x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(int parameterIndex, Blob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(String parameterName, Blob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(String parameterName, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBlob(String parameterName, InputStream inputStream, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(int parameterIndex, Clob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(String parameterName, Clob x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(String parameterName, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setClob(String parameterName, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(int parameterIndex, NClob value) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(String parameterName, NClob value) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(String parameterName, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNClob(String parameterName, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNString(String parameterName, String value) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setRowId(int parameterIndex, RowId x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setRowId(String parameterName, RowId x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setSQLXML(String parameterName, SQLXML xmlObject) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setDate(String parameterName, Date x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setDate(String parameterName, Date x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTime(String parameterName, Time x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTime(String parameterName, Time x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTimestamp(String parameterName, Timestamp x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setTimestamp(String parameterName, Timestamp x, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setURL(String parameterName, URL val) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNull(String parameterName, int sqlType) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setNull(String parameterName, int sqlType, String typeName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBoolean(String parameterName, boolean x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setByte(String parameterName, byte x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setShort(String parameterName, short x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setInt(String parameterName, int x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setLong(String parameterName, long x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setFloat(String parameterName, float x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setDouble(String parameterName, double x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBigDecimal(String parameterName, BigDecimal x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setString(String parameterName, String x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setBytes(String parameterName, byte[] x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setObject(String parameterName, Object x, int targetSqlType, int scale) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setObject(String parameterName, Object x, int targetSqlType) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void setObject(String parameterName, Object x) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType, int scale) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public void registerOutParameter(String parameterName, int sqlType, String typeName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public String getString(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public boolean getBoolean(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public byte getByte(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public short getShort(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public int getInt(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public long getLong(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public float getFloat(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public double getDouble(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public byte[] getBytes(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Date getDate(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Time getTime(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Timestamp getTimestamp(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Object getObject(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public BigDecimal getBigDecimal(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Object getObject(String parameterName, Map<String, Class<?>> map) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Ref getRef(int parameterIndex) throws SQLException {
        return null;
    }

    @Override
    public Ref getRef(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Blob getBlob(int parameterIndex) throws SQLException {
        return null;
    }

    @Override
    public Blob getBlob(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Clob getClob(int parameterIndex) throws SQLException {
        return null;
    }

    @Override
    public Clob getClob(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public NClob getNClob(int parameterIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public NClob getNClob(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public String getNString(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Reader getNCharacterStream(int parameterIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Reader getNCharacterStream(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Reader getCharacterStream(int parameterIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Reader getCharacterStream(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Array getArray(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Date getDate(String parameterName, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Time getTime(String parameterName, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public Timestamp getTimestamp(String parameterName, Calendar cal) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public URL getURL(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public RowId getRowId(int parameterIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public RowId getRowId(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public SQLXML getSQLXML(int parameterIndex) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }

    @Override
    public SQLXML getSQLXML(String parameterName) throws SQLException {
        throw new SQLFeatureNotSupportedException("Unsupported Operation");
    }
}
