/*
 * Copyright (c) 2018, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * The Universal Permissive License (UPL), Version 1.0
 *
 * Subject to the condition set forth below, permission is hereby granted to any
 * person obtaining a copy of this software, associated documentation and/or
 * data (collectively the "Software"), free of charge and under any and all
 * copyright rights in the Software, and any and all patent rights owned or
 * freely licensable by each licensor hereunder covering either (i) the
 * unmodified Software as contributed to or provided by such licensor, or (ii)
 * the Larger Works (as defined below), to deal in both
 *
 * (a) the Software, and
 *
 * (b) any piece of software and/or hardware listed in the lrgrwrks.txt file if
 * one is included with the Software each a "Larger Work" to which the Software
 * is contributed by such licensors),
 *
 * without restriction, including without limitation the rights to copy, create
 * derivative works of, display, perform, and distribute the Software and make,
 * use, sell, offer for sale, import, export, have made, and have sold the
 * Software and the Larger Work(s), and to sublicense the foregoing rights on
 * either these or other terms.
 *
 * This license is subject to the following condition:
 *
 * The above copyright notice and either this complete permission notice or at a
 * minimum a reference to the UPL must be included in all copies or substantial
 * portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package com.oracle.truffle.js.nodes.access;

import com.oracle.truffle.api.frame.VirtualFrame;
import com.oracle.truffle.api.profiles.BranchProfile;
import com.oracle.truffle.api.profiles.ConditionProfile;
import com.oracle.truffle.js.nodes.JavaScriptNode;
import com.oracle.truffle.js.nodes.unary.IsCallableNode;
import com.oracle.truffle.js.runtime.Errors;
import com.oracle.truffle.js.runtime.JSContext;
import com.oracle.truffle.js.runtime.objects.Null;
import com.oracle.truffle.js.runtime.objects.Undefined;

/**
 * ES6 7.3.9 GetMethod(O, P).
 */
public class GetMethodNode extends JSTargetableNode {
    @Child private JavaScriptNode targetNode;
    @Child private PropertyGetNode cacheNode;
    @Child private IsCallableNode isCallableNode;
    private final ConditionProfile undefinedOrNull = ConditionProfile.createBinaryProfile();
    private final BranchProfile notCallableBranch = BranchProfile.create();

    protected GetMethodNode(JSContext context, JavaScriptNode target, Object propertyKey) {
        this.targetNode = target;
        this.cacheNode = PropertyGetNode.create(propertyKey, false, context);
        this.isCallableNode = IsCallableNode.create();
    }

    public static GetMethodNode create(JSContext ctx, JavaScriptNode target, Object key) {
        return new GetMethodNode(ctx, target, key);
    }

    @Override
    public Object executeWithTarget(VirtualFrame frame, Object target) {
        return executeWithTarget(target);
    }

    public Object executeWithTarget(Object target) {
        Object method = cacheNode.getValue(target);
        if (isCallableNode.executeBoolean(method)) {
            return method;
        } else if (undefinedOrNull.profile(method == Undefined.instance || method == Null.instance)) {
            return Undefined.instance;
        } else {
            notCallableBranch.enter();
            throw Errors.createTypeErrorNotAFunction(method, this);
        }
    }

    @Override
    public Object evaluateTarget(VirtualFrame frame) {
        return targetNode.execute(frame);
    }

    @Override
    public JavaScriptNode getTarget() {
        return targetNode;
    }

    @Override
    public Object execute(VirtualFrame frame) {
        return executeWithTarget(evaluateTarget(frame));
    }

    final JSContext getContext() {
        return cacheNode.getContext();
    }

    @Override
    protected JavaScriptNode copyUninitialized() {
        return new GetMethodNode(getContext(), cloneUninitialized(targetNode), cacheNode.getKey());
    }
}
