/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.rest;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.client.internal.node.NodeClient;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.breaker.CircuitBreaker;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.BytesStream;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.path.PathTrie;
import org.elasticsearch.common.util.Maps;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.RestApiVersion;
import org.elasticsearch.core.Streams;
import org.elasticsearch.http.HttpServerTransport;
import org.elasticsearch.indices.breaker.CircuitBreakerService;
import org.elasticsearch.rest.BaseRestHandler;
import org.elasticsearch.rest.DeprecationRestHandler;
import org.elasticsearch.rest.MethodHandlers;
import org.elasticsearch.rest.RestChannel;
import org.elasticsearch.rest.RestHandler;
import org.elasticsearch.rest.RestHeaderDefinition;
import org.elasticsearch.rest.RestRequest;
import org.elasticsearch.rest.RestResponse;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.rest.RestUtils;
import org.elasticsearch.tracing.Tracer;
import org.elasticsearch.usage.SearchUsageHolder;
import org.elasticsearch.usage.UsageService;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentType;

public class RestController
implements HttpServerTransport.Dispatcher {
    private static final Logger logger = LogManager.getLogger(RestController.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(RestController.class);
    static final Set<String> SAFELISTED_MEDIA_TYPES = Set.of("application/x-www-form-urlencoded", "multipart/form-data", "text/plain");
    static final String ELASTIC_PRODUCT_HTTP_HEADER = "X-elastic-product";
    static final String ELASTIC_PRODUCT_HTTP_HEADER_VALUE = "Elasticsearch";
    static final Set<String> RESERVED_PATHS = Set.of("/__elb_health__", "/__elb_health__/zk", "/_health", "/_health/zk");
    private static final BytesReference FAVICON_RESPONSE;
    private final PathTrie<MethodHandlers> handlers = new PathTrie(RestUtils.REST_DECODER);
    private final UnaryOperator<RestHandler> handlerWrapper;
    private final NodeClient client;
    private final CircuitBreakerService circuitBreakerService;
    private final Set<RestHeaderDefinition> headersToCopy;
    private final UsageService usageService;
    private final Tracer tracer;

    public RestController(Set<RestHeaderDefinition> headersToCopy, UnaryOperator<RestHandler> handlerWrapper, NodeClient client, CircuitBreakerService circuitBreakerService, UsageService usageService, Tracer tracer) {
        this.headersToCopy = headersToCopy;
        this.usageService = usageService;
        this.tracer = tracer;
        if (handlerWrapper == null) {
            handlerWrapper = h -> h;
        }
        this.handlerWrapper = handlerWrapper;
        this.client = client;
        this.circuitBreakerService = circuitBreakerService;
        this.registerHandlerNoWrap(RestRequest.Method.GET, "/favicon.ico", RestApiVersion.current(), (request, channel, clnt) -> channel.sendResponse(new RestResponse(RestStatus.OK, "image/x-icon", FAVICON_RESPONSE)));
    }

    protected void registerAsDeprecatedHandler(RestRequest.Method method, String path, RestApiVersion version, RestHandler handler, String deprecationMessage) {
        this.registerAsDeprecatedHandler(method, path, version, handler, deprecationMessage, null);
    }

    protected void registerAsDeprecatedHandler(RestRequest.Method method, String path, RestApiVersion version, RestHandler handler, String deprecationMessage, @Nullable Level deprecationLevel) {
        assert (!(handler instanceof DeprecationRestHandler));
        if (version == RestApiVersion.current()) {
            this.registerHandler(method, path, version, new DeprecationRestHandler(handler, method, path, deprecationLevel, deprecationMessage, deprecationLogger, false));
        } else if (version == RestApiVersion.minimumSupported()) {
            this.registerHandler(method, path, version, new DeprecationRestHandler(handler, method, path, deprecationLevel, deprecationMessage, deprecationLogger, true));
        } else {
            logger.debug("Deprecated route [" + method + " " + path + "] for handler [" + handler.getClass() + "] with version [" + version + "], which is less than the minimum supported version [" + RestApiVersion.minimumSupported() + "]");
        }
    }

    protected void registerAsReplacedHandler(RestRequest.Method method, String path, RestApiVersion version, RestHandler handler, RestRequest.Method replacedMethod, String replacedPath, RestApiVersion replacedVersion) {
        String replacedMessage = "[" + replacedMethod.name() + " " + replacedPath + "] is deprecated! Use [" + method.name() + " " + path + "] instead.";
        this.registerHandler(method, path, version, handler);
        this.registerAsDeprecatedHandler(replacedMethod, replacedPath, replacedVersion, handler, replacedMessage);
    }

    protected void registerHandler(RestRequest.Method method, String path, RestApiVersion version, RestHandler handler) {
        if (handler instanceof BaseRestHandler) {
            this.usageService.addRestHandler((BaseRestHandler)handler);
        }
        this.registerHandlerNoWrap(method, path, version, (RestHandler)this.handlerWrapper.apply(handler));
    }

    private void registerHandlerNoWrap(RestRequest.Method method, String path, RestApiVersion version, RestHandler handler) {
        assert (RestApiVersion.minimumSupported() == version || RestApiVersion.current() == version) : "REST API compatibility is only supported for version " + RestApiVersion.minimumSupported().major;
        if (RESERVED_PATHS.contains(path)) {
            throw new IllegalArgumentException("path [" + path + "] is a reserved path and may not be registered");
        }
        this.handlers.insertOrUpdate(path, new MethodHandlers(path).addMethod(method, version, handler), (handlers, ignoredHandler) -> handlers.addMethod(method, version, handler));
    }

    public void registerHandler(RestHandler.Route route, RestHandler handler) {
        if (route.isReplacement()) {
            RestHandler.Route replaced = route.getReplacedRoute();
            this.registerAsReplacedHandler(route.getMethod(), route.getPath(), route.getRestApiVersion(), handler, replaced.getMethod(), replaced.getPath(), replaced.getRestApiVersion());
        } else if (route.isDeprecated()) {
            this.registerAsDeprecatedHandler(route.getMethod(), route.getPath(), route.getRestApiVersion(), handler, route.getDeprecationMessage(), route.getDeprecationLevel());
        } else {
            this.registerHandler(route.getMethod(), route.getPath(), route.getRestApiVersion(), handler);
        }
    }

    public void registerHandler(RestHandler handler) {
        handler.routes().forEach(route -> this.registerHandler((RestHandler.Route)route, handler));
    }

    @Override
    public void dispatchRequest(RestRequest request, RestChannel channel, ThreadContext threadContext) {
        threadContext.addResponseHeader(ELASTIC_PRODUCT_HTTP_HEADER, ELASTIC_PRODUCT_HTTP_HEADER_VALUE);
        try {
            this.tryAllHandlers(request, channel, threadContext);
        }
        catch (Exception e) {
            try {
                channel.sendResponse(new RestResponse(channel, e));
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                logger.error(() -> "failed to send failure response for uri [" + request.uri() + "]", (Throwable)inner);
            }
        }
    }

    @Override
    public void dispatchBadRequest(RestChannel channel, ThreadContext threadContext, Throwable cause) {
        threadContext.addResponseHeader(ELASTIC_PRODUCT_HTTP_HEADER, ELASTIC_PRODUCT_HTTP_HEADER_VALUE);
        try {
            Exception e = cause == null ? new ElasticsearchException("unknown cause", new Object[0]) : (cause instanceof Exception ? (Exception)cause : new ElasticsearchException(cause));
            channel.sendResponse(new RestResponse(channel, RestStatus.BAD_REQUEST, e));
        }
        catch (IOException e) {
            if (cause != null) {
                e.addSuppressed(cause);
            }
            logger.warn("failed to send bad request response", (Throwable)e);
            channel.sendResponse(new RestResponse(RestStatus.INTERNAL_SERVER_ERROR, "text/plain; charset=UTF-8", BytesArray.EMPTY));
        }
    }

    private void dispatchRequest(RestRequest request, RestChannel channel, RestHandler handler, ThreadContext threadContext) throws Exception {
        int contentLength = request.contentLength();
        if (contentLength > 0) {
            if (RestController.isContentTypeDisallowed(request) || !handler.mediaTypesValid(request)) {
                RestController.sendContentTypeErrorMessage(request.getAllHeaderValues("Content-Type"), channel);
                return;
            }
            XContentType xContentType = request.getXContentType();
            if (handler.supportsContentStream() && XContentType.JSON != xContentType.canonical() && XContentType.SMILE != xContentType.canonical()) {
                channel.sendResponse(RestResponse.createSimpleErrorResponse(channel, RestStatus.NOT_ACCEPTABLE, "Content-Type [" + xContentType + "] does not support stream parsing. Use JSON or SMILE instead"));
                return;
            }
        }
        RestChannel responseChannel = channel;
        try {
            if (handler.canTripCircuitBreaker()) {
                RestController.inFlightRequestsBreaker(this.circuitBreakerService).addEstimateBytesAndMaybeBreak(contentLength, "<http_request>");
            } else {
                RestController.inFlightRequestsBreaker(this.circuitBreakerService).addWithoutBreaking(contentLength);
            }
            responseChannel = new ResourceHandlingHttpChannel(channel, this.circuitBreakerService, contentLength);
            if (!handler.allowsUnsafeBuffers()) {
                request.ensureSafeBuffers();
            }
            if (!handler.allowSystemIndexAccessByDefault()) {
                String prodOriginValue = request.header("X-elastic-product-origin");
                if (prodOriginValue != null) {
                    threadContext.putHeader("_system_index_access_allowed", Boolean.TRUE.toString());
                    threadContext.putHeader("_external_system_index_access_origin", prodOriginValue);
                } else {
                    threadContext.putHeader("_system_index_access_allowed", Boolean.FALSE.toString());
                }
            } else {
                threadContext.putHeader("_system_index_access_allowed", Boolean.TRUE.toString());
            }
            handler.handleRequest(request, responseChannel, this.client);
        }
        catch (Exception e) {
            responseChannel.sendResponse(new RestResponse(responseChannel, e));
        }
    }

    private static boolean isContentTypeDisallowed(RestRequest request) {
        return request.getParsedContentType() != null && SAFELISTED_MEDIA_TYPES.contains(request.getParsedContentType().mediaTypeWithoutParameters());
    }

    private boolean handleNoHandlerFound(ThreadContext threadContext, String rawPath, RestRequest.Method method, String uri, RestChannel channel) {
        Set<RestRequest.Method> validMethodSet = this.getValidHandlerMethodSet(rawPath);
        if (!validMethodSet.contains((Object)method)) {
            if (method == RestRequest.Method.OPTIONS) {
                this.startTrace(threadContext, channel);
                RestController.handleOptionsRequest(channel, validMethodSet);
                return true;
            }
            if (!validMethodSet.isEmpty()) {
                this.startTrace(threadContext, channel);
                RestController.handleUnsupportedHttpMethod(uri, method, channel, validMethodSet, null);
                return true;
            }
        }
        return false;
    }

    private void startTrace(ThreadContext threadContext, RestChannel channel) {
        this.startTrace(threadContext, channel, null);
    }

    private void startTrace(ThreadContext threadContext, RestChannel channel, String restPath) {
        RestRequest req = channel.request();
        if (restPath == null) {
            restPath = req.path();
        }
        String method = null;
        try {
            method = req.method().name();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        Object name = method != null ? method + " " + restPath : restPath;
        Map<String, Object> attributes = Maps.newMapWithExpectedSize(req.getHeaders().size() + 3);
        req.getHeaders().forEach((key, values) -> {
            String lowerKey = key.toLowerCase(Locale.ROOT).replace('-', '_');
            attributes.put("http.request.headers." + lowerKey, values.size() == 1 ? values.get(0) : String.join((CharSequence)"; ", values));
        });
        attributes.put("http.method", method);
        attributes.put("http.url", req.uri());
        switch (req.getHttpRequest().protocolVersion()) {
            case HTTP_1_0: {
                attributes.put("http.flavour", "1.0");
                break;
            }
            case HTTP_1_1: {
                attributes.put("http.flavour", "1.1");
            }
        }
        this.tracer.startTrace(threadContext, "rest-" + channel.request().getRequestId(), (String)name, attributes);
    }

    private void traceException(RestChannel channel, Throwable e) {
        this.tracer.addError("rest-" + channel.request().getRequestId(), e);
    }

    private static void sendContentTypeErrorMessage(@Nullable List<String> contentTypeHeader, RestChannel channel) throws IOException {
        Object errorMessage = contentTypeHeader == null ? "Content-Type header is missing" : "Content-Type header [" + Strings.collectionToCommaDelimitedString(contentTypeHeader) + "] is not supported";
        channel.sendResponse(RestResponse.createSimpleErrorResponse(channel, RestStatus.NOT_ACCEPTABLE, (String)errorMessage));
    }

    private void tryAllHandlers(RestRequest request, RestChannel channel, ThreadContext threadContext) throws Exception {
        RestRequest.Method requestMethod;
        try {
            this.copyRestHeaders(request, threadContext);
            RestController.validateErrorTrace(request, channel);
        }
        catch (IllegalArgumentException e) {
            this.startTrace(threadContext, channel);
            channel.sendResponse(RestResponse.createSimpleErrorResponse(channel, RestStatus.BAD_REQUEST, e.getMessage()));
            return;
        }
        String rawPath = request.rawPath();
        String uri = request.uri();
        RestApiVersion restApiVersion = request.getRestApiVersion();
        try {
            requestMethod = request.method();
            Iterator<MethodHandlers> allHandlers = this.getAllHandlers(request.params(), rawPath);
            while (allHandlers.hasNext()) {
                MethodHandlers handlers = allHandlers.next();
                RestHandler handler = handlers == null ? null : handlers.getHandler(requestMethod, restApiVersion);
                if (handler == null) {
                    if (!this.handleNoHandlerFound(threadContext, rawPath, requestMethod, uri, channel)) continue;
                    return;
                }
                this.startTrace(threadContext, channel, handlers.getPath());
                this.dispatchRequest(request, channel, handler, threadContext);
                return;
            }
        }
        catch (IllegalArgumentException e) {
            this.startTrace(threadContext, channel);
            this.traceException(channel, e);
            RestController.handleUnsupportedHttpMethod(uri, null, channel, this.getValidHandlerMethodSet(rawPath), e);
            return;
        }
        this.startTrace(threadContext, channel);
        RestController.handleBadRequest(uri, requestMethod, channel);
    }

    private static void validateErrorTrace(RestRequest request, RestChannel channel) {
        if (request.paramAsBoolean("error_trace", false) && !channel.detailedErrorsEnabled()) {
            throw new IllegalArgumentException("error traces in responses are disabled.");
        }
    }

    private void copyRestHeaders(RestRequest request, ThreadContext threadContext) {
        for (RestHeaderDefinition restHeader : this.headersToCopy) {
            String name = restHeader.getName();
            List<String> headerValues = request.getAllHeaderValues(name);
            if (headerValues == null || headerValues.isEmpty()) continue;
            List distinctHeaderValues = headerValues.stream().distinct().toList();
            if (!restHeader.isMultiValueAllowed() && distinctHeaderValues.size() > 1) {
                throw new IllegalArgumentException("multiple values for single-valued header [" + name + "].");
            }
            if (name.equals("traceparent")) {
                String traceparent = (String)distinctHeaderValues.get(0);
                Optional<String> traceId = RestUtils.extractTraceId(traceparent);
                if (!traceId.isPresent()) continue;
                threadContext.putHeader("trace.id", traceId.get());
                threadContext.putTransient("parent_traceparent", traceparent);
                continue;
            }
            if (name.equals("tracestate")) {
                threadContext.putTransient("parent_tracestate", distinctHeaderValues.get(0));
                continue;
            }
            threadContext.putHeader(name, String.join((CharSequence)",", distinctHeaderValues));
        }
    }

    Iterator<MethodHandlers> getAllHandlers(@Nullable Map<String, String> requestParamsRef, String rawPath) {
        Supplier<Map<String, String>> paramsSupplier;
        if (requestParamsRef == null) {
            paramsSupplier = () -> null;
        } else {
            Map<String, String> originalParams = Map.copyOf(requestParamsRef);
            paramsSupplier = () -> {
                requestParamsRef.clear();
                requestParamsRef.putAll(originalParams);
                return requestParamsRef;
            };
        }
        return this.handlers.retrieveAll(rawPath, paramsSupplier);
    }

    public SearchUsageHolder getSearchUsageHolder() {
        return this.usageService.getSearchUsageHolder();
    }

    private static void handleUnsupportedHttpMethod(String uri, @Nullable RestRequest.Method method, RestChannel channel, Set<RestRequest.Method> validMethodSet, @Nullable IllegalArgumentException exception) {
        try {
            StringBuilder msg = new StringBuilder();
            if (exception == null) {
                msg.append("Incorrect HTTP method for uri [").append(uri);
                msg.append("] and method [").append((Object)method).append("]");
            } else {
                msg.append(exception.getMessage());
            }
            if (!validMethodSet.isEmpty()) {
                msg.append(", allowed: ").append(validMethodSet);
            }
            RestResponse restResponse = RestResponse.createSimpleErrorResponse(channel, RestStatus.METHOD_NOT_ALLOWED, msg.toString());
            if (!validMethodSet.isEmpty()) {
                restResponse.addHeader("Allow", Strings.collectionToDelimitedString(validMethodSet, ","));
            }
            channel.sendResponse(restResponse);
        }
        catch (IOException e) {
            logger.warn("failed to send bad request response", (Throwable)e);
            channel.sendResponse(new RestResponse(RestStatus.INTERNAL_SERVER_ERROR, "text/plain; charset=UTF-8", BytesArray.EMPTY));
        }
    }

    private static void handleOptionsRequest(RestChannel channel, Set<RestRequest.Method> validMethodSet) {
        RestResponse restResponse = new RestResponse(RestStatus.OK, "text/plain; charset=UTF-8", BytesArray.EMPTY);
        if (!validMethodSet.isEmpty()) {
            restResponse.addHeader("Allow", Strings.collectionToDelimitedString(validMethodSet, ","));
        }
        channel.sendResponse(restResponse);
    }

    public static void handleBadRequest(String uri, RestRequest.Method method, RestChannel channel) throws IOException {
        try (XContentBuilder builder = channel.newErrorBuilder();){
            builder.startObject();
            builder.field("error", "no handler found for uri [" + uri + "] and method [" + method + "]");
            builder.endObject();
            channel.sendResponse(new RestResponse(RestStatus.BAD_REQUEST, builder));
        }
    }

    private Set<RestRequest.Method> getValidHandlerMethodSet(String rawPath) {
        HashSet<RestRequest.Method> validMethods = new HashSet<RestRequest.Method>();
        Iterator<MethodHandlers> allHandlers = this.getAllHandlers(null, rawPath);
        while (allHandlers.hasNext()) {
            MethodHandlers methodHandlers = allHandlers.next();
            if (methodHandlers == null) continue;
            validMethods.addAll(methodHandlers.getValidMethods());
        }
        return validMethods;
    }

    private static CircuitBreaker inFlightRequestsBreaker(CircuitBreakerService circuitBreakerService) {
        return circuitBreakerService.getBreaker("inflight_requests");
    }

    static {
        try (InputStream stream = RestController.class.getResourceAsStream("/config/favicon.ico");){
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            Streams.copy((InputStream)stream, (OutputStream)out);
            FAVICON_RESPONSE = new BytesArray(out.toByteArray());
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
    }

    private static final class ResourceHandlingHttpChannel
    implements RestChannel {
        private final RestChannel delegate;
        private final CircuitBreakerService circuitBreakerService;
        private final int contentLength;
        private final AtomicBoolean closed = new AtomicBoolean();

        ResourceHandlingHttpChannel(RestChannel delegate, CircuitBreakerService circuitBreakerService, int contentLength) {
            this.delegate = delegate;
            this.circuitBreakerService = circuitBreakerService;
            this.contentLength = contentLength;
        }

        @Override
        public XContentBuilder newBuilder() throws IOException {
            return this.delegate.newBuilder();
        }

        @Override
        public XContentBuilder newErrorBuilder() throws IOException {
            return this.delegate.newErrorBuilder();
        }

        @Override
        public XContentBuilder newBuilder(@Nullable XContentType xContentType, boolean useFiltering) throws IOException {
            return this.delegate.newBuilder(xContentType, useFiltering);
        }

        @Override
        public XContentBuilder newBuilder(XContentType xContentType, XContentType responseContentType, boolean useFiltering) throws IOException {
            return this.delegate.newBuilder(xContentType, responseContentType, useFiltering);
        }

        @Override
        public XContentBuilder newBuilder(XContentType xContentType, XContentType responseContentType, boolean useFiltering, OutputStream out) throws IOException {
            return this.delegate.newBuilder(xContentType, responseContentType, useFiltering, out);
        }

        @Override
        public BytesStream bytesOutput() {
            return this.delegate.bytesOutput();
        }

        @Override
        public void releaseOutputBuffer() {
            this.delegate.releaseOutputBuffer();
        }

        @Override
        public RestRequest request() {
            return this.delegate.request();
        }

        @Override
        public boolean detailedErrorsEnabled() {
            return this.delegate.detailedErrorsEnabled();
        }

        @Override
        public void sendResponse(RestResponse response) {
            boolean success = false;
            try {
                this.close();
                this.delegate.sendResponse(response);
                success = true;
            }
            finally {
                if (!success) {
                    this.releaseOutputBuffer();
                }
            }
        }

        private void close() {
            if (!this.closed.compareAndSet(false, true)) {
                throw new IllegalStateException("Channel is already closed");
            }
            RestController.inFlightRequestsBreaker(this.circuitBreakerService).addWithoutBreaking(-this.contentLength);
        }
    }
}

