/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.geo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.elasticsearch.common.geo.GeoUtils;
import org.elasticsearch.core.Tuple;
import org.elasticsearch.geometry.LinearRing;
import org.elasticsearch.geometry.MultiPolygon;
import org.elasticsearch.geometry.Point;
import org.elasticsearch.geometry.Polygon;
import org.locationtech.spatial4j.exception.InvalidShapeException;

public class GeoPolygonDecomposer {
    private static final double DATELINE = 180.0;
    private static final Comparator<Edge> INTERSECTION_ORDER = Comparator.comparingDouble(o -> o.intersect.getY());

    private GeoPolygonDecomposer() {
    }

    public static void decomposeMultiPolygon(MultiPolygon multiPolygon, boolean orientation, List<Polygon> collector) {
        for (Polygon polygon : multiPolygon) {
            GeoPolygonDecomposer.decomposePolygon(polygon, orientation, collector);
        }
    }

    public static void decomposePolygon(Polygon polygon, boolean orientation, List<Polygon> collector) {
        if (polygon.isEmpty()) {
            return;
        }
        LinearRing shell = GeoPolygonDecomposer.filterRing(polygon.getPolygon());
        LinearRing[] holes = new LinearRing[polygon.getNumberOfHoles()];
        int numEdges = shell.length() - 1;
        for (int i = 0; i < polygon.getNumberOfHoles(); ++i) {
            holes[i] = GeoPolygonDecomposer.filterRing(polygon.getHole(i));
            numEdges += holes[i].length() - 1;
            GeoPolygonDecomposer.validateHole(shell, holes[i]);
        }
        Edge[] edges = new Edge[numEdges];
        Edge[] holeComponents = new Edge[holes.length];
        AtomicBoolean translated = new AtomicBoolean(false);
        int offset = GeoPolygonDecomposer.createEdges(0, orientation, shell, null, edges, 0, translated);
        for (int i = 0; i < polygon.getNumberOfHoles(); ++i) {
            int length = GeoPolygonDecomposer.createEdges(i + 1, orientation, shell, holes[i], edges, offset, translated);
            holeComponents[i] = edges[offset];
            offset += length;
        }
        int numHoles = holeComponents.length;
        numHoles = GeoPolygonDecomposer.merge(edges, 0, GeoPolygonDecomposer.intersections(180.0, edges), holeComponents, numHoles);
        numHoles = GeoPolygonDecomposer.merge(edges, 0, GeoPolygonDecomposer.intersections(-180.0, edges), holeComponents, numHoles);
        GeoPolygonDecomposer.compose(edges, holeComponents, numHoles, collector);
    }

    private static LinearRing filterRing(LinearRing linearRing) {
        int numPoints = linearRing.length();
        int count = 2;
        for (int i = 1; i < numPoints - 1; ++i) {
            if (linearRing.getLon(i - 1) == linearRing.getLon(i) && (linearRing.getLat(i - 1) == linearRing.getLat(i) || linearRing.getLon(i - 1) == linearRing.getLon(i + 1) && linearRing.getLat(i - 1) > linearRing.getLat(i) != linearRing.getLat(i + 1) > linearRing.getLat(i))) continue;
            ++count;
        }
        if (numPoints == count) {
            return linearRing;
        }
        double[] lons = new double[count];
        double[] lats = new double[count];
        double d = linearRing.getLat(0);
        lats[count - 1] = d;
        lats[0] = d;
        double d2 = linearRing.getLon(0);
        lons[count - 1] = d2;
        lons[0] = d2;
        count = 0;
        for (int i = 1; i < numPoints - 1; ++i) {
            if (linearRing.getLon(i - 1) == linearRing.getLon(i) && (linearRing.getLat(i - 1) == linearRing.getLat(i) || linearRing.getLon(i - 1) == linearRing.getLon(i + 1))) continue;
            lats[++count] = linearRing.getLat(i);
            lons[count] = linearRing.getLon(i);
        }
        return new LinearRing(lons, lats);
    }

    private static void validateHole(LinearRing shell, LinearRing hole) {
        int i;
        HashSet<Point> exterior = new HashSet<Point>();
        HashSet<Point> interior = new HashSet<Point>();
        for (i = 0; i < shell.length(); ++i) {
            exterior.add(new Point(shell.getX(i), shell.getY(i)));
        }
        for (i = 0; i < hole.length(); ++i) {
            interior.add(new Point(hole.getX(i), hole.getY(i)));
        }
        exterior.retainAll(interior);
        if (exterior.size() >= 2) {
            throw new IllegalArgumentException("Invalid polygon, interior cannot share more than one point with the exterior");
        }
    }

    private static Point position(Point p1, Point p2, double position) {
        if (position == 0.0) {
            return p1;
        }
        if (position == 1.0) {
            return p2;
        }
        double x = p1.getX() + position * (p2.getX() - p1.getX());
        double y = p1.getY() + position * (p2.getY() - p1.getY());
        return new Point(x, y);
    }

    private static int createEdges(int component, boolean orientation, LinearRing shell, LinearRing hole, Edge[] edges, int offset, AtomicBoolean translated) {
        boolean direction = component == 0 ^ orientation;
        Point[] points = hole != null ? GeoPolygonDecomposer.points(hole) : GeoPolygonDecomposer.points(shell);
        GeoPolygonDecomposer.ring(component, direction, !orientation, points, 0, edges, offset, points.length - 1, translated);
        return points.length - 1;
    }

    private static Point[] points(LinearRing linearRing) {
        Point[] points = new Point[linearRing.length()];
        for (int i = 0; i < linearRing.length(); ++i) {
            points[i] = new Point(linearRing.getX(i), linearRing.getY(i));
        }
        return points;
    }

    private static Edge[] ring(int component, boolean direction, boolean handedness, Point[] points, int offset, Edge[] edges, int toffset, int length, AtomicBoolean translated) {
        boolean incorrectOrientation;
        double signedArea = 0.0;
        double minX = Double.POSITIVE_INFINITY;
        double maxX = Double.NEGATIVE_INFINITY;
        for (int i = offset; i < offset + length; ++i) {
            signedArea += points[i].getX() * points[i + 1].getY() - points[i].getY() * points[i + 1].getX();
            minX = Math.min(minX, points[i].getX());
            maxX = Math.max(maxX, points[i].getX());
        }
        if (signedArea == 0.0) {
            throw new InvalidShapeException("Cannot determine orientation: signed area equal to 0. Points are collinear or polygon self-intersects.");
        }
        boolean orientation = signedArea < 0.0;
        double rng = maxX - minX;
        boolean bl = incorrectOrientation = component == 0 && handedness != orientation;
        if (incorrectOrientation && rng > 180.0 && rng != 360.0 || translated.get() && component != 0) {
            GeoPolygonDecomposer.translate(points);
            if (component == 0) {
                translated.set(true);
            }
            if (component == 0 || component != 0 && handedness == orientation) {
                orientation = !orientation;
            }
        }
        return GeoPolygonDecomposer.concat(component, direction ^ orientation, points, offset, edges, toffset, length);
    }

    private static void translate(Point[] points) {
        for (int i = 0; i < points.length; ++i) {
            if (!(points[i].getX() < 0.0)) continue;
            points[i] = new Point(points[i].getX() + 360.0, points[i].getY());
        }
    }

    private static int merge(Edge[] intersections, int offset, int length, Edge[] holes, int numHoles) {
        for (int i = 0; i < length; i += 2) {
            Edge e1 = intersections[offset + i + 0];
            Edge e2 = intersections[offset + i + 1];
            if (e2.component > 0) {
                holes[e2.component - 1] = holes[--numHoles];
                holes[numHoles] = null;
            }
            if (e1.intersect == Edge.MAX_COORDINATE || e2.intersect == Edge.MAX_COORDINATE || e1.next.next.coordinate.equals((Object)e2.coordinate) && Math.abs(e1.next.coordinate.getX()) == 180.0 && Math.abs(e2.coordinate.getX()) == 180.0) continue;
            GeoPolygonDecomposer.connect(e1, e2);
        }
        return numHoles;
    }

    private static void connect(Edge in, Edge out) {
        assert (in != null && out != null);
        assert (in != out);
        if (in.intersect != in.next.coordinate) {
            Edge e1 = new Edge(in.intersect, in.next);
            if (out.intersect != out.next.coordinate) {
                Edge e2 = new Edge(out.intersect, out.next);
                in.next = new Edge(in.intersect, e2, in.intersect);
            } else {
                in.next = new Edge(in.intersect, out.next, in.intersect);
            }
            out.next = new Edge(out.intersect, e1, out.intersect);
        } else if (in.next != out && in.coordinate != out.intersect) {
            Edge e2 = new Edge(out.intersect, in.next, out.intersect);
            if (out.intersect != out.next.coordinate) {
                Edge e1 = new Edge(out.intersect, out.next);
                in.next = new Edge(in.intersect, e1, in.intersect);
            } else {
                in.next = new Edge(in.intersect, out.next, in.intersect);
            }
            out.next = e2;
        }
    }

    private static Edge[] concat(int component, boolean direction, Point[] points, int pointOffset, Edge[] edges, int edgeOffset, int length) {
        assert (edges.length >= length + edgeOffset);
        assert (points.length >= length + pointOffset);
        edges[edgeOffset] = new Edge(new Point(points[pointOffset].getX(), points[pointOffset].getY()), null);
        for (int i = 1; i < length; ++i) {
            Point nextPoint = new Point(points[pointOffset + i].getX(), points[pointOffset + i].getY());
            if (direction) {
                edges[edgeOffset + i] = new Edge(nextPoint, edges[edgeOffset + i - 1]);
                edges[edgeOffset + i].component = component;
                continue;
            }
            if (!edges[edgeOffset + i - 1].coordinate.equals((Object)nextPoint)) {
                Edge edge = new Edge(nextPoint, null);
                edges[edgeOffset + i] = edge;
                edges[edgeOffset + i - 1].next = edge;
                edges[edgeOffset + i - 1].component = component;
                continue;
            }
            throw new InvalidShapeException("Provided shape has duplicate consecutive coordinates at: (" + nextPoint + ")");
        }
        if (direction) {
            edges[edgeOffset].setNext(edges[edgeOffset + length - 1]);
            edges[edgeOffset].component = component;
        } else {
            edges[edgeOffset + length - 1].setNext(edges[edgeOffset]);
            edges[edgeOffset + length - 1].component = component;
        }
        return edges;
    }

    private static int intersections(double dateline, Edge[] edges) {
        int i;
        int numIntersections = 0;
        assert (!Double.isNaN(dateline));
        int maxComponent = 0;
        for (i = 0; i < edges.length; ++i) {
            Point p1 = edges[i].coordinate;
            Point p2 = edges[i].next.coordinate;
            assert (!Double.isNaN(p2.getX()) && !Double.isNaN(p1.getX()));
            edges[i].intersect = Edge.MAX_COORDINATE;
            double position = GeoPolygonDecomposer.intersection(p1.getX(), p2.getX(), dateline);
            if (Double.isNaN(position)) continue;
            edges[i].intersection(position);
            ++numIntersections;
            maxComponent = Math.max(maxComponent, edges[i].component);
        }
        if (maxComponent > 0) {
            for (i = 0; i < maxComponent; ++i) {
                if (!GeoPolygonDecomposer.clearComponentTouchingDateline(edges, i + 1)) continue;
                --numIntersections;
            }
        }
        Arrays.sort(edges, INTERSECTION_ORDER);
        return numIntersections;
    }

    private static boolean clearComponentTouchingDateline(Edge[] edges, int component) {
        Edge intersection = null;
        for (int j = 0; j < edges.length; ++j) {
            if (edges[j].intersect == Edge.MAX_COORDINATE || edges[j].component != component) continue;
            if (intersection == null) {
                intersection = edges[j];
                continue;
            }
            return false;
        }
        if (intersection != null) {
            intersection.intersect = Edge.MAX_COORDINATE;
        }
        return intersection != null;
    }

    private static Edge[] edges(Edge[] edges, int numHoles, List<List<Point[]>> components) {
        ArrayList<Edge> mainEdges = new ArrayList<Edge>(edges.length);
        for (int i = 0; i < edges.length; ++i) {
            if (edges[i].component < 0) continue;
            double[] partitionPoint = new double[3];
            int length = GeoPolygonDecomposer.component(edges[i], -(components.size() + numHoles + 1), mainEdges, partitionPoint);
            ArrayList<Point[]> component = new ArrayList<Point[]>();
            component.add(GeoPolygonDecomposer.coordinates(edges[i], new Point[length + 1], partitionPoint));
            components.add(component);
        }
        return mainEdges.toArray(new Edge[mainEdges.size()]);
    }

    private static void compose(Edge[] edges, Edge[] holes, int numHoles, List<Polygon> collector) {
        ArrayList<List<Point[]>> components = new ArrayList<List<Point[]>>();
        GeoPolygonDecomposer.assign(holes, GeoPolygonDecomposer.holes(holes, numHoles), numHoles, GeoPolygonDecomposer.edges(edges, numHoles, components), components);
        GeoPolygonDecomposer.buildPoints(components, collector);
    }

    private static void assign(Edge[] holes, Point[][] points, int numHoles, Edge[] edges, List<List<Point[]>> components) {
        for (int i = 0; i < numHoles; ++i) {
            Edge current = new Edge(holes[i].coordinate, holes[i].next);
            current.intersect = current.coordinate;
            int intersections = GeoPolygonDecomposer.intersections(current.coordinate.getX(), edges);
            if (intersections == 0) {
                throw new InvalidShapeException("Invalid shape: Hole is not within polygon");
            }
            boolean sharedVertex = false;
            int pos = Arrays.binarySearch(edges, 0, intersections, current, INTERSECTION_ORDER);
            if (pos >= 0 && !(sharedVertex = edges[pos].intersect.equals((Object)current.coordinate))) {
                throw new InvalidShapeException("Invalid shape: Hole is not within polygon");
            }
            int index = sharedVertex ? 0 : (pos == -1 ? 0 : -(pos + 2));
            int component = -edges[index].component - numHoles - 1;
            components.get(component).add(points[i]);
        }
    }

    private static int component(Edge edge, int id, ArrayList<Edge> edges, double[] partitionPoint) {
        Edge any = edge;
        while ((any.coordinate.getX() == 180.0 || any.coordinate.getX() == -180.0) && (any = any.next) != edge) {
        }
        double shiftOffset = any.coordinate.getX() > 180.0 ? 180.0 : (any.coordinate.getX() < -180.0 ? -180.0 : 0.0);
        int length = 0;
        int connectedComponents = 0;
        int splitIndex = 1;
        Edge current = edge;
        Edge prev = edge;
        HashMap<Point, Tuple> visitedEdge = new HashMap<Point, Tuple>();
        do {
            current.coordinate = GeoPolygonDecomposer.shift(current.coordinate, shiftOffset);
            current.component = id;
            if (edges != null) {
                if (visitedEdge.containsKey(current.coordinate)) {
                    partitionPoint[0] = current.coordinate.getX();
                    partitionPoint[1] = current.coordinate.getY();
                    partitionPoint[2] = current.coordinate.getZ();
                    if (connectedComponents > 0 && current.next != edge) {
                        throw new InvalidShapeException("Shape contains more than one shared point");
                    }
                    int visitID = -id;
                    Edge firstAppearance = (Edge)((Tuple)visitedEdge.get(current.coordinate)).v2();
                    Edge temp = firstAppearance.next;
                    firstAppearance.next = current.next;
                    current.next = temp;
                    current.component = visitID;
                    do {
                        prev.component = visitID;
                        prev = (Edge)((Tuple)visitedEdge.get(prev.coordinate)).v1();
                        ++splitIndex;
                    } while (!current.coordinate.equals((Object)prev.coordinate));
                    ++connectedComponents;
                } else {
                    visitedEdge.put(current.coordinate, new Tuple((Object)prev, (Object)current));
                }
                edges.add(current);
                prev = current;
            }
            ++length;
        } while (connectedComponents == 0 && (current = current.next) != edge);
        return splitIndex != 1 ? length - splitIndex : length;
    }

    private static Point[] coordinates(Edge component, Point[] coordinates, double[] partitionPoint) {
        for (int i = 0; i < coordinates.length; ++i) {
            component = component.next;
            coordinates[i] = component.coordinate;
        }
        if (!coordinates[0].equals((Object)coordinates[coordinates.length - 1])) {
            if (Double.isNaN(partitionPoint[2])) {
                throw new InvalidShapeException("Self-intersection at or near point [" + partitionPoint[0] + "," + partitionPoint[1] + "]");
            }
            throw new InvalidShapeException("Self-intersection at or near point [" + partitionPoint[0] + "," + partitionPoint[1] + "," + partitionPoint[2] + "]");
        }
        return coordinates;
    }

    private static void buildPoints(List<List<Point[]>> components, List<Polygon> collector) {
        for (List<Point[]> component : components) {
            collector.add(GeoPolygonDecomposer.buildPolygon(component));
        }
    }

    private static Polygon buildPolygon(List<Point[]> polygon) {
        List holes;
        Point[] shell = polygon.get(0);
        if (polygon.size() > 1) {
            holes = new ArrayList(polygon.size() - 1);
            for (int i = 1; i < polygon.size(); ++i) {
                Point[] coords = polygon.get(i);
                double[] x = new double[coords.length];
                double[] y = new double[coords.length];
                for (int c = 0; c < coords.length; ++c) {
                    x[c] = GeoUtils.normalizeLon(coords[c].getX());
                    y[c] = GeoUtils.normalizeLat(coords[c].getY());
                }
                holes.add(new LinearRing(x, y));
            }
        } else {
            holes = Collections.emptyList();
        }
        double[] x = new double[shell.length];
        double[] y = new double[shell.length];
        for (int i = 0; i < shell.length; ++i) {
            x[i] = GeoPolygonDecomposer.normalizeLonMinus180Inclusive(shell[i].getX());
            y[i] = GeoUtils.normalizeLat(shell[i].getY());
        }
        return new Polygon(new LinearRing(x, y), holes);
    }

    private static Point[][] holes(Edge[] holes, int numHoles) {
        if (numHoles == 0) {
            return new Point[0][];
        }
        Point[][] points = new Point[numHoles][];
        for (int i = 0; i < numHoles; ++i) {
            double[] partitionPoint = new double[3];
            int length = GeoPolygonDecomposer.component(holes[i], -(i + 1), null, partitionPoint);
            points[i] = GeoPolygonDecomposer.coordinates(holes[i], new Point[length + 1], partitionPoint);
        }
        return points;
    }

    private static double normalizeLonMinus180Inclusive(double lon) {
        return Math.abs(lon) > 180.0 ? GeoUtils.normalizeLon(lon) : lon;
    }

    private static Point shift(Point coordinate, double dateline) {
        if (dateline == 0.0) {
            return coordinate;
        }
        return new Point(-2.0 * dateline + coordinate.getX(), coordinate.getY());
    }

    private static double intersection(double p1x, double p2x, double dateline) {
        if (p1x == p2x && p1x != dateline) {
            return Double.NaN;
        }
        if (p1x == p2x && p1x == dateline) {
            return 1.0;
        }
        double t = (dateline - p1x) / (p2x - p1x);
        if (t > 1.0 || t <= 0.0) {
            return Double.NaN;
        }
        return t;
    }

    private static final class Edge {
        Point coordinate;
        Edge next;
        Point intersect;
        int component = -1;
        static final Point MAX_COORDINATE = new Point(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);

        Edge(Point coordinate, Edge next, Point intersection) {
            this.coordinate = coordinate;
            this.setNext(next);
            this.intersect = intersection;
            if (next != null) {
                this.component = next.component;
            }
        }

        Edge(Point coordinate, Edge next) {
            this(coordinate, next, MAX_COORDINATE);
        }

        void setNext(Edge next) {
            if (next != null) {
                if (this.coordinate.equals((Object)next.coordinate)) {
                    throw new InvalidShapeException("Provided shape has duplicate consecutive coordinates at: " + this.coordinate);
                }
                this.next = next;
            }
        }

        Point intersection(double position) {
            this.intersect = GeoPolygonDecomposer.position(this.coordinate, this.next.coordinate, position);
            return this.intersect;
        }

        public String toString() {
            return "Edge[Component=" + this.component + "; start=" + this.coordinate + " ; intersection=" + this.intersect + "]";
        }
    }
}

