/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.admin.indices.rollover;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.elasticsearch.Version;
import org.elasticsearch.action.admin.indices.create.CreateIndexClusterStateUpdateRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.rollover.Condition;
import org.elasticsearch.action.admin.indices.rollover.RolloverInfo;
import org.elasticsearch.action.support.ActiveShardCount;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.AliasAction;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.MetadataCreateDataStreamService;
import org.elasticsearch.cluster.metadata.MetadataCreateIndexService;
import org.elasticsearch.cluster.metadata.MetadataIndexAliasesService;
import org.elasticsearch.cluster.metadata.MetadataIndexTemplateService;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.indices.SystemDataStreamDescriptor;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.snapshots.SnapshotInProgressException;
import org.elasticsearch.snapshots.SnapshotsService;
import org.elasticsearch.threadpool.ThreadPool;

public class MetadataRolloverService {
    private static final Pattern INDEX_NAME_PATTERN = Pattern.compile("^.*-\\d+$");
    private static final List<IndexAbstraction.Type> VALID_ROLLOVER_TARGETS = org.elasticsearch.common.collect.List.of((Object)((Object)IndexAbstraction.Type.ALIAS), (Object)((Object)IndexAbstraction.Type.DATA_STREAM));
    private final ThreadPool threadPool;
    private final MetadataCreateIndexService createIndexService;
    private final MetadataIndexAliasesService indexAliasesService;
    private final IndexNameExpressionResolver indexNameExpressionResolver;
    private final SystemIndices systemIndices;

    @Inject
    public MetadataRolloverService(ThreadPool threadPool, MetadataCreateIndexService createIndexService, MetadataIndexAliasesService indexAliasesService, IndexNameExpressionResolver indexNameExpressionResolver, SystemIndices systemIndices) {
        this.threadPool = threadPool;
        this.createIndexService = createIndexService;
        this.indexAliasesService = indexAliasesService;
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        this.systemIndices = systemIndices;
    }

    public RolloverResult rolloverClusterState(ClusterState currentState, String rolloverTarget, String newIndexName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, boolean silent, boolean onlyValidate) throws Exception {
        MetadataRolloverService.validate(currentState.metadata(), rolloverTarget, newIndexName, createIndexRequest);
        IndexAbstraction indexAbstraction = (IndexAbstraction)currentState.metadata().getIndicesLookup().get(rolloverTarget);
        switch (indexAbstraction.getType()) {
            case ALIAS: {
                return this.rolloverAlias(currentState, (IndexAbstraction.Alias)indexAbstraction, rolloverTarget, newIndexName, createIndexRequest, metConditions, silent, onlyValidate);
            }
            case DATA_STREAM: {
                return this.rolloverDataStream(currentState, (IndexAbstraction.DataStream)indexAbstraction, rolloverTarget, createIndexRequest, metConditions, silent, onlyValidate);
            }
        }
        throw new IllegalStateException("unable to roll over type [" + indexAbstraction.getType().getDisplayName() + "]");
    }

    public void validateIndexName(ClusterState state, String index) {
        this.createIndexService.validateIndexName(index, state);
    }

    public NameResolution resolveRolloverNames(ClusterState currentState, String rolloverTarget, String newIndexName, CreateIndexRequest createIndexRequest) {
        MetadataRolloverService.validate(currentState.metadata(), rolloverTarget, newIndexName, createIndexRequest);
        IndexAbstraction indexAbstraction = (IndexAbstraction)currentState.metadata().getIndicesLookup().get(rolloverTarget);
        switch (indexAbstraction.getType()) {
            case ALIAS: {
                return this.resolveAliasRolloverNames((IndexAbstraction.Alias)indexAbstraction, newIndexName);
            }
            case DATA_STREAM: {
                return this.resolveDataStreamRolloverNames(currentState, (IndexAbstraction.DataStream)indexAbstraction);
            }
        }
        throw new IllegalStateException("unable to roll over type [" + indexAbstraction.getType().getDisplayName() + "]");
    }

    private NameResolution resolveAliasRolloverNames(IndexAbstraction.Alias alias, String newIndexName) {
        IndexMetadata writeIndex = alias.getWriteIndex();
        String sourceProvidedName = writeIndex.getSettings().get("index.provided_name", writeIndex.getIndex().getName());
        String sourceIndexName = writeIndex.getIndex().getName();
        String unresolvedName = newIndexName != null ? newIndexName : MetadataRolloverService.generateRolloverIndexName(sourceProvidedName, this.indexNameExpressionResolver);
        String rolloverIndexName = this.indexNameExpressionResolver.resolveDateMathExpression(unresolvedName);
        return new NameResolution(sourceIndexName, unresolvedName, rolloverIndexName);
    }

    private NameResolution resolveDataStreamRolloverNames(ClusterState currentState, IndexAbstraction.DataStream dataStream) {
        Version minNodeVersion = currentState.nodes().getMinNodeVersion();
        DataStream ds = dataStream.getDataStream();
        IndexMetadata originalWriteIndex = dataStream.getWriteIndex();
        DataStream rolledDataStream = ds.rollover(currentState.getMetadata(), "uuid", minNodeVersion);
        return new NameResolution(originalWriteIndex.getIndex().getName(), null, rolledDataStream.getWriteIndex().getName());
    }

    private RolloverResult rolloverAlias(ClusterState currentState, IndexAbstraction.Alias alias, String aliasName, String newIndexName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, boolean silent, boolean onlyValidate) throws Exception {
        NameResolution names = this.resolveAliasRolloverNames(alias, newIndexName);
        String sourceIndexName = names.sourceName;
        String rolloverIndexName = names.rolloverName;
        String unresolvedName = names.unresolvedName;
        Metadata metadata = currentState.metadata();
        IndexMetadata writeIndex = alias.getWriteIndex();
        AliasMetadata aliasMetadata = writeIndex.getAliases().get(alias.getName());
        boolean explicitWriteIndex = Boolean.TRUE.equals(aliasMetadata.writeIndex());
        Boolean isHidden = IndexMetadata.INDEX_HIDDEN_SETTING.exists(createIndexRequest.settings()) ? IndexMetadata.INDEX_HIDDEN_SETTING.get(createIndexRequest.settings()) : null;
        this.createIndexService.validateIndexName(rolloverIndexName, currentState);
        MetadataRolloverService.checkNoDuplicatedAliasInIndexTemplate(metadata, rolloverIndexName, aliasName, isHidden);
        if (onlyValidate) {
            return new RolloverResult(rolloverIndexName, sourceIndexName, currentState);
        }
        CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest = MetadataRolloverService.prepareCreateIndexRequest(unresolvedName, rolloverIndexName, createIndexRequest);
        ClusterState newState = this.createIndexService.applyCreateIndexRequest(currentState, createIndexClusterStateRequest, silent);
        newState = this.indexAliasesService.applyAliasActions(newState, MetadataRolloverService.rolloverAliasToNewIndex(sourceIndexName, rolloverIndexName, explicitWriteIndex, aliasMetadata.isHidden(), aliasName));
        RolloverInfo rolloverInfo = new RolloverInfo(aliasName, metConditions, this.threadPool.absoluteTimeInMillis());
        newState = ClusterState.builder(newState).metadata(Metadata.builder(newState.metadata()).put(IndexMetadata.builder(newState.metadata().index(sourceIndexName)).putRolloverInfo(rolloverInfo))).build();
        return new RolloverResult(rolloverIndexName, sourceIndexName, newState);
    }

    private RolloverResult rolloverDataStream(ClusterState currentState, IndexAbstraction.DataStream dataStream, String dataStreamName, CreateIndexRequest createIndexRequest, List<Condition<?>> metConditions, boolean silent, boolean onlyValidate) throws Exception {
        SystemDataStreamDescriptor systemDataStreamDescriptor;
        if (!SnapshotsService.snapshottingDataStreams(currentState, Collections.singleton(dataStream.getName())).isEmpty()) {
            throw new SnapshotInProgressException("Cannot roll over data stream that is being snapshotted: " + dataStream.getName() + ". Try again after snapshot finishes or cancel the currently running snapshot.");
        }
        if (!dataStream.isSystem()) {
            systemDataStreamDescriptor = null;
            MetadataCreateDataStreamService.lookupTemplateForDataStream(dataStreamName, currentState.metadata());
        } else {
            systemDataStreamDescriptor = this.systemIndices.findMatchingDataStreamDescriptor(dataStreamName);
            if (systemDataStreamDescriptor == null) {
                throw new IllegalArgumentException("no system data stream descriptor found for data stream [" + dataStreamName + "]");
            }
        }
        Version minNodeVersion = currentState.nodes().getMinNodeVersion();
        DataStream ds = dataStream.getDataStream();
        IndexMetadata originalWriteIndex = dataStream.getWriteIndex();
        DataStream rolledDataStream = ds.rollover(currentState.metadata(), "uuid", minNodeVersion);
        this.createIndexService.validateIndexName(rolledDataStream.getWriteIndex().getName(), currentState);
        if (onlyValidate) {
            return new RolloverResult(rolledDataStream.getWriteIndex().getName(), originalWriteIndex.getIndex().getName(), currentState);
        }
        CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest = MetadataRolloverService.prepareDataStreamCreateIndexRequest(dataStreamName, rolledDataStream.getWriteIndex().getName(), createIndexRequest, systemDataStreamDescriptor);
        ClusterState newState = this.createIndexService.applyCreateIndexRequest(currentState, createIndexClusterStateRequest, silent, (builder, indexMetadata) -> builder.put(ds.rollover(currentState.metadata(), indexMetadata.getIndexUUID(), minNodeVersion)));
        RolloverInfo rolloverInfo = new RolloverInfo(dataStreamName, metConditions, this.threadPool.absoluteTimeInMillis());
        newState = ClusterState.builder(newState).metadata(Metadata.builder(newState.metadata()).put(IndexMetadata.builder(newState.metadata().index(originalWriteIndex.getIndex())).putRolloverInfo(rolloverInfo))).build();
        return new RolloverResult(rolledDataStream.getWriteIndex().getName(), originalWriteIndex.getIndex().getName(), newState);
    }

    static String generateRolloverIndexName(String sourceIndexName, IndexNameExpressionResolver indexNameExpressionResolver) {
        boolean isDateMath;
        String resolvedName = indexNameExpressionResolver.resolveDateMathExpression(sourceIndexName);
        boolean bl = isDateMath = !sourceIndexName.equals(resolvedName);
        if (INDEX_NAME_PATTERN.matcher(resolvedName).matches()) {
            int numberIndex = sourceIndexName.lastIndexOf("-");
            assert (numberIndex != -1) : "no separator '-' found";
            int counter = Integer.parseInt(sourceIndexName.substring(numberIndex + 1, isDateMath ? sourceIndexName.length() - 1 : sourceIndexName.length()));
            String newName = sourceIndexName.substring(0, numberIndex) + "-" + String.format(Locale.ROOT, "%06d", ++counter) + (isDateMath ? ">" : "");
            return newName;
        }
        throw new IllegalArgumentException("index name [" + sourceIndexName + "] does not match pattern '^.*-\\d+$'");
    }

    static CreateIndexClusterStateUpdateRequest prepareDataStreamCreateIndexRequest(String dataStreamName, String targetIndexName, CreateIndexRequest createIndexRequest, SystemDataStreamDescriptor descriptor) {
        Settings settings = descriptor != null ? Settings.EMPTY : Settings.builder().put("index.hidden", true).build();
        return MetadataRolloverService.prepareCreateIndexRequest(targetIndexName, targetIndexName, "rollover_data_stream", createIndexRequest, settings).dataStreamName(dataStreamName).systemDataStreamDescriptor(descriptor);
    }

    static CreateIndexClusterStateUpdateRequest prepareCreateIndexRequest(String providedIndexName, String targetIndexName, CreateIndexRequest createIndexRequest) {
        return MetadataRolloverService.prepareCreateIndexRequest(providedIndexName, targetIndexName, "rollover_index", createIndexRequest, null);
    }

    static CreateIndexClusterStateUpdateRequest prepareCreateIndexRequest(String providedIndexName, String targetIndexName, String cause, CreateIndexRequest createIndexRequest, Settings settings) {
        Settings.Builder b = Settings.builder().put(createIndexRequest.settings());
        if (settings != null) {
            b.put(settings);
        }
        return ((CreateIndexClusterStateUpdateRequest)((CreateIndexClusterStateUpdateRequest)new CreateIndexClusterStateUpdateRequest(cause, targetIndexName, providedIndexName).ackTimeout(createIndexRequest.timeout())).masterNodeTimeout(createIndexRequest.masterNodeTimeout())).settings(b.build()).aliases(createIndexRequest.aliases()).waitForActiveShards(ActiveShardCount.NONE).mappings(createIndexRequest.mappings());
    }

    static List<AliasAction> rolloverAliasToNewIndex(String oldIndex, String newIndex, boolean explicitWriteIndex, @Nullable Boolean isHidden, String alias) {
        if (explicitWriteIndex) {
            return Collections.unmodifiableList(Arrays.asList(new AliasAction.Add(newIndex, alias, null, null, null, true, isHidden), new AliasAction.Add(oldIndex, alias, null, null, null, false, isHidden)));
        }
        return Collections.unmodifiableList(Arrays.asList(new AliasAction.Add(newIndex, alias, null, null, null, null, isHidden), new AliasAction.Remove(oldIndex, alias, null)));
    }

    static void checkNoDuplicatedAliasInIndexTemplate(Metadata metadata, String rolloverIndexName, String rolloverRequestAlias, @Nullable Boolean isHidden) {
        String matchedV2Template = MetadataIndexTemplateService.findV2Template(metadata, rolloverIndexName, isHidden == null ? false : isHidden);
        if (matchedV2Template != null) {
            List<Map<String, AliasMetadata>> aliases = MetadataIndexTemplateService.resolveAliases(metadata, matchedV2Template, false);
            for (Map<String, AliasMetadata> aliasConfig : aliases) {
                if (!aliasConfig.containsKey(rolloverRequestAlias)) continue;
                throw new IllegalArgumentException(String.format(Locale.ROOT, "Rollover alias [%s] can point to multiple indices, found duplicated alias [%s] in index template [%s]", rolloverRequestAlias, aliasConfig.keySet(), matchedV2Template));
            }
            return;
        }
        List<IndexTemplateMetadata> matchedTemplates = MetadataIndexTemplateService.findV1Templates(metadata, rolloverIndexName, isHidden);
        for (IndexTemplateMetadata template : matchedTemplates) {
            if (!template.aliases().containsKey(rolloverRequestAlias)) continue;
            throw new IllegalArgumentException(String.format(Locale.ROOT, "Rollover alias [%s] can point to multiple indices, found duplicated alias [%s] in index template [%s]", rolloverRequestAlias, template.aliases().keys(), template.name()));
        }
    }

    static void validate(Metadata metadata, String rolloverTarget, String newIndexName, CreateIndexRequest request) {
        IndexAbstraction indexAbstraction = (IndexAbstraction)metadata.getIndicesLookup().get(rolloverTarget);
        if (indexAbstraction == null) {
            throw new IllegalArgumentException("rollover target [" + rolloverTarget + "] does not exist");
        }
        if (!VALID_ROLLOVER_TARGETS.contains((Object)indexAbstraction.getType())) {
            throw new IllegalArgumentException("rollover target is a [" + indexAbstraction.getType().getDisplayName() + "] but one of [" + Strings.collectionToCommaDelimitedString(VALID_ROLLOVER_TARGETS.stream().map(IndexAbstraction.Type::getDisplayName).collect(Collectors.toList())) + "] was expected");
        }
        if (indexAbstraction.getWriteIndex() == null) {
            throw new IllegalArgumentException("rollover target [" + indexAbstraction.getName() + "] does not point to a write index");
        }
        if (indexAbstraction.getType() == IndexAbstraction.Type.DATA_STREAM) {
            if (!Strings.isNullOrEmpty(newIndexName)) {
                throw new IllegalArgumentException("new index name may not be specified when rolling over a data stream");
            }
            if (!request.settings().equals(Settings.EMPTY) || request.aliases().size() > 0 || request.mappings().size() > 0) {
                throw new IllegalArgumentException("aliases, mappings, and index settings may not be specified when rolling over a data stream");
            }
        }
    }

    public static class RolloverResult {
        public final String rolloverIndexName;
        public final String sourceIndexName;
        public final ClusterState clusterState;

        private RolloverResult(String rolloverIndexName, String sourceIndexName, ClusterState clusterState) {
            this.rolloverIndexName = rolloverIndexName;
            this.sourceIndexName = sourceIndexName;
            this.clusterState = clusterState;
        }

        public String toString() {
            return String.format(Locale.ROOT, "cluster state version [%d], rollover index name [%s], source index name [%s]", this.clusterState.version(), this.rolloverIndexName, this.sourceIndexName);
        }
    }

    public static class NameResolution {
        final String sourceName;
        @Nullable
        final String unresolvedName;
        final String rolloverName;

        NameResolution(String sourceName, String unresolvedName, String rolloverName) {
            this.sourceName = sourceName;
            this.unresolvedName = unresolvedName;
            this.rolloverName = rolloverName;
        }
    }
}

