/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.search.suggest.completion.context;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.apache.lucene.document.LatLonDocValuesField;
import org.apache.lucene.document.LatLonPoint;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.DocValuesType;
import org.apache.lucene.index.IndexableField;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.common.geo.GeoPoint;
import org.elasticsearch.common.geo.GeoUtils;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.geometry.utils.Geohash;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.index.mapper.ParseContext;
import org.elasticsearch.search.suggest.completion.context.ContextBuilder;
import org.elasticsearch.search.suggest.completion.context.ContextMapping;
import org.elasticsearch.search.suggest.completion.context.GeoQueryContext;

public class GeoContextMapping
extends ContextMapping<GeoQueryContext> {
    public static final String FIELD_PRECISION = "precision";
    public static final String FIELD_FIELDNAME = "path";
    public static final int DEFAULT_PRECISION = 6;
    static final String CONTEXT_VALUE = "context";
    static final String CONTEXT_BOOST = "boost";
    static final String CONTEXT_PRECISION = "precision";
    static final String CONTEXT_NEIGHBOURS = "neighbours";
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(GeoContextMapping.class);
    private final int precision;
    private final String fieldName;

    private GeoContextMapping(String name, String fieldName, int precision) {
        super(ContextMapping.Type.GEO, name);
        this.precision = precision;
        this.fieldName = fieldName;
    }

    public String getFieldName() {
        return this.fieldName;
    }

    public int getPrecision() {
        return this.precision;
    }

    protected static GeoContextMapping load(String name, Map<String, Object> config) {
        Builder builder = new Builder(name);
        if (config != null) {
            Object fieldName;
            Object configPrecision = config.get("precision");
            if (configPrecision != null) {
                if (configPrecision instanceof Integer) {
                    builder.precision((Integer)configPrecision);
                } else if (configPrecision instanceof Long) {
                    builder.precision(((Long)configPrecision).longValue());
                } else if (configPrecision instanceof Double) {
                    builder.precision((Double)configPrecision);
                } else if (configPrecision instanceof Float) {
                    builder.precision(((Float)configPrecision).floatValue());
                } else {
                    builder.precision(configPrecision.toString());
                }
                config.remove("precision");
            }
            if ((fieldName = config.get(FIELD_FIELDNAME)) != null) {
                builder.field(fieldName.toString());
                config.remove(FIELD_FIELDNAME);
            }
        }
        return builder.build();
    }

    @Override
    protected XContentBuilder toInnerXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.field("precision", this.precision);
        if (this.fieldName != null) {
            builder.field(FIELD_FIELDNAME, this.fieldName);
        }
        return builder;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public Set<String> parseContext(ParseContext parseContext, XContentParser parser) throws IOException, ElasticsearchParseException {
        HashSet<String> contexts = new HashSet<String>();
        XContentParser.Token token = parser.currentToken();
        if (token == XContentParser.Token.START_ARRAY) {
            token = parser.nextToken();
            if (token == XContentParser.Token.VALUE_NUMBER) {
                double lon = parser.doubleValue();
                if (parser.nextToken() != XContentParser.Token.VALUE_NUMBER) throw new ElasticsearchParseException("latitude must be a numeric value", new Object[0]);
                double lat = parser.doubleValue();
                if (parser.nextToken() != XContentParser.Token.END_ARRAY) throw new ElasticsearchParseException("only two values [lon, lat] expected", new Object[0]);
                contexts.add(Geohash.stringEncode((double)lon, (double)lat, (int)this.precision));
                return contexts;
            } else {
                while (token != XContentParser.Token.END_ARRAY) {
                    GeoPoint point = GeoUtils.parseGeoPoint(parser);
                    contexts.add(Geohash.stringEncode((double)point.getLon(), (double)point.getLat(), (int)this.precision));
                    token = parser.nextToken();
                }
            }
            return contexts;
        } else if (token == XContentParser.Token.VALUE_STRING) {
            String geoHash = parser.text();
            CharSequence truncatedGeoHash = geoHash.subSequence(0, Math.min(geoHash.length(), this.precision));
            contexts.add(truncatedGeoHash.toString());
            return contexts;
        } else {
            GeoPoint point = GeoUtils.parseGeoPoint(parser);
            contexts.add(Geohash.stringEncode((double)point.getLon(), (double)point.getLat(), (int)this.precision));
        }
        return contexts;
    }

    @Override
    public Set<String> parseContext(ParseContext.Document document) {
        HashSet<String> geohashes = new HashSet<String>();
        if (this.fieldName != null) {
            IndexableField[] fields = document.getFields(this.fieldName);
            GeoPoint spare = new GeoPoint();
            if (fields.length == 0) {
                IndexableField[] lonFields = document.getFields(this.fieldName + ".lon");
                IndexableField[] latFields = document.getFields(this.fieldName + ".lat");
                if (lonFields.length > 0 && latFields.length > 0) {
                    for (int i = 0; i < lonFields.length; ++i) {
                        IndexableField lonField = lonFields[i];
                        IndexableField latField = latFields[i];
                        assert (lonField.fieldType().docValuesType() == latField.fieldType().docValuesType());
                        if (lonField.fieldType().docValuesType() != DocValuesType.NONE) continue;
                        spare.reset(latField.numericValue().doubleValue(), lonField.numericValue().doubleValue());
                        geohashes.add(Geohash.stringEncode((double)spare.getLon(), (double)spare.getLat(), (int)this.precision));
                    }
                }
            } else {
                for (IndexableField field : fields) {
                    if (field instanceof StringField) {
                        spare.resetFromString(field.stringValue());
                        geohashes.add(spare.geohash());
                        continue;
                    }
                    if (!(field instanceof LatLonPoint) && !(field instanceof LatLonDocValuesField)) continue;
                    spare.resetFromIndexableField(field);
                    geohashes.add(spare.geohash());
                }
            }
        }
        HashSet<String> locations = new HashSet<String>();
        for (String geohash : geohashes) {
            int precision = Math.min(this.precision, geohash.length());
            String truncatedGeohash = geohash.substring(0, precision);
            locations.add(truncatedGeohash);
        }
        return locations;
    }

    @Override
    protected GeoQueryContext fromXContent(XContentParser parser) throws IOException {
        return GeoQueryContext.fromXContent(parser);
    }

    @Override
    public List<ContextMapping.InternalQueryContext> toInternalQueryContexts(List<GeoQueryContext> queryContexts) {
        ArrayList<ContextMapping.InternalQueryContext> internalQueryContextList = new ArrayList<ContextMapping.InternalQueryContext>();
        for (GeoQueryContext queryContext : queryContexts) {
            int minPrecision = Math.min(this.precision, queryContext.getPrecision());
            GeoPoint point = queryContext.getGeoPoint();
            HashSet<String> locations = new HashSet<String>();
            String geoHash = Geohash.stringEncode((double)point.getLon(), (double)point.getLat(), (int)minPrecision);
            locations.add(geoHash);
            if (queryContext.getNeighbours().isEmpty() && geoHash.length() == this.precision) {
                Geohash.addNeighbors((String)geoHash, locations);
            } else if (!queryContext.getNeighbours().isEmpty()) {
                queryContext.getNeighbours().stream().filter(neighbourPrecision -> neighbourPrecision < geoHash.length()).forEach(neighbourPrecision -> {
                    String truncatedGeoHash = geoHash.substring(0, (int)neighbourPrecision);
                    locations.add(truncatedGeoHash);
                    Geohash.addNeighbors((String)truncatedGeoHash, (Collection)locations);
                });
            }
            internalQueryContextList.addAll(locations.stream().map(location -> new ContextMapping.InternalQueryContext((String)location, queryContext.getBoost(), location.length() < this.precision)).collect(Collectors.toList()));
        }
        return internalQueryContextList;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @Override
    public void validateReferences(Version indexVersionCreated, Function<String, MappedFieldType> fieldResolver) {
        if (this.fieldName == null) return;
        MappedFieldType mappedFieldType = fieldResolver.apply(this.fieldName);
        if (mappedFieldType == null) {
            if (!indexVersionCreated.before(Version.V_7_0_0)) throw new ElasticsearchParseException("field [{}] referenced in context [{}] is not defined in the mapping", this.fieldName, this.name);
            deprecationLogger.deprecate("geo_context_mapping", "field [{}] referenced in context [{}] is not defined in the mapping", this.fieldName, this.name);
            return;
        } else {
            if ("geo_point".equals(mappedFieldType.typeName())) return;
            if (!indexVersionCreated.before(Version.V_7_0_0)) throw new ElasticsearchParseException("field [{}] referenced in context [{}] must be mapped to geo_point, found [{}]", this.fieldName, this.name, mappedFieldType.typeName());
            deprecationLogger.deprecate("geo_context_mapping", "field [{}] referenced in context [{}] must be mapped to geo_point, found [{}]", this.fieldName, this.name, mappedFieldType.typeName());
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        GeoContextMapping that = (GeoContextMapping)o;
        if (this.precision != that.precision) {
            return false;
        }
        return !(this.fieldName == null ? that.fieldName != null : !this.fieldName.equals(that.fieldName));
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), this.precision, this.fieldName);
    }

    public static class Builder
    extends ContextBuilder<GeoContextMapping> {
        private int precision = 6;
        private String fieldName = null;

        public Builder(String name) {
            super(name);
        }

        public Builder precision(String precision) {
            return this.precision(DistanceUnit.parse(precision, DistanceUnit.METERS, DistanceUnit.METERS));
        }

        public Builder precision(double precision, DistanceUnit unit) {
            return this.precision(unit.toMeters(precision));
        }

        public Builder precision(double meters) {
            int level = GeoUtils.geoHashLevelsForPrecision(meters);
            if (GeoUtils.geoHashCellSize(level) < meters) {
                level = Math.max(1, level - 1);
            }
            return this.precision(level);
        }

        public Builder precision(int level) {
            this.precision = level;
            return this;
        }

        public Builder field(String fieldName) {
            this.fieldName = fieldName;
            return this;
        }

        @Override
        public GeoContextMapping build() {
            return new GeoContextMapping(this.name, this.fieldName, this.precision);
        }
    }
}

