/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.flush;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.flush.FlushRequest;
import org.elasticsearch.action.admin.indices.flush.SyncedFlushResponse;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.util.concurrent.CountDown;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.engine.CommitStats;
import org.elasticsearch.index.engine.Engine;
import org.elasticsearch.index.shard.IndexEventListener;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.shard.ShardNotFoundException;
import org.elasticsearch.indices.IndexClosedException;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.indices.flush.ShardsSyncedFlushResult;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class SyncedFlushService
extends AbstractComponent
implements IndexEventListener {
    private static final String PRE_SYNCED_FLUSH_ACTION_NAME = "internal:indices/flush/synced/pre";
    private static final String SYNCED_FLUSH_ACTION_NAME = "internal:indices/flush/synced/sync";
    private static final String IN_FLIGHT_OPS_ACTION_NAME = "internal:indices/flush/synced/in_flight";
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final IndexNameExpressionResolver indexNameExpressionResolver;

    @Inject
    public SyncedFlushService(Settings settings, IndicesService indicesService, ClusterService clusterService, TransportService transportService, IndexNameExpressionResolver indexNameExpressionResolver) {
        super(settings);
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        transportService.registerRequestHandler(PRE_SYNCED_FLUSH_ACTION_NAME, PreShardSyncedFlushRequest::new, "flush", new PreSyncedFlushTransportHandler());
        transportService.registerRequestHandler(SYNCED_FLUSH_ACTION_NAME, ShardSyncedFlushRequest::new, "flush", new SyncedFlushTransportHandler());
        transportService.registerRequestHandler(IN_FLIGHT_OPS_ACTION_NAME, InFlightOpsRequest::new, "same", new InFlightOpCountTransportHandler());
    }

    @Override
    public void onShardInactive(final IndexShard indexShard) {
        if (indexShard.routingEntry().primary()) {
            this.attemptSyncedFlush(indexShard.shardId(), new ActionListener<ShardsSyncedFlushResult>(){

                @Override
                public void onResponse(ShardsSyncedFlushResult syncedFlushResult) {
                    SyncedFlushService.this.logger.trace("{} sync flush on inactive shard returned successfully for sync_id: {}", (Object)syncedFlushResult.getShardId(), (Object)syncedFlushResult.syncId());
                }

                @Override
                public void onFailure(Exception e) {
                    SyncedFlushService.this.logger.debug(() -> new ParameterizedMessage("{} sync flush on inactive shard failed", (Object)indexShard.shardId()), (Throwable)e);
                }
            });
        }
    }

    public void attemptSyncedFlush(String[] aliasesOrIndices, IndicesOptions indicesOptions, final ActionListener<SyncedFlushResponse> listener) {
        ClusterState state = this.clusterService.state();
        Index[] concreteIndices = this.indexNameExpressionResolver.concreteIndices(state, indicesOptions, aliasesOrIndices);
        final ConcurrentMap<String, List<ShardsSyncedFlushResult>> results = ConcurrentCollections.newConcurrentMap();
        int numberOfShards = 0;
        for (Index index : concreteIndices) {
            IndexMetaData indexMetaData = state.metaData().getIndexSafe(index);
            numberOfShards += indexMetaData.getNumberOfShards();
            results.put(index.getName(), Collections.synchronizedList(new ArrayList()));
        }
        if (numberOfShards == 0) {
            listener.onResponse(new SyncedFlushResponse(results));
            return;
        }
        final CountDown countDown = new CountDown(numberOfShards);
        for (Index concreteIndex : concreteIndices) {
            final String index = concreteIndex.getName();
            final IndexMetaData indexMetaData = state.metaData().getIndexSafe(concreteIndex);
            int indexNumberOfShards = indexMetaData.getNumberOfShards();
            for (int shard = 0; shard < indexNumberOfShards; ++shard) {
                final ShardId shardId = new ShardId(indexMetaData.getIndex(), shard);
                this.innerAttemptSyncedFlush(shardId, state, new ActionListener<ShardsSyncedFlushResult>(){

                    @Override
                    public void onResponse(ShardsSyncedFlushResult syncedFlushResult) {
                        ((List)results.get(index)).add(syncedFlushResult);
                        if (countDown.countDown()) {
                            listener.onResponse(new SyncedFlushResponse(results));
                        }
                    }

                    @Override
                    public void onFailure(Exception e) {
                        SyncedFlushService.this.logger.debug("{} unexpected error while executing synced flush", (Object)shardId);
                        int totalShards = indexMetaData.getNumberOfReplicas() + 1;
                        ((List)results.get(index)).add(new ShardsSyncedFlushResult(shardId, totalShards, e.getMessage()));
                        if (countDown.countDown()) {
                            listener.onResponse(new SyncedFlushResponse(results));
                        }
                    }
                });
            }
        }
    }

    public void attemptSyncedFlush(ShardId shardId, ActionListener<ShardsSyncedFlushResult> actionListener) {
        this.innerAttemptSyncedFlush(shardId, this.clusterService.state(), actionListener);
    }

    private void innerAttemptSyncedFlush(final ShardId shardId, final ClusterState state, final ActionListener<ShardsSyncedFlushResult> actionListener) {
        try {
            final IndexShardRoutingTable shardRoutingTable = this.getShardRoutingTable(shardId, state);
            final List<ShardRouting> activeShards = shardRoutingTable.activeShards();
            final int totalShards = shardRoutingTable.getSize();
            if (activeShards.size() == 0) {
                actionListener.onResponse(new ShardsSyncedFlushResult(shardId, totalShards, "no active shards"));
                return;
            }
            ActionListener<Map<String, PreSyncedFlushResponse>> presyncListener = new ActionListener<Map<String, PreSyncedFlushResponse>>(){

                @Override
                public void onResponse(final Map<String, PreSyncedFlushResponse> presyncResponses) {
                    if (presyncResponses.isEmpty()) {
                        actionListener.onResponse(new ShardsSyncedFlushResult(shardId, totalShards, "all shards failed to commit on pre-sync"));
                        return;
                    }
                    ActionListener<InFlightOpsResponse> inflightOpsListener = new ActionListener<InFlightOpsResponse>(){

                        @Override
                        public void onResponse(InFlightOpsResponse response) {
                            int inflight = response.opCount();
                            assert (inflight >= 0);
                            if (inflight != 0) {
                                actionListener.onResponse(new ShardsSyncedFlushResult(shardId, totalShards, "[" + inflight + "] ongoing operations on primary"));
                            } else {
                                String syncId = UUIDs.base64UUID();
                                SyncedFlushService.this.sendSyncRequests(syncId, activeShards, state, presyncResponses, shardId, totalShards, actionListener);
                            }
                        }

                        @Override
                        public void onFailure(Exception e) {
                            actionListener.onFailure(e);
                        }
                    };
                    SyncedFlushService.this.getInflightOpsCount(shardId, state, shardRoutingTable, inflightOpsListener);
                }

                @Override
                public void onFailure(Exception e) {
                    actionListener.onFailure(e);
                }
            };
            this.sendPreSyncRequests(activeShards, state, shardId, presyncListener);
        }
        catch (Exception e) {
            actionListener.onFailure(e);
        }
    }

    final IndexShardRoutingTable getShardRoutingTable(ShardId shardId, ClusterState state) {
        IndexRoutingTable indexRoutingTable = state.routingTable().index(shardId.getIndexName());
        if (indexRoutingTable == null) {
            IndexMetaData index = state.getMetaData().index(shardId.getIndex());
            if (index != null && index.getState() == IndexMetaData.State.CLOSE) {
                throw new IndexClosedException(shardId.getIndex());
            }
            throw new IndexNotFoundException(shardId.getIndexName());
        }
        IndexShardRoutingTable shardRoutingTable = indexRoutingTable.shard(shardId.id());
        if (shardRoutingTable == null) {
            throw new ShardNotFoundException(shardId);
        }
        return shardRoutingTable;
    }

    protected void getInflightOpsCount(final ShardId shardId, ClusterState state, IndexShardRoutingTable shardRoutingTable, final ActionListener<InFlightOpsResponse> listener) {
        try {
            ShardRouting primaryShard = shardRoutingTable.primaryShard();
            DiscoveryNode primaryNode = state.nodes().get(primaryShard.currentNodeId());
            if (primaryNode == null) {
                this.logger.trace("{} failed to resolve node for primary shard {}, skipping sync", (Object)shardId, (Object)primaryShard);
                listener.onResponse(new InFlightOpsResponse(-1));
                return;
            }
            this.logger.trace("{} retrieving in flight operation count", (Object)shardId);
            this.transportService.sendRequest(primaryNode, IN_FLIGHT_OPS_ACTION_NAME, new InFlightOpsRequest(shardId), new TransportResponseHandler<InFlightOpsResponse>(){

                @Override
                public InFlightOpsResponse newInstance() {
                    return new InFlightOpsResponse();
                }

                @Override
                public void handleResponse(InFlightOpsResponse response) {
                    listener.onResponse(response);
                }

                @Override
                public void handleException(TransportException exp) {
                    SyncedFlushService.this.logger.debug("{} unexpected error while retrieving in flight op count", (Object)shardId);
                    listener.onFailure(exp);
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    private int numDocsOnPrimary(List<ShardRouting> shards, Map<String, PreSyncedFlushResponse> preSyncResponses) {
        for (ShardRouting shard : shards) {
            PreSyncedFlushResponse resp;
            if (!shard.primary() || (resp = preSyncResponses.get(shard.currentNodeId())) == null) continue;
            return resp.numDocs;
        }
        return -1;
    }

    void sendSyncRequests(final String syncId, final List<ShardRouting> shards, ClusterState state, Map<String, PreSyncedFlushResponse> preSyncResponses, final ShardId shardId, final int totalShards, final ActionListener<ShardsSyncedFlushResult> listener) {
        final CountDown countDown = new CountDown(shards.size());
        final ConcurrentMap<ShardRouting, ShardSyncedFlushResponse> results = ConcurrentCollections.newConcurrentMap();
        int numDocsOnPrimary = this.numDocsOnPrimary(shards, preSyncResponses);
        for (final ShardRouting shard : shards) {
            final DiscoveryNode node = state.nodes().get(shard.currentNodeId());
            if (node == null) {
                this.logger.trace("{} is assigned to an unknown node. skipping for sync id [{}]. shard routing {}", (Object)shardId, (Object)syncId, (Object)shard);
                results.put(shard, new ShardSyncedFlushResponse("unknown node"));
                this.countDownAndSendResponseIfDone(syncId, shards, shardId, totalShards, listener, countDown, results);
                continue;
            }
            PreSyncedFlushResponse preSyncedResponse = preSyncResponses.get(shard.currentNodeId());
            if (preSyncedResponse == null) {
                this.logger.trace("{} can't resolve expected commit id for current node, skipping for sync id [{}]. shard routing {}", (Object)shardId, (Object)syncId, (Object)shard);
                results.put(shard, new ShardSyncedFlushResponse("no commit id from pre-sync flush"));
                this.countDownAndSendResponseIfDone(syncId, shards, shardId, totalShards, listener, countDown, results);
                continue;
            }
            if (preSyncedResponse.numDocs != numDocsOnPrimary && preSyncedResponse.numDocs != -1 && numDocsOnPrimary != -1) {
                this.logger.warn("{} can't to issue sync id [{}] for out of sync replica [{}] with num docs [{}]; num docs on primary [{}]", (Object)shardId, (Object)syncId, (Object)shard, (Object)preSyncedResponse.numDocs, (Object)numDocsOnPrimary);
                results.put(shard, new ShardSyncedFlushResponse("out of sync replica; num docs on replica [" + preSyncedResponse.numDocs + "]; num docs on primary [" + numDocsOnPrimary + "]"));
                this.countDownAndSendResponseIfDone(syncId, shards, shardId, totalShards, listener, countDown, results);
                continue;
            }
            this.logger.trace("{} sending synced flush request to {}. sync id [{}].", (Object)shardId, (Object)shard, (Object)syncId);
            this.transportService.sendRequest(node, SYNCED_FLUSH_ACTION_NAME, new ShardSyncedFlushRequest(shard.shardId(), syncId, preSyncedResponse.commitId), new TransportResponseHandler<ShardSyncedFlushResponse>(){

                @Override
                public ShardSyncedFlushResponse newInstance() {
                    return new ShardSyncedFlushResponse();
                }

                @Override
                public void handleResponse(ShardSyncedFlushResponse response) {
                    ShardSyncedFlushResponse existing = results.put(shard, response);
                    assert (existing == null) : "got two answers for node [" + node + "]";
                    SyncedFlushService.this.countDownAndSendResponseIfDone(syncId, shards, shardId, totalShards, listener, countDown, results);
                }

                @Override
                public void handleException(TransportException exp) {
                    SyncedFlushService.this.logger.trace(() -> new ParameterizedMessage("{} error while performing synced flush on [{}], skipping", (Object)shardId, (Object)shard), (Throwable)exp);
                    results.put(shard, new ShardSyncedFlushResponse(exp.getMessage()));
                    SyncedFlushService.this.countDownAndSendResponseIfDone(syncId, shards, shardId, totalShards, listener, countDown, results);
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }
    }

    private void countDownAndSendResponseIfDone(String syncId, List<ShardRouting> shards, ShardId shardId, int totalShards, ActionListener<ShardsSyncedFlushResult> listener, CountDown countDown, Map<ShardRouting, ShardSyncedFlushResponse> results) {
        if (countDown.countDown()) {
            assert (results.size() == shards.size());
            listener.onResponse(new ShardsSyncedFlushResult(shardId, syncId, totalShards, results));
        }
    }

    void sendPreSyncRequests(List<ShardRouting> shards, ClusterState state, final ShardId shardId, final ActionListener<Map<String, PreSyncedFlushResponse>> listener) {
        final CountDown countDown = new CountDown(shards.size());
        final ConcurrentMap presyncResponses = ConcurrentCollections.newConcurrentMap();
        for (final ShardRouting shard : shards) {
            this.logger.trace("{} sending pre-synced flush request to {}", (Object)shardId, (Object)shard);
            final DiscoveryNode node = state.nodes().get(shard.currentNodeId());
            if (node == null) {
                this.logger.trace("{} shard routing {} refers to an unknown node. skipping.", (Object)shardId, (Object)shard);
                if (!countDown.countDown()) continue;
                listener.onResponse(presyncResponses);
                continue;
            }
            this.transportService.sendRequest(node, PRE_SYNCED_FLUSH_ACTION_NAME, new PreShardSyncedFlushRequest(shard.shardId()), new TransportResponseHandler<PreSyncedFlushResponse>(){

                @Override
                public PreSyncedFlushResponse newInstance() {
                    return new PreSyncedFlushResponse();
                }

                @Override
                public void handleResponse(PreSyncedFlushResponse response) {
                    PreSyncedFlushResponse existing = presyncResponses.putIfAbsent(node.getId(), response);
                    assert (existing == null) : "got two answers for node [" + node + "]";
                    if (countDown.countDown()) {
                        listener.onResponse(presyncResponses);
                    }
                }

                @Override
                public void handleException(TransportException exp) {
                    SyncedFlushService.this.logger.trace(() -> new ParameterizedMessage("{} error while performing pre synced flush on [{}], skipping", (Object)shardId, (Object)shard), (Throwable)exp);
                    if (countDown.countDown()) {
                        listener.onResponse(presyncResponses);
                    }
                }

                @Override
                public String executor() {
                    return "same";
                }
            });
        }
    }

    private PreSyncedFlushResponse performPreSyncedFlush(PreShardSyncedFlushRequest request) {
        IndexShard indexShard = this.indicesService.indexServiceSafe(request.shardId().getIndex()).getShard(request.shardId().id());
        FlushRequest flushRequest = new FlushRequest(new String[0]).force(false).waitIfOngoing(true);
        this.logger.trace("{} performing pre sync flush", (Object)request.shardId());
        indexShard.flush(flushRequest);
        CommitStats commitStats = indexShard.commitStats();
        Engine.CommitId commitId = commitStats.getRawCommitId();
        this.logger.trace("{} pre sync flush done. commit id {}, num docs {}", (Object)request.shardId(), (Object)commitId, (Object)commitStats.getNumDocs());
        return new PreSyncedFlushResponse(commitId, commitStats.getNumDocs());
    }

    private ShardSyncedFlushResponse performSyncedFlush(ShardSyncedFlushRequest request) {
        IndexService indexService = this.indicesService.indexServiceSafe(request.shardId().getIndex());
        IndexShard indexShard = indexService.getShard(request.shardId().id());
        this.logger.trace("{} performing sync flush. sync id [{}], expected commit id {}", (Object)request.shardId(), (Object)request.syncId(), (Object)request.expectedCommitId());
        Engine.SyncedFlushResult result = indexShard.syncFlush(request.syncId(), request.expectedCommitId());
        this.logger.trace("{} sync flush done. sync id [{}], result [{}]", (Object)request.shardId(), (Object)request.syncId(), (Object)result);
        switch (result) {
            case SUCCESS: {
                return new ShardSyncedFlushResponse();
            }
            case COMMIT_MISMATCH: {
                return new ShardSyncedFlushResponse("commit has changed");
            }
            case PENDING_OPERATIONS: {
                return new ShardSyncedFlushResponse("pending operations");
            }
        }
        throw new ElasticsearchException("unknown synced flush result [" + (Object)((Object)result) + "]", new Object[0]);
    }

    private InFlightOpsResponse performInFlightOps(InFlightOpsRequest request) {
        IndexService indexService = this.indicesService.indexServiceSafe(request.shardId().getIndex());
        IndexShard indexShard = indexService.getShard(request.shardId().id());
        if (!indexShard.routingEntry().primary()) {
            throw new IllegalStateException("[" + request.shardId() + "] expected a primary shard");
        }
        int opCount = indexShard.getActiveOperationsCount();
        this.logger.trace("{} in flight operations sampled at [{}]", (Object)request.shardId(), (Object)opCount);
        return new InFlightOpsResponse(opCount);
    }

    private final class InFlightOpCountTransportHandler
    implements TransportRequestHandler<InFlightOpsRequest> {
        private InFlightOpCountTransportHandler() {
        }

        @Override
        public void messageReceived(InFlightOpsRequest request, TransportChannel channel) throws Exception {
            channel.sendResponse(SyncedFlushService.this.performInFlightOps(request));
        }
    }

    private final class SyncedFlushTransportHandler
    implements TransportRequestHandler<ShardSyncedFlushRequest> {
        private SyncedFlushTransportHandler() {
        }

        @Override
        public void messageReceived(ShardSyncedFlushRequest request, TransportChannel channel) throws Exception {
            channel.sendResponse(SyncedFlushService.this.performSyncedFlush(request));
        }
    }

    private final class PreSyncedFlushTransportHandler
    implements TransportRequestHandler<PreShardSyncedFlushRequest> {
        private PreSyncedFlushTransportHandler() {
        }

        @Override
        public void messageReceived(PreShardSyncedFlushRequest request, TransportChannel channel) throws Exception {
            channel.sendResponse(SyncedFlushService.this.performPreSyncedFlush(request));
        }
    }

    static final class InFlightOpsResponse
    extends TransportResponse {
        int opCount;

        InFlightOpsResponse() {
        }

        InFlightOpsResponse(int opCount) {
            this.opCount = opCount;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.opCount = in.readVInt();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeVInt(this.opCount);
        }

        public int opCount() {
            return this.opCount;
        }

        public String toString() {
            return "InFlightOpsResponse{opCount=" + this.opCount + '}';
        }
    }

    public static final class InFlightOpsRequest
    extends TransportRequest {
        private ShardId shardId;

        public InFlightOpsRequest() {
        }

        public InFlightOpsRequest(ShardId shardId) {
            this.shardId = shardId;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardId = ShardId.readShardId(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.shardId.writeTo(out);
        }

        public ShardId shardId() {
            return this.shardId;
        }

        public String toString() {
            return "InFlightOpsRequest{shardId=" + this.shardId + '}';
        }
    }

    public static final class ShardSyncedFlushResponse
    extends TransportResponse {
        String failureReason;

        public ShardSyncedFlushResponse() {
            this.failureReason = null;
        }

        public ShardSyncedFlushResponse(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.failureReason = in.readOptionalString();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeOptionalString(this.failureReason);
        }

        public boolean success() {
            return this.failureReason == null;
        }

        public String failureReason() {
            return this.failureReason;
        }

        public String toString() {
            return "ShardSyncedFlushResponse{success=" + this.success() + ", failureReason='" + this.failureReason + '\'' + '}';
        }

        public static ShardSyncedFlushResponse readSyncedFlushResponse(StreamInput in) throws IOException {
            ShardSyncedFlushResponse shardSyncedFlushResponse = new ShardSyncedFlushResponse();
            shardSyncedFlushResponse.readFrom(in);
            return shardSyncedFlushResponse;
        }
    }

    public static final class ShardSyncedFlushRequest
    extends TransportRequest {
        private String syncId;
        private Engine.CommitId expectedCommitId;
        private ShardId shardId;

        public ShardSyncedFlushRequest() {
        }

        public ShardSyncedFlushRequest(ShardId shardId, String syncId, Engine.CommitId expectedCommitId) {
            this.expectedCommitId = expectedCommitId;
            this.shardId = shardId;
            this.syncId = syncId;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardId = ShardId.readShardId(in);
            this.expectedCommitId = new Engine.CommitId(in);
            this.syncId = in.readString();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.shardId.writeTo(out);
            this.expectedCommitId.writeTo(out);
            out.writeString(this.syncId);
        }

        public ShardId shardId() {
            return this.shardId;
        }

        public String syncId() {
            return this.syncId;
        }

        public Engine.CommitId expectedCommitId() {
            return this.expectedCommitId;
        }

        public String toString() {
            return "ShardSyncedFlushRequest{shardId=" + this.shardId + ",syncId='" + this.syncId + '\'' + '}';
        }
    }

    static final class PreSyncedFlushResponse
    extends TransportResponse {
        static final int UNKNOWN_NUM_DOCS = -1;
        public static final int V_6_2_2_ID = 6020299;
        Engine.CommitId commitId;
        int numDocs;

        PreSyncedFlushResponse() {
        }

        PreSyncedFlushResponse(Engine.CommitId commitId, int numDocs) {
            this.commitId = commitId;
            this.numDocs = numDocs;
        }

        Engine.CommitId commitId() {
            return this.commitId;
        }

        int numDocs() {
            return this.numDocs;
        }

        boolean includeNumDocs(Version version) {
            if (version.major == Version.V_5_6_8_UNRELEASED.major) {
                return version.onOrAfter(Version.V_5_6_8_UNRELEASED);
            }
            return version.id >= 6020299;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.commitId = new Engine.CommitId(in);
            this.numDocs = this.includeNumDocs(in.getVersion()) ? in.readInt() : -1;
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.commitId.writeTo(out);
            if (this.includeNumDocs(out.getVersion())) {
                out.writeInt(this.numDocs);
            }
        }
    }

    public static final class PreShardSyncedFlushRequest
    extends TransportRequest {
        private ShardId shardId;

        public PreShardSyncedFlushRequest() {
        }

        public PreShardSyncedFlushRequest(ShardId shardId) {
            this.shardId = shardId;
        }

        public String toString() {
            return "PreShardSyncedFlushRequest{shardId=" + this.shardId + '}';
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.shardId.writeTo(out);
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardId = ShardId.readShardId(in);
        }

        public ShardId shardId() {
            return this.shardId;
        }
    }
}

