/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetRequest;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.indexedscripts.delete.DeleteIndexedScriptRequest;
import org.elasticsearch.action.indexedscripts.get.GetIndexedScriptRequest;
import org.elasticsearch.action.indexedscripts.put.PutIndexedScriptRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.base.Charsets;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.query.TemplateQueryParser;
import org.elasticsearch.node.settings.NodeSettingsService;
import org.elasticsearch.script.CompiledScript;
import org.elasticsearch.script.ExecutableScript;
import org.elasticsearch.script.ScriptEngineService;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.script.groovy.GroovyScriptEngineService;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class ScriptService
extends AbstractComponent {
    public static final String DEFAULT_SCRIPTING_LANGUAGE_SETTING = "script.default_lang";
    public static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    public static final String SCRIPT_CACHE_SIZE_SETTING = "script.cache.max_size";
    public static final String SCRIPT_CACHE_EXPIRE_SETTING = "script.cache.expire";
    public static final String DISABLE_DYNAMIC_SCRIPTING_DEFAULT = "sandbox";
    public static final String SCRIPT_INDEX = ".scripts";
    public static final String DEFAULT_LANG = "groovy";
    private final String defaultLang;
    private final ImmutableMap<String, ScriptEngineService> scriptEngines;
    private final ConcurrentMap<String, CompiledScript> staticCache = ConcurrentCollections.newConcurrentMap();
    private final Cache<CacheKey, CompiledScript> cache;
    private final File scriptsDirectory;
    private final FileWatcher fileWatcher;
    private final DynamicScriptDisabling dynamicScriptingDisabled;
    private Client client = null;
    public static final ParseField SCRIPT_LANG = new ParseField("lang", "script_lang");
    public static final ParseField SCRIPT_FILE = new ParseField("script_file", "file");
    public static final ParseField SCRIPT_ID = new ParseField("script_id", "id");
    public static final ParseField SCRIPT_INLINE = new ParseField("script", "scriptField");
    public static final ParseField VALUE_SCRIPT_FILE = new ParseField("value_script_file", new String[0]);
    public static final ParseField VALUE_SCRIPT_ID = new ParseField("value_script_id", new String[0]);
    public static final ParseField VALUE_SCRIPT_INLINE = new ParseField("value_script", new String[0]);
    public static final ParseField KEY_SCRIPT_FILE = new ParseField("key_script_file", new String[0]);
    public static final ParseField KEY_SCRIPT_ID = new ParseField("key_script_id", new String[0]);
    public static final ParseField KEY_SCRIPT_INLINE = new ParseField("key_script", new String[0]);

    @Inject
    public ScriptService(Settings settings, Environment env, Set<ScriptEngineService> scriptEngines, ResourceWatcherService resourceWatcherService, NodeSettingsService nodeSettingsService) throws IOException {
        super(settings);
        int cacheMaxSize = settings.getAsInt(SCRIPT_CACHE_SIZE_SETTING, (Integer)100);
        TimeValue cacheExpire = settings.getAsTime(SCRIPT_CACHE_EXPIRE_SETTING, null);
        this.logger.debug("using script cache with max_size [{}], expire [{}]", cacheMaxSize, cacheExpire);
        this.defaultLang = settings.get(DEFAULT_SCRIPTING_LANGUAGE_SETTING, DEFAULT_LANG);
        this.dynamicScriptingDisabled = DynamicScriptDisabling.parse(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING, DISABLE_DYNAMIC_SCRIPTING_DEFAULT));
        CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.maximumSize(cacheMaxSize);
        }
        if (cacheExpire != null) {
            cacheBuilder.expireAfterAccess(cacheExpire.nanos(), TimeUnit.NANOSECONDS);
        }
        cacheBuilder.removalListener(new ScriptCacheRemovalListener());
        this.cache = cacheBuilder.build();
        ImmutableMap.Builder<String, ScriptEngineService> builder = ImmutableMap.builder();
        for (ScriptEngineService scriptEngine : scriptEngines) {
            for (String type : scriptEngine.types()) {
                builder.put(type, scriptEngine);
            }
        }
        this.scriptEngines = builder.build();
        this.scriptsDirectory = new File(env.configFile(), "scripts");
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Using scripts directory [{}] ", this.scriptsDirectory);
        }
        this.fileWatcher = new FileWatcher(this.scriptsDirectory);
        this.fileWatcher.addListener(new ScriptChangesListener());
        if (this.componentSettings.getAsBoolean("auto_reload_enabled", (Boolean)true).booleanValue()) {
            resourceWatcherService.add(this.fileWatcher);
        } else {
            this.fileWatcher.init();
        }
        nodeSettingsService.addListener(new ApplySettings());
    }

    @Inject(optional=true)
    public void setClient(Client client) {
        this.client = client;
    }

    public void close() {
        for (ScriptEngineService engineService : this.scriptEngines.values()) {
            engineService.close();
        }
    }

    public CompiledScript compile(String script) {
        return this.compile(this.defaultLang, script);
    }

    public CompiledScript compile(String lang, String script) {
        return this.compile(lang, script, ScriptType.INLINE);
    }

    public void clearCache() {
        this.logger.debug("clearing script cache", new Object[0]);
        this.cache.invalidateAll();
        this.cache.cleanUp();
        this.staticCache.clear();
        this.fileWatcher.clearState();
    }

    public CompiledScript compile(String lang, String script, ScriptType scriptType) {
        CompiledScript compiled;
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Compiling lang: [{}] type: [{}] script: {}", new Object[]{lang, scriptType, script});
        }
        if (lang == null) {
            lang = this.defaultLang;
        }
        if (scriptType == ScriptType.INDEXED) {
            if (this.client == null) {
                throw new ElasticsearchIllegalArgumentException("Got an indexed script with no Client registered.");
            }
            IndexedScript indexedScript = new IndexedScript(lang, script);
            this.verifyDynamicScripting(indexedScript.lang);
            script = this.getScriptFromIndex(this.client, indexedScript.lang, indexedScript.id);
        } else if (scriptType == ScriptType.FILE) {
            CompiledScript compiled2 = (CompiledScript)this.staticCache.get(script);
            if (compiled2 != null) {
                return compiled2;
            }
            throw new ElasticsearchIllegalArgumentException("Unable to find on disk script " + script);
        }
        if (scriptType != ScriptType.INDEXED && (compiled = (CompiledScript)this.staticCache.get(script)) != null) {
            return compiled;
        }
        this.verifyDynamicScripting(lang);
        CacheKey cacheKey = new CacheKey(lang, script);
        compiled = this.cache.getIfPresent(cacheKey);
        if (compiled != null) {
            return compiled;
        }
        if (!this.dynamicScriptEnabled(lang)) {
            throw new ScriptException("dynamic scripting for [" + lang + "] disabled");
        }
        compiled = this.getCompiledScript(lang, script);
        this.cache.put(cacheKey, compiled);
        return compiled;
    }

    private CompiledScript getCompiledScript(String lang, String script) {
        ScriptEngineService service = this.scriptEngines.get(lang);
        if (service == null) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        CompiledScript compiled = new CompiledScript(lang, service.compile(script));
        return compiled;
    }

    private void verifyDynamicScripting(String lang) {
        if (!this.dynamicScriptEnabled(lang)) {
            throw new ScriptException("dynamic scripting for [" + lang + "] disabled");
        }
    }

    public void queryScriptIndex(GetIndexedScriptRequest request, ActionListener<GetResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        GetRequest getRequest = new GetRequest(request, SCRIPT_INDEX).type(scriptLang).id(request.id()).version(request.version()).versionType(request.versionType()).preference("_local");
        this.client.get(getRequest, listener);
    }

    private String validateScriptLanguage(String scriptLang) {
        if (scriptLang == null) {
            scriptLang = this.defaultLang;
        } else if (!this.scriptEngines.containsKey(scriptLang)) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + scriptLang + "]");
        }
        return scriptLang;
    }

    private String getScriptFromIndex(Client client, String scriptLang, String id) {
        GetRequest getRequest = new GetRequest(SCRIPT_INDEX, scriptLang = this.validateScriptLanguage(scriptLang), id);
        GetResponse responseFields = client.get(getRequest).actionGet();
        if (responseFields.isExists()) {
            return ScriptService.getScriptFromResponse(responseFields);
        }
        throw new ElasticsearchIllegalArgumentException("Unable to find script [.scripts/" + scriptLang + "/" + id + "]");
    }

    private void validate(BytesReference scriptBytes, String scriptLang) {
        block6: {
            try {
                XContentParser parser = XContentFactory.xContent(scriptBytes).createParser(scriptBytes);
                TemplateQueryParser.TemplateContext context = TemplateQueryParser.parse(parser, "params", "script", "template");
                if (Strings.hasLength(context.template())) {
                    try {
                        CompiledScript compiledScript = this.compile(scriptLang, context.template(), ScriptType.INLINE);
                        if (compiledScript == null) {
                            throw new ElasticsearchIllegalArgumentException("Unable to parse [" + context.template() + "] lang [" + scriptLang + "] (ScriptService.compile returned null)");
                        }
                        break block6;
                    }
                    catch (Exception e) {
                        throw new ElasticsearchIllegalArgumentException("Unable to parse [" + context.template() + "] lang [" + scriptLang + "]", e);
                    }
                }
                throw new ElasticsearchIllegalArgumentException("Unable to find script in : " + scriptBytes.toUtf8());
            }
            catch (IOException e) {
                throw new ElasticsearchIllegalArgumentException("failed to parse template script", e);
            }
        }
    }

    public void putScriptToIndex(PutIndexedScriptRequest request, ActionListener<IndexResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        this.validate(request.safeSource(), scriptLang);
        IndexRequest indexRequest = ((IndexRequest)new IndexRequest(request).index(SCRIPT_INDEX)).type(scriptLang).id(request.id()).version(request.version()).versionType(request.versionType()).source(request.safeSource(), true).opType(request.opType()).refresh(true);
        this.client.index(indexRequest, listener);
    }

    public void deleteScriptFromIndex(DeleteIndexedScriptRequest request, ActionListener<DeleteResponse> listener) {
        String scriptLang = this.validateScriptLanguage(request.scriptLang());
        DeleteRequest deleteRequest = ((DeleteRequest)new DeleteRequest(request).index(SCRIPT_INDEX)).type(scriptLang).id(request.id()).refresh(true).version(request.version()).versionType(request.versionType());
        this.client.delete(deleteRequest, listener);
    }

    public static String getScriptFromResponse(GetResponse responseFields) {
        Map<String, Object> source = responseFields.getSourceAsMap();
        if (source.containsKey("template")) {
            try {
                XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
                Object template = source.get("template");
                if (template instanceof Map) {
                    builder.map((Map)template);
                    return builder.string();
                }
                return template.toString();
            }
            catch (IOException | ClassCastException e) {
                throw new ElasticsearchIllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
            }
        }
        if (source.containsKey("script")) {
            return source.get("script").toString();
        }
        try {
            XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
            builder.map(responseFields.getSource());
            return builder.string();
        }
        catch (IOException | ClassCastException e) {
            throw new ElasticsearchIllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
        }
    }

    public ExecutableScript executable(String lang, String script, ScriptType scriptType, Map vars) {
        return this.executable(this.compile(lang, script, scriptType), vars);
    }

    public ExecutableScript executable(CompiledScript compiledScript, Map vars) {
        return this.scriptEngines.get(compiledScript.lang()).executable(compiledScript.compiled(), vars);
    }

    public SearchScript search(CompiledScript compiledScript, SearchLookup lookup, @Nullable Map<String, Object> vars) {
        return this.scriptEngines.get(compiledScript.lang()).search(compiledScript.compiled(), lookup, vars);
    }

    public SearchScript search(SearchLookup lookup, String lang, String script, ScriptType scriptType, @Nullable Map<String, Object> vars) {
        return this.search(this.compile(lang, script, scriptType), lookup, vars);
    }

    private boolean dynamicScriptEnabled(String lang) {
        ScriptEngineService service = this.scriptEngines.get(lang);
        if (service == null) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        if (this.dynamicScriptingDisabled == DynamicScriptDisabling.EVERYTHING_ALLOWED || "native".equals(lang) || "mustache".equals(lang)) {
            return true;
        }
        if (this.dynamicScriptingDisabled == DynamicScriptDisabling.ONLY_DISK_ALLOWED) {
            return false;
        }
        return service.sandboxed();
    }

    public static final class CacheKey {
        public final String lang;
        public final String script;

        public CacheKey(String lang, String script) {
            this.lang = lang;
            this.script = script;
        }

        public boolean equals(Object o) {
            if (!(o instanceof CacheKey)) {
                return false;
            }
            CacheKey other = (CacheKey)o;
            return this.lang.equals(other.lang) && this.script.equals(other.script);
        }

        public int hashCode() {
            return this.lang.hashCode() + 31 * this.script.hashCode();
        }
    }

    private class ScriptChangesListener
    extends FileChangesListener {
        private ScriptChangesListener() {
        }

        private Tuple<String, String> scriptNameExt(File file) {
            String scriptPath = ScriptService.this.scriptsDirectory.toURI().relativize(file.toURI()).getPath();
            int extIndex = scriptPath.lastIndexOf(46);
            if (extIndex != -1) {
                String ext = scriptPath.substring(extIndex + 1);
                String scriptName = scriptPath.substring(0, extIndex).replace(File.separatorChar, '_');
                return new Tuple<String, String>(scriptName, ext);
            }
            return null;
        }

        @Override
        public void onFileInit(File file) {
            Tuple<String, String> scriptNameExt;
            if (ScriptService.this.logger.isTraceEnabled()) {
                ScriptService.this.logger.trace("Loading script file : [{}]", file);
            }
            if ((scriptNameExt = this.scriptNameExt(file)) != null) {
                boolean found = false;
                for (ScriptEngineService engineService : ScriptService.this.scriptEngines.values()) {
                    for (String s : engineService.extensions()) {
                        if (!s.equals(scriptNameExt.v2())) continue;
                        found = true;
                        try {
                            ScriptService.this.logger.info("compiling script file [{}]", file.getAbsolutePath());
                            String script = Streams.copyToString(new InputStreamReader((InputStream)new FileInputStream(file), Charsets.UTF_8));
                            ScriptService.this.staticCache.put(scriptNameExt.v1(), new CompiledScript(engineService.types()[0], engineService.compile(script)));
                        }
                        catch (Throwable e) {
                            ScriptService.this.logger.warn("failed to load/compile script [{}]", e, scriptNameExt.v1());
                        }
                        break;
                    }
                    if (!found) continue;
                    break;
                }
                if (!found) {
                    ScriptService.this.logger.warn("no script engine found for [{}]", scriptNameExt.v2());
                }
            }
        }

        @Override
        public void onFileCreated(File file) {
            this.onFileInit(file);
        }

        @Override
        public void onFileDeleted(File file) {
            Tuple<String, String> scriptNameExt = this.scriptNameExt(file);
            if (scriptNameExt != null) {
                ScriptService.this.logger.info("removing script file [{}]", file.getAbsolutePath());
                ScriptService.this.staticCache.remove(scriptNameExt.v1());
            }
        }

        @Override
        public void onFileChanged(File file) {
            this.onFileInit(file);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, CompiledScript> {
        private ScriptCacheRemovalListener() {
        }

        @Override
        public void onRemoval(RemovalNotification<CacheKey, CompiledScript> notification) {
            if (ScriptService.this.logger.isDebugEnabled()) {
                ScriptService.this.logger.debug("notifying script services of script removal due to: [{}]", new Object[]{notification.getCause()});
            }
            for (ScriptEngineService service : ScriptService.this.scriptEngines.values()) {
                try {
                    service.scriptRemoved(notification.getValue());
                }
                catch (Exception e) {
                    ScriptService.this.logger.warn("exception calling script removal listener for script service", e, new Object[0]);
                }
            }
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            String[] patches;
            boolean blacklistChanged;
            GroovyScriptEngineService engine = (GroovyScriptEngineService)ScriptService.this.scriptEngines.get(ScriptService.DEFAULT_LANG);
            if (engine != null && (blacklistChanged = engine.addToBlacklist(patches = settings.getAsArray(GroovyScriptEngineService.GROOVY_SCRIPT_BLACKLIST_PATCH, Strings.EMPTY_ARRAY)))) {
                ScriptService.this.logger.info("adding {} to [{}], new blacklisted methods: {}", patches, GroovyScriptEngineService.GROOVY_SCRIPT_BLACKLIST_PATCH, engine.blacklistAdditions());
                engine.reloadConfig();
                ScriptService.this.clearCache();
            }
        }
    }

    static class IndexedScript {
        private final String lang;
        private final String id;

        IndexedScript(String lang, String script) {
            this.lang = lang;
            String[] parts = script.split("/");
            if (parts.length == 1) {
                this.id = script;
            } else {
                if (parts.length != 3) {
                    throw new ElasticsearchIllegalArgumentException("Illegal index script format [" + script + "]" + " should be /lang/id");
                }
                if (!parts[1].equals(this.lang)) {
                    throw new ElasticsearchIllegalStateException("Conflicting script language, found [" + parts[1] + "] expected + [" + this.lang + "]");
                }
                this.id = parts[2];
            }
        }
    }

    public static enum ScriptType {
        INLINE,
        INDEXED,
        FILE;

        private static final int INLINE_VAL = 0;
        private static final int INDEXED_VAL = 1;
        private static final int FILE_VAL = 2;

        public static ScriptType readFrom(StreamInput in) throws IOException {
            int scriptTypeVal = in.readVInt();
            switch (scriptTypeVal) {
                case 1: {
                    return INDEXED;
                }
                case 0: {
                    return INLINE;
                }
                case 2: {
                    return FILE;
                }
            }
            throw new ElasticsearchIllegalArgumentException("Unexpected value read for ScriptType got [" + scriptTypeVal + "] expected one of [" + 0 + "," + 1 + "," + 2 + "]");
        }

        public static void writeTo(ScriptType scriptType, StreamOutput out) throws IOException {
            if (scriptType != null) {
                switch (scriptType) {
                    case INDEXED: {
                        out.writeVInt(1);
                        return;
                    }
                    case INLINE: {
                        out.writeVInt(0);
                        return;
                    }
                    case FILE: {
                        out.writeVInt(2);
                        return;
                    }
                }
                throw new ElasticsearchIllegalStateException("Unknown ScriptType " + (Object)((Object)scriptType));
            }
            out.writeVInt(0);
        }
    }

    static enum DynamicScriptDisabling {
        EVERYTHING_ALLOWED,
        ONLY_DISK_ALLOWED,
        SANDBOXED_ONLY;


        static DynamicScriptDisabling parse(String s) {
            switch (s.toLowerCase(Locale.ROOT)) {
                case "true": 
                case "all": {
                    return ONLY_DISK_ALLOWED;
                }
                case "false": 
                case "none": {
                    return EVERYTHING_ALLOWED;
                }
                case "sandbox": 
                case "sandboxed": {
                    return SANDBOXED_ONLY;
                }
            }
            throw new ElasticsearchIllegalArgumentException("Unrecognized script allowance setting: [" + s + "]");
        }
    }
}

