/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

public class EquiDepthStreamHistogram {
    private static final Log LOG = LogFactory.getLog(EquiDepthStreamHistogram.class);
    private static final double MAX_COEF = 1.7;
    private static final short DEFAULT_EXPANSION_FACTOR = 7;
    private int numBuckets;
    private int maxBars;
    @VisibleForTesting
    long totalCount;
    @VisibleForTesting
    List<Bar> bars;

    public EquiDepthStreamHistogram(int numBuckets) {
        this(numBuckets, 7);
    }

    public EquiDepthStreamHistogram(int numBuckets, int expansionFactor) {
        this.numBuckets = numBuckets;
        this.maxBars = numBuckets * expansionFactor;
        this.bars = new ArrayList<Bar>(this.maxBars);
    }

    public void addValue(byte[] value) {
        Bar bar = this.getBar(value);
        bar.incrementCount();
        ++this.totalCount;
        if (bar.getSize() > this.getMaxBarSize()) {
            this.splitBar(bar);
        }
    }

    public List<Bucket> computeBuckets() {
        Preconditions.checkState((this.bars.size() >= this.numBuckets ? 1 : 0) != 0, (Object)"Not enough data points to compute buckets");
        ArrayList<Bucket> buckets = new ArrayList<Bucket>();
        long idealBuckSize = (long)Math.ceil((double)this.totalCount / (double)this.numBuckets);
        long currCount = 0L;
        int barsIdx = 0;
        byte[] prevBound = this.bars.get((int)0).leftBoundInclusive;
        Bar currBar = null;
        for (int i = 0; i < this.numBuckets; ++i) {
            while (currCount <= idealBuckSize && barsIdx < this.bars.size()) {
                currBar = this.bars.get(barsIdx++);
                currCount += currBar.getSize();
            }
            long surplus = Math.max(currCount - idealBuckSize, 0L);
            int closestSplitIdx = (int)((1.0 - (double)surplus / (double)currBar.getSize()) * 9.0);
            byte[][] splits = Bytes.split((byte[])currBar.leftBoundInclusive, (byte[])currBar.rightBoundExclusive, (int)8);
            Bucket bucket = new Bucket(prevBound, splits[closestSplitIdx]);
            bucket.incrementCountEstimate(currCount - surplus);
            prevBound = splits[closestSplitIdx];
            buckets.add(bucket);
            currCount = surplus;
        }
        return buckets;
    }

    public long getTotalCount() {
        return this.totalCount;
    }

    @VisibleForTesting
    void splitBar(Bar origBar) {
        Bar smallerBar;
        boolean mergeSuccessful;
        if (Bytes.compareTo((byte[])origBar.leftBoundInclusive, (byte[])origBar.rightBoundExclusive) == 0) {
            return;
        }
        if (this.bars.size() == this.maxBars && !(mergeSuccessful = this.mergeBars())) {
            return;
        }
        byte[] mid = Bytes.split((byte[])origBar.getLeftBoundInclusive(), (byte[])origBar.getRightBoundExclusive(), (int)1)[1];
        Bar newLeft = new Bar(origBar.getLeftBoundInclusive(), mid);
        Bar newRight = new Bar(mid, origBar.getRightBoundExclusive());
        long leftSize = 0L;
        long bbAggCount = origBar.getBlockedBarsSize();
        for (Bar bb : origBar.getBlockedBars()) {
            long bbSize = bb.getSize();
            if (leftSize + bbSize < bbAggCount / 2L) {
                leftSize += bbSize;
                newLeft.addBlockedBar(bb);
                continue;
            }
            newRight.addBlockedBar(bb);
        }
        long countToDistribute = origBar.getSize() - bbAggCount;
        long rightSize = newRight.getSize();
        long sizeDiff = Math.abs(leftSize - rightSize);
        Bar bar = smallerBar = leftSize <= rightSize ? newLeft : newRight;
        if (sizeDiff <= countToDistribute) {
            smallerBar.incrementCount(sizeDiff);
            long halfDistrib = (countToDistribute -= sizeDiff) / 2L;
            newLeft.incrementCount(halfDistrib);
            newRight.incrementCount(countToDistribute - halfDistrib);
        } else {
            smallerBar.incrementCount(countToDistribute);
        }
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Split orig=%s , newLeft=%s , newRight=%s", origBar, newLeft, newRight));
        }
        this.bars.remove(origBar);
        this.bars.add(newLeft);
        this.bars.add(newRight);
        Collections.sort(this.bars);
    }

    @VisibleForTesting
    boolean mergeBars() {
        Preconditions.checkState((this.bars.size() > 1 ? 1 : 0) != 0, (Object)"Need at least two bars to merge");
        int currIdx = 0;
        Bar currBar = this.bars.get(currIdx);
        Bar nextBar = this.bars.get(currIdx + 1);
        long currMinSum = Long.MAX_VALUE;
        int currMinIdx = currIdx;
        Pair minBars = new Pair((Object)currBar, (Object)nextBar);
        while (nextBar != null) {
            long sum = currBar.getSize() + nextBar.getSize();
            if (sum < currMinSum) {
                currMinSum = sum;
                minBars = new Pair((Object)currBar, (Object)nextBar);
                currMinIdx = currIdx;
            }
            currBar = nextBar;
            nextBar = ++currIdx < this.bars.size() - 1 ? this.bars.get(currIdx + 1) : null;
        }
        if (currMinSum >= this.getMaxBarSize()) {
            return false;
        }
        Bar leftBar = (Bar)minBars.getFirst();
        Bar rightBar = (Bar)minBars.getSecond();
        Bar newBar = new Bar(leftBar.getLeftBoundInclusive(), rightBar.getRightBoundExclusive());
        if (leftBar.getSize() >= rightBar.getSize()) {
            newBar.incrementCount(rightBar.getCount());
            newBar.addBlockedBar(new Bar(leftBar));
        } else {
            newBar.incrementCount(leftBar.getCount());
            newBar.addBlockedBar(new Bar(rightBar));
        }
        newBar.addBlockedBars(leftBar.getBlockedBars());
        newBar.addBlockedBars(rightBar.getBlockedBars());
        this.bars.subList(currMinIdx, currMinIdx + 2).clear();
        this.bars.add(newBar);
        Collections.sort(this.bars);
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)String.format("Merged left=%s , right=%s , newBar=%s", leftBar, rightBar, newBar));
        }
        return true;
    }

    @VisibleForTesting
    Bar getBar(byte[] value) {
        Bar searchKey = new Bar(value, value);
        int searchIdx = Collections.binarySearch(this.bars, searchKey);
        if (searchIdx < 0) {
            byte[] newBound = Bytes.copy((byte[])value);
            if (this.bars.size() == 0) {
                Bar firstBar = new Bar(newBound, newBound);
                this.bars.add(firstBar);
                return firstBar;
            }
            int expectedIndex = Math.abs(searchIdx + 1);
            if (expectedIndex == this.bars.size()) {
                Bar lastBar = this.bars.get(expectedIndex - 1);
                lastBar.setRightBoundExclusive(newBound);
                return lastBar;
            }
            Bar nextBar = this.bars.get(expectedIndex);
            nextBar.setLeftBoundInclusive(newBound);
            return nextBar;
        }
        return this.bars.get(searchIdx);
    }

    private long getMaxBarSize() {
        return (long)(1.7 * (double)(this.totalCount / (long)this.maxBars));
    }

    @VisibleForTesting
    static class Bar
    extends Bucket
    implements Comparable<Bar> {
        private List<Bar> blockedBars = new ArrayList<Bar>();

        public Bar(byte[] leftBoundInclusive, byte[] rightBoundExclusive) {
            super(leftBoundInclusive, rightBoundExclusive);
        }

        public Bar(Bar bar) {
            super(bar.leftBoundInclusive, bar.rightBoundExclusive);
            this.count = bar.count;
        }

        @Override
        public int compareTo(Bar other) {
            int leftComp = Bytes.compareTo((byte[])this.leftBoundInclusive, (byte[])other.leftBoundInclusive);
            int rightComp = Bytes.compareTo((byte[])this.rightBoundExclusive, (byte[])other.rightBoundExclusive);
            if (leftComp >= 0 && rightComp < 0 || leftComp <= 0 && rightComp > 0 || leftComp == 0 && rightComp == 0) {
                return 0;
            }
            if (Bytes.compareTo((byte[])this.leftBoundInclusive, (byte[])other.rightBoundExclusive) >= 0) {
                return 1;
            }
            if (Bytes.compareTo((byte[])this.rightBoundExclusive, (byte[])other.leftBoundInclusive) <= 0) {
                return -1;
            }
            throw new AssertionError((Object)"Cannot not have overlapping bars");
        }

        public long getSize() {
            long blockedBarSum = this.getBlockedBarsSize();
            return this.count + blockedBarSum;
        }

        public long getBlockedBarsSize() {
            long blockedBarSum = 0L;
            for (Bar bb : this.blockedBars) {
                blockedBarSum += bb.getSize();
            }
            return blockedBarSum;
        }

        public void addBlockedBar(Bar bar) {
            this.blockedBars.add(bar);
        }

        public void addBlockedBars(List<Bar> bars) {
            this.blockedBars.addAll(bars);
        }

        public List<Bar> getBlockedBars() {
            return this.blockedBars;
        }

        public long getCount() {
            return this.count;
        }

        public void incrementCount() {
            ++this.count;
        }

        public void incrementCount(long increment) {
            this.count += increment;
        }

        @Override
        public int hashCode() {
            int prime = 31;
            int result = super.hashCode();
            result = 31 * result + (this.blockedBars == null ? 0 : this.blockedBars.hashCode());
            result = 31 * result + (int)(this.count ^ this.count >>> 32);
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!super.equals(obj)) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Bar other = (Bar)obj;
            if (this.blockedBars == null ? other.blockedBars != null : !this.blockedBars.equals(other.blockedBars)) {
                return false;
            }
            return this.count == other.count;
        }

        @Override
        public String toString() {
            return "Bar[count=" + this.count + ", blockedBars=" + this.blockedBars + ", leftBoundInclusive=" + Bytes.toString((byte[])this.leftBoundInclusive) + ", rightBoundExclusive=" + Bytes.toString((byte[])this.rightBoundExclusive) + "]";
        }
    }

    public static class Bucket {
        protected long count = 0L;
        protected byte[] leftBoundInclusive;
        protected byte[] rightBoundExclusive;

        public Bucket(byte[] leftBoundInclusive, byte[] rightBoundExclusive) {
            this.leftBoundInclusive = leftBoundInclusive;
            this.rightBoundExclusive = rightBoundExclusive;
        }

        public byte[] getLeftBoundInclusive() {
            return this.leftBoundInclusive;
        }

        public void setLeftBoundInclusive(byte[] leftBoundInclusive) {
            this.leftBoundInclusive = leftBoundInclusive;
        }

        public byte[] getRightBoundExclusive() {
            return this.rightBoundExclusive;
        }

        public void setRightBoundExclusive(byte[] rightBoundExclusive) {
            this.rightBoundExclusive = rightBoundExclusive;
        }

        public long getCountEstimate() {
            return this.count;
        }

        public void incrementCountEstimate(long count) {
            this.count += count;
        }

        public int hashCode() {
            int prime = 31;
            int result = 1;
            result = 31 * result + Arrays.hashCode(this.leftBoundInclusive);
            result = 31 * result + Arrays.hashCode(this.rightBoundExclusive);
            return result;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj == null) {
                return false;
            }
            if (this.getClass() != obj.getClass()) {
                return false;
            }
            Bucket other = (Bucket)obj;
            if (!Arrays.equals(this.leftBoundInclusive, other.leftBoundInclusive)) {
                return false;
            }
            return Arrays.equals(this.rightBoundExclusive, other.rightBoundExclusive);
        }

        public String toString() {
            return "Bucket [count=" + this.count + ", leftBoundInclusive=" + Bytes.toString((byte[])this.leftBoundInclusive) + ", rightBoundExclusive=" + Bytes.toString((byte[])this.rightBoundExclusive) + "]";
        }
    }
}

