/*
 * Copyright (C) 2015 Tatsuhiro Tsujikawa
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package net.sourceforge.argparse4j.impl.type;

import net.sourceforge.argparse4j.helper.TextHelper;
import net.sourceforge.argparse4j.inf.Argument;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.ArgumentParserException;
import net.sourceforge.argparse4j.inf.ArgumentType;
import net.sourceforge.argparse4j.inf.MetavarInference;

/**
 * <p>
 * ArgumentType subclass for enum type with case-insensitive match.
 * </p>
 * <p>
 * If enum type is passed to {@link Argument#type(Class)}, string comparison is
 * performed in case-sensitive (this is default for all other type conversions).
 * Usually in convention, enum name is all upper case letters, some times you
 * want to do case-insensitive match for convenience. This class does that job.
 * </p>
 * 
 * @param <T>
 *            Type of enum
 */
public class CaseInsensitiveEnumArgumentType<T extends Enum<T>> implements
        ArgumentType<T>, MetavarInference {

    private Class<T> type_;

    public CaseInsensitiveEnumArgumentType(Class<T> type) {
        type_ = type;
    }

    @Override
    public T convert(ArgumentParser parser, Argument arg, String value)
            throws ArgumentParserException {
        for (T t : type_.getEnumConstants()) {
            if (t.name().toLowerCase().equals(value.toLowerCase())) {
                return t;
            }
        }

        String choices = TextHelper.concat(type_.getEnumConstants(), 0, ",",
                "{", "}");
        throw new ArgumentParserException(String.format(TextHelper.LOCALE_ROOT,
                "could not convert '%s' (choose from %s)", value, choices),
                parser, arg);
    }

    @Override
    public String[] inferMetavar() {
        return new String[] { TextHelper.concat(type_.getEnumConstants(), 0,
                ",", "{", "}") };
    }

}
