/*
 * Copyright (c) 2008-2025, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.hazelcast.jet.impl.client.protocol.task;

import com.hazelcast.client.impl.protocol.ClientMessage;
import com.hazelcast.client.impl.protocol.codec.JetSubmitJobCodec;
import com.hazelcast.instance.impl.Node;
import com.hazelcast.internal.nio.Connection;
import com.hazelcast.jet.config.JobConfig;
import com.hazelcast.jet.impl.operation.SubmitJobOperation;
import com.hazelcast.security.permission.ActionConstants;
import com.hazelcast.security.permission.UserCodeNamespacePermission;
import com.hazelcast.spi.impl.operationservice.Operation;

import javax.annotation.Nullable;
import java.security.Permission;
import java.util.UUID;

public class JetSubmitJobMessageTask extends AbstractJetMessageTask<JetSubmitJobCodec.RequestParameters, Void> {
    private transient JobConfig deserializedJobConfig;

    protected JetSubmitJobMessageTask(ClientMessage clientMessage, Node node, Connection connection) {
        super(clientMessage, node, connection, JetSubmitJobCodec::decodeRequest,
                o -> JetSubmitJobCodec.encodeResponse());
    }

    @Override
    protected UUID getLightJobCoordinator() {
        return parameters.lightJobCoordinator;
    }

    @Override
    protected Operation prepareOperation() {
        return new SubmitJobOperation(parameters.jobId, null, deserializedJobConfig, parameters.dag, parameters.jobConfig,
                parameters.lightJobCoordinator != null, endpoint.getSubject());
    }

    @Override
    public String getMethodName() {
        return "submitJob";
    }

    @Override
    public Object[] getParameters() {
        return new Object[]{};
    }

    @Nullable
    @Override
    public Permission getUserCodeNamespacePermission() {
        // we need to deserialize the JobConfig to read it, so save it for prepareOperation() later too
        if (deserializedJobConfig == null) {
            deserializedJobConfig = nodeEngine.getSerializationService().toObject(parameters.jobConfig);
        }
        String userCodeNamespace = deserializedJobConfig.getUserCodeNamespace();
        return userCodeNamespace != null ? new UserCodeNamespacePermission(userCodeNamespace, ActionConstants.ACTION_USE) : null;
    }

    @Nullable
    @Override
    public String[] actions() {
        return new String[]{ActionConstants.ACTION_SUBMIT};
    }
}
