/*-
 * #%L
 * JSQLParser library
 * %%
 * Copyright (C) 2004 - 2019 JSQLParser
 * %%
 * Dual licensed under GNU LGPL 2.1 or Apache License 2.0
 * #L%
 */
package net.sf.jsqlparser.statement;

import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.operators.relational.ExpressionList;
import net.sf.jsqlparser.statement.select.PlainSelect;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

public final class SetStatement implements Statement {

    private final List<NameExpr> values = new ArrayList<>();
    private String effectParameter;

    public SetStatement() {
        // empty constructor
    }

    public SetStatement(Object name, ExpressionList<?> value) {
        add(name, value, true);
    }

    public void add(Object name, ExpressionList<?> value, boolean useEqual) {
        values.add(new NameExpr(name, value, useEqual));
    }

    public void remove(int idx) {
        values.remove(idx);
    }

    public int getCount() {
        return values.size();
    }

    public boolean isUseEqual(int idx) {
        return values.get(idx).useEqual;
    }

    public boolean isUseEqual() {
        return isUseEqual(0);
    }

    public SetStatement setUseEqual(boolean useEqual) {
        return setUseEqual(0, useEqual);
    }

    public SetStatement withUseEqual(int idx, boolean useEqual) {
        this.setUseEqual(idx, useEqual);
        return this;
    }

    public SetStatement setUseEqual(int idx, boolean useEqual) {
        values.get(idx).useEqual = useEqual;
        return this;
    }

    public SetStatement withUseEqual(boolean useEqual) {
        this.setUseEqual(useEqual);
        return this;
    }

    public Object getName() {
        return getName(0);
    }

    public void setName(String name) {
        setName(0, name);
    }

    public Object getName(int idx) {
        return values.get(idx).name;
    }

    public void setName(int idx, String name) {
        values.get(idx).name = name;
    }

    public List<Expression> getExpressions(int idx) {
        return values.get(idx).expressions;
    }

    public List<Expression> getExpressions() {
        return getExpressions(0);
    }

    public void setExpressions(ExpressionList<?> expressions) {
        setExpressions(0, expressions);
    }

    public void setExpressions(int idx, ExpressionList<?> expressions) {
        values.get(idx).expressions = expressions;
    }

    private String toString(NameExpr ne) {
        return ne.name + (ne.useEqual ? " = " : " ")
                + PlainSelect.getStringList(ne.expressions, true, false);
    }

    @Override
    public String toString() {
        StringBuilder b = new StringBuilder("SET ");
        if (effectParameter != null) {
            b.append(effectParameter).append(" ");
        }
        boolean addComma = false;
        for (NameExpr ne : values) {
            if (addComma) {
                b.append(", ");
            } else {
                addComma = true;
            }
            b.append(toString(ne));
        }

        return b.toString();
    }

    public List<NameExpr> getKeyValuePairs() {
        return values;
    }

    public void addKeyValuePairs(Collection<NameExpr> keyValuePairs) {
        values.addAll(keyValuePairs);
    }

    public void addKeyValuePairs(NameExpr... keyValuePairs) {
        addKeyValuePairs(Arrays.asList(keyValuePairs));
    }

    public void clear() {
        values.clear();
        effectParameter = null;
    }

    @Override
    public <T, S> T accept(StatementVisitor<T> statementVisitor, S context) {
        return statementVisitor.visit(this, context);
    }

    public String getEffectParameter() {
        return effectParameter;
    }

    public void setEffectParameter(String effectParameter) {
        this.effectParameter = effectParameter;
    }

    public SetStatement withEffectParameter(String effectParameter) {
        this.effectParameter = effectParameter;
        return this;
    }

    static class NameExpr implements Serializable {
        Object name;
        ExpressionList expressions;
        boolean useEqual;

        public NameExpr(Object name, ExpressionList<?> expressions, boolean useEqual) {
            this.name = name;
            this.expressions = expressions;
            this.useEqual = useEqual;
        }

        public Object getName() {
            return name;
        }

        public void setName(Object name) {
            this.name = name;
        }

        public ExpressionList<?> getExpressions() {
            return expressions;
        }

        public void setExpressions(ExpressionList<?> expressions) {
            this.expressions = expressions;
        }

        public boolean isUseEqual() {
            return useEqual;
        }

        public void setUseEqual(boolean useEqual) {
            this.useEqual = useEqual;
        }
    }
}
