package org.violet.common.security.config;

import cn.dev33.satoken.context.SaTokenContext;
import cn.dev33.satoken.exception.SaTokenException;
import cn.dev33.satoken.filter.SaPathCheckFilterForJakartaServlet;
import cn.dev33.satoken.interceptor.SaInterceptor;
import cn.dev33.satoken.stp.StpInterface;
import cn.dev33.satoken.stp.StpUtil;
import cn.hutool.core.collection.CollUtil;
import org.violet.common.launch.config.launch.VioletProperties;
import lombok.AllArgsConstructor;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;
import org.violet.common.security.constants.AuthConst;
import org.violet.common.security.context.VioletSaTokenContextForSpringInJakartaServlet;
import org.violet.common.security.util.StpUserUtil;

import java.util.List;

/***
 * saToken 核心配置
 * @author wjx
 * @date 2024/11/28 13:46
 */
@AutoConfiguration
@AllArgsConstructor
@EnableConfigurationProperties(VioletProperties.class)
public class SaTokenConfiguration implements WebMvcConfigurer {

    private final VioletProperties violetProperties;

    /**
     * 获取上下文处理器组件 (SpringBoot3 Jakarta Servlet 版)
     *
     * @return /
     */
    @Bean
    public SaTokenContext getSaTokenContextForSpringInJakartaServlet() {
        return new VioletSaTokenContextForSpringInJakartaServlet();
    }

    /**
     * 请求 path 校验过滤器
     *
     * @return /
     */
    @Bean
    public SaPathCheckFilterForJakartaServlet saPathCheckFilterForJakartaServlet() {
        return new SaPathCheckFilterForJakartaServlet();
    }

    /**
     * 获取角色、权限，可自定义覆盖
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    public StpInterface stpInterface() {
        return new StpInterfaceImpl();
    }

    /**
     * 配置satoken拦截器
     *
     * @param registry
     */
    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        List<String> excludePath = CollUtil.newArrayList(AuthConst.EXCLUDE_PATH);
        if (CollUtil.isNotEmpty(this.violetProperties.getWhiteList())) {
            CollUtil.addAll(excludePath, this.violetProperties.getWhiteList());
        }

        registry.addInterceptor(new SaInterceptor((handle) -> {
            if (!StpUtil.isLogin() && !StpUserUtil.isLogin()) {
                throw new SaTokenException("用户未登录");
            }
        })).addPathPatterns(new String[]{"/**"}).excludePathPatterns(excludePath);
    }
}
