package org.violet.commom.crypto.advice;

import lombok.RequiredArgsConstructor;

import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.support.WebDataBinderFactory;
import org.springframework.web.context.request.NativeWebRequest;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.method.support.ModelAndViewContainer;
import org.violet.commom.crypto.annotation.decrypt.ApiDecrypt;
import org.violet.commom.crypto.bean.CryptoInfoBean;
import org.violet.commom.crypto.config.ApiCryptoProperties;
import org.violet.commom.crypto.util.ApiCryptoUtil;
import org.violet.common.core.util.Charsets;
import org.violet.common.core.util.JsonUtil;
import org.violet.common.core.util.StringUtil;

import java.lang.reflect.Parameter;

/**
 * param 参数 解析,自动将参数名为data的数据解密后绑定到指定参数上面去
 */
@RequiredArgsConstructor
public class ApiDecryptParamResolver implements HandlerMethodArgumentResolver {
    private final ApiCryptoProperties properties;

    @Override
    public boolean supportsParameter(MethodParameter parameter) {
        return AnnotatedElementUtils.hasAnnotation(parameter.getParameter(), ApiDecrypt.class);
    }

    @Nullable
    @Override
    public Object resolveArgument(MethodParameter methodParameter, ModelAndViewContainer mavContainer,
                                  NativeWebRequest webRequest, WebDataBinderFactory binderFactory) {
        Parameter parameter = methodParameter.getParameter();
        ApiDecrypt apiDecrypt = AnnotatedElementUtils.getMergedAnnotation(parameter, ApiDecrypt.class);
        String text = webRequest.getParameter(properties.getParamName());
        if (StringUtil.isBlank(text)) {
            return null;
        }
        CryptoInfoBean infoBean = new CryptoInfoBean(apiDecrypt.value(), apiDecrypt.secretKey());
        byte[] textBytes = text.getBytes(Charsets.UTF_8);
        byte[] decryptData = ApiCryptoUtil.decryptData(properties, textBytes, infoBean);
        return JsonUtil.readValue(decryptData, parameter.getType());
    }
}
