package org.violet.common.core.util;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import org.violet.common.core.intefaces.ITreeNode;


import java.util.ArrayList;
import java.util.List;


public class TreeUtil {
    public static <T extends ITreeNode> List<T> buildTree(List<? extends ITreeNode> nodeList) {
        return buildTree(nodeList, false);
    }

    /**
     * 根据节点列表构造树形结构,通过cascade控制是否强制级联,如果为true的情况下 如果节点未找到上级节点则不会加入到树中
     *
     * @param nodeList
     * @param cascade
     * @param <T>
     * @return
     */
    public static <T extends ITreeNode> List<T> buildTree(List<? extends ITreeNode> nodeList, boolean cascade) {
        List<T> tree = new ArrayList<>();
        //先找到所有的根节点,所有找不到父节点的均认为是根节点
        if (CollUtil.isNotEmpty(nodeList)) {
            for (ITreeNode node : nodeList) {
                ITreeNode pnode = CollUtil.findOne(nodeList, n -> ObjectUtil.equals(n.id(), node.pid()));
                if (pnode == null) {
                    //只会分析父节点为null或0的,如果强制级联为true则不会分析找不到父节点的node
                    if (node.pid() == null || node.pid().toString().equals("0") || !cascade) {
                        tree.add((T) node);
                    }
                }
            }
        }
        //根据根节点创建一个待处理列表
        List<T> todoList = CollUtil.newArrayList(tree);
        while (!todoList.isEmpty()) {
            T todoNode = todoList.get(0);
            todoList.remove(0);
            nodeList.forEach(node -> {
                //从原始列表中查找子节点,找到后添加给当前节点作为子节点,同时将该子节点添加到待处理列表中
                //待处理列表会不断的增加减少,等到待处理列表中没有要处理的对象了 代表所有的节点都查找完毕
                if (ObjectUtil.equals(todoNode.id(), node.pid())) {
                    todoNode.addChild(node);
                    todoList.add((T) node);
                }
            });
        }
        return tree;
    }

    /**
     * 从树形节点列表中查找到指定ID的节点
     *
     * @param nodeList
     * @param id
     * @param <T>
     * @return
     */
    public static <T extends ITreeNode> T findOneById(List<? extends ITreeNode> nodeList, Object id) {
        T oneNode = null;
        if (CollUtil.isEmpty(nodeList)) {
            return oneNode;
        } else {
            for (ITreeNode node : nodeList) {
                if (node.id().equals(id)) {
                    oneNode = (T) node;
                } else {
                    if (CollUtil.isNotEmpty(node.getChildren())) {
                        oneNode = TreeUtil.findOneById((List<? extends ITreeNode>) node.getChildren(), id);
                    }
                }
                if (oneNode != null) {
                    return oneNode;
                }
            }
        }
        return oneNode;
    }

    public static <T> List<T> getChildIds(List<? extends ITreeNode> nodeList, T id, boolean includeSelf) {
        List<T> childIds = new ArrayList<>();
        ITreeNode node = findOneById(nodeList, id);
        if (node != null) {
            if (includeSelf) {
                childIds.add(id);
            }
            addChildIds(node.getChildren(), childIds);
        }
        return childIds;
    }

    public static void addChildIds(List<ITreeNode> nodeList, List ids) {
        for (ITreeNode node : nodeList) {
            ids.add(node.id());
            if (CollUtil.isNotEmpty(node.getChildren())) {
                addChildIds(node.getChildren(), ids);
            }
        }
    }

    public static <T extends ITreeNode> List<T> findByPath(List<T> nodeList, Object targetId) {
        List<T> path = new ArrayList<>();
        if (findPath(path, nodeList, targetId)) {
            return path;
        } else {
            return null;
        }
    }


    private static <T extends ITreeNode> boolean findPath(List<T> pathList, List<T> nodeList, Object targetId) {
        if (CollUtil.isEmpty(nodeList)) {
            return false;
        }

        for (T node : nodeList) {
            if (targetId.equals(node.id())) {
                pathList.add(node);
                return true;
            } else {
                if (findPath(pathList, node.getChildren(), targetId)) {
                    pathList.add(node);
                    return true;
                } else {
                    if (CollUtil.isNotEmpty(pathList)) {
                        pathList.remove(pathList.size() - 1);
                    }
                }
            }
        }
        return false;
    }

}
