package org.violet.common.core.filter;

import jakarta.servlet.*;
import jakarta.servlet.http.HttpServletRequest;
import lombok.AllArgsConstructor;
import org.springframework.util.AntPathMatcher;
import org.violet.common.core.config.mvc.RequestProperties;
import org.violet.common.core.config.mvc.VioletHttpServletRequestWrapper;
import org.violet.common.core.xss.XssHttpServletRequestWrapper;
import org.violet.common.core.xss.XssProperties;

import java.io.IOException;

/**
 * Request全局过滤
 *
 */
@AllArgsConstructor
public class VioletRequestFilter implements Filter {

    private final RequestProperties requestProperties;
    private final XssProperties xssProperties;
    private final AntPathMatcher antPathMatcher = new AntPathMatcher();

    @Override
    public void init(FilterConfig config) {

    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        String path = ((HttpServletRequest) request).getServletPath();
        // 跳过 Request 包装
        if (!requestProperties.getEnabled() || isRequestSkip(path)) {
            chain.doFilter(request, response);
        }
        // 默认 Request 包装
        else if (!xssProperties.getEnabled() || isXssSkip(path)) {
            VioletHttpServletRequestWrapper violetRequest = new VioletHttpServletRequestWrapper((HttpServletRequest) request);
            chain.doFilter(violetRequest, response);
        }
        // Xss Request 包装
        else {
            XssHttpServletRequestWrapper xssRequest = new XssHttpServletRequestWrapper((HttpServletRequest) request);
            chain.doFilter(xssRequest, response);
        }
    }

    private boolean isRequestSkip(String path) {
        return requestProperties.getSkipUrl().stream().anyMatch(pattern -> antPathMatcher.match(pattern, path));
    }

    private boolean isXssSkip(String path) {
        return xssProperties.getSkipUrl().stream().anyMatch(pattern -> antPathMatcher.match(pattern, path));
    }

    @Override
    public void destroy() {

    }

}
