package org.violet.common.core.util;


import cn.hutool.core.collection.CollUtil;

import com.alibaba.fastjson2.JSONObject;
import org.violet.common.core.exception.BizException;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 树型数据工具类
 *
 * @author xu.wenchang
 * @version 1.0 2023/07/18
 */
public class TreeBuilder {
    private static final String KEY_LABEL = "key";
    private static final String PARENT_KEY_LABEL = "parentKey";
    private static final String TITLE_LABEL = "title";
    private static final String CHILDREN_LABEL = "children";

    private String keyLabel = KEY_LABEL;
    private String parentKeyLabel = PARENT_KEY_LABEL;
    private String titleLabel = TITLE_LABEL;
    private String childrenLabel = CHILDREN_LABEL;
    private String keyMap;
    private String parentKeyMap;
    private String titleMap;
    private List<?> data;
    private RootNodeParentKeyType rootNodeParentKeyType = RootNodeParentKeyType.ZERO;
    private boolean isNotRewriteAttrNames;

    public static TreeBuilder builder() {
        return new TreeBuilder();
    }

    public List<?> tree() {
        if (this.isNotRewriteAttrNames) {
            this.keyLabel = this.keyMap;
            this.parentKeyLabel = this.parentKeyMap;
            this.titleLabel = this.titleMap;
        }

        if (CollUtil.isEmpty(this.data)) {
            return Collections.emptyList();
        }

        Map<Object, JSONObject> map = new HashMap<>();

        try {
            for (Object obj : data) {
                Field[] fields = obj.getClass()
                        .getDeclaredFields();

                JSONObject item = new JSONObject();

                for (Field field : fields) {
                    if (field.getName()
                            .equals(this.keyMap)) {
                        field.setAccessible(true);
                        item.put(this.keyLabel, field.get(obj));
                    }

                    if (field.getName()
                            .equals(this.parentKeyMap)) {
                        field.setAccessible(true);
                        item.put(this.parentKeyLabel, field.get(obj));
                    }

                    if (field.getName()
                            .equals(this.titleMap)) {
                        field.setAccessible(true);
                        item.put(this.titleLabel, field.get(obj));
                    }
                }

                item.put("data", obj);

                if (item.containsKey(this.keyLabel)) {
                    Object key = item.get(this.keyLabel);

                    if (map.containsKey(key)) {
                        JSONObject exists = map.get(key);
                        item.put(this.childrenLabel, exists.get(this.childrenLabel));
                    }

                    map.put(key, item);

                    Object parentKey = item.get(this.parentKeyLabel);

                    if (!this.isRootNode(parentKey)) {
                        if (!map.containsKey(parentKey)) {
                            map.put(parentKey, new JSONObject());
                        }

                        JSONObject parent = map.get(parentKey);

                        if (!parent.containsKey(this.childrenLabel)) {
                            parent.put(this.childrenLabel, new ArrayList<JSONObject>());
                        }

                        @SuppressWarnings("unchecked") List<JSONObject> children = (List<JSONObject>) parent.get(this.childrenLabel);
                        children.add(item);
                    }
                }
            }

            return map.values()
                    .stream()
                    .filter(jsonObject -> !map.containsKey(jsonObject.get(this.parentKeyLabel)))
                    .collect(Collectors.toList());
        } catch (Exception e) {
            throw new BizException(e.getMessage(), e);
        }
    }

    public TreeBuilder useData(List<?> data) {
        this.data = data;
        return this;
    }

    public TreeBuilder keyMap(String keyMap) {
        this.keyMap = keyMap;
        return this;
    }

    public TreeBuilder parentKeyMap(String parentKeyMap) {
        this.parentKeyMap = parentKeyMap;
        return this;
    }

    public TreeBuilder titleMap(String titleMap) {
        this.titleMap = titleMap;
        return this;
    }

    public TreeBuilder keyLabel(String keyLabel) {
        this.keyLabel = keyLabel;
        return this;
    }

    public TreeBuilder parentKeyLabel(String parentKeyLabel) {
        this.parentKeyLabel = parentKeyLabel;
        return this;
    }

    public TreeBuilder titleLabel(String titleLabel) {
        this.titleLabel = titleLabel;
        return this;
    }

    public TreeBuilder childrenLabel(String childrenLabel) {
        this.childrenLabel = childrenLabel;
        return this;
    }

    public TreeBuilder rootNodeParentKeyType(RootNodeParentKeyType type) {
        this.rootNodeParentKeyType = type;
        return this;
    }

    public TreeBuilder notRewriteAttrName() {
        this.isNotRewriteAttrNames = true;
        return this;
    }

    private boolean isRootNode(Object parentValue) {
        if (null == parentValue) {
            return true;
        }

        if (this.rootNodeParentKeyType == RootNodeParentKeyType.ZERO) {
            return Long.parseLong(parentValue.toString()) == 0L;
        }

        return false;
    }

    public enum RootNodeParentKeyType {
        NULL,
        ZERO
    }
}
