package org.violet.common.cloud.version;

import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.lang.Nullable;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.RequestMappingInfo;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;
import org.springframework.web.util.pattern.PathPatternParser;
import org.violet.common.cloud.annotation.ApiVersion;
import org.violet.common.cloud.annotation.UrlVersion;
import org.violet.common.core.util.StringPool;
import org.violet.common.core.util.StringUtil;

import java.lang.reflect.Method;
import java.util.Map;

/**
 * 扩展mvc中的RequestMappingHandlerMapping，使其可以支持api多版本
 *
 * <p>
 * url: /v1/user/{id}  拼接url
 * header: Accept application/vnd.violet.VERSION+json 自定义媒体请求头
 * </p>
 * 上述两种方式只能使用一种,且均优先加载加在方法上面的注解值
 */
public class VioletRequestMappingHandlerMapping extends RequestMappingHandlerMapping {
    private RequestMappingInfo.BuilderConfiguration config = new RequestMappingInfo.BuilderConfiguration();

    /**
     * 拼接URI
     *
     * @param method
     * @param handlerType
     * @return
     */
    @Nullable
    @Override
    protected RequestMappingInfo getMappingForMethod(Method method, Class<?> handlerType) {
        RequestMappingInfo mappinginfo = super.getMappingForMethod(method, handlerType);
        if (mappinginfo != null) {
            RequestMappingInfo apiVersionMappingInfo = getApiVersionMappingInfo(method, handlerType);
            return apiVersionMappingInfo == null ? mappinginfo : apiVersionMappingInfo.combine(mappinginfo);
        }
        return null;
    }

    /**
     * 获取当前的api版本信息，一个是对于url，一个是针对移动端app
     *
     * @param method
     * @param handlerType
     * @return
     */
    @Nullable
    private RequestMappingInfo getApiVersionMappingInfo(Method method, Class<?> handlerType) {
        // url 上的版本，优先获取方法上的版本
        UrlVersion urlVersion = AnnotatedElementUtils.findMergedAnnotation(method, UrlVersion.class);
        // 再次尝试类上的版本
        if (urlVersion == null || StringUtil.isBlank(urlVersion.value())) {
            urlVersion = AnnotatedElementUtils.findMergedAnnotation(handlerType, UrlVersion.class);
        }
        // Media Types 版本信息
        ApiVersion apiVersion = AnnotatedElementUtils.findMergedAnnotation(method, ApiVersion.class);
        // 再次尝试类上的版本
        if (apiVersion == null || StringUtil.isBlank(apiVersion.value())) {
            apiVersion = AnnotatedElementUtils.findMergedAnnotation(handlerType, ApiVersion.class);
        }
        boolean nonUrlVersion = urlVersion == null || StringUtil.isBlank(urlVersion.value());
        boolean nonApiVersion = apiVersion == null || StringUtil.isBlank(apiVersion.value());
        // 先判断同时不存在
        if (nonUrlVersion && nonApiVersion) {
            return null;
        }

        RequestMappingInfo.Builder mappingInfoBuilder = null;
        // 如果 header 版本不存在
        if (nonApiVersion) {
            mappingInfoBuilder = RequestMappingInfo.paths(urlVersion.value());
        } else {
            mappingInfoBuilder = RequestMappingInfo.paths(StringPool.EMPTY);
        }
        // 如果url版本不存在
        if (nonUrlVersion) {
            String versionMediaTypes = new VioletMediaType(apiVersion.value()).toString();
            mappingInfoBuilder.produces(versionMediaTypes);
        }
        PathPatternParser patternParser = getPatternParser();
        // fix patternParser,不然无法使用api版本
        if (this.config.getPatternParser() == null) {
            this.config.setPatternParser(patternParser);
        }
        this.config.setPathMatcher(getPathMatcher());
        return mappingInfoBuilder.options(config).build();
    }

    @Override
    protected void handlerMethodsInitialized(Map<RequestMappingInfo, HandlerMethod> handlerMethods) {
        // 打印路由信息 spring boot 2.1 去掉了这个 日志的打印
        if (logger.isInfoEnabled()) {
            for (Map.Entry<RequestMappingInfo, HandlerMethod> entry : handlerMethods.entrySet()) {
                RequestMappingInfo mapping = entry.getKey();
                HandlerMethod handlerMethod = entry.getValue();
                logger.info("Mapped \"" + mapping + "\" onto " + handlerMethod);
            }
        }
        super.handlerMethodsInitialized(handlerMethods);
    }
}