package org.violet.common.cloud.http.restemplate;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.cloud.client.loadbalancer.LoadBalanced;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.lang.Nullable;
import org.springframework.web.client.RestTemplate;
import org.violet.common.cloud.feign.header.VioletFeignHeadersProperties;
import org.violet.common.cloud.http.client.OkHttp3ClientHttpRequestFactory;
import org.violet.common.core.util.Charsets;

import java.util.List;

@Slf4j
@RequiredArgsConstructor
@AutoConfiguration
@ConditionalOnClass(OkHttpClient.class)
@ConditionalOnProperty(value = "violet.http.enabled", matchIfMissing = true)
@EnableConfigurationProperties(VioletFeignHeadersProperties.class)
public class RestTemplateConfiguration {
    @Bean
    public RestTemplateHeaderInterceptor requestHeaderInterceptor(
            VioletFeignHeadersProperties properties) {
        return new RestTemplateHeaderInterceptor(properties);
    }

    @AutoConfiguration
    @RequiredArgsConstructor
    @ConditionalOnClass(OkHttpClient.class)
    @ConditionalOnProperty(value = "violet.http.rest-template.enable")
    public static class RestTemplateAutoConfiguration {
        private final ApplicationContext context;

        /**
         * 普通的 RestTemplate，不透传请求头，一般只做外部 http 调用
         *
         * @param okHttpClient OkHttpClient
         * @return RestTemplate
         */
        @Bean
        @ConditionalOnMissingBean
        public RestTemplate restTemplate(RestTemplateBuilder restTemplateBuilder, OkHttpClient okHttpClient) {
            restTemplateBuilder.requestFactory(() -> new OkHttp3ClientHttpRequestFactory(okHttpClient));
            RestTemplate restTemplate = restTemplateBuilder.build();
            configMessageConverters(context, restTemplate.getMessageConverters());
            return restTemplate;
        }
    }

    @AutoConfiguration
    @RequiredArgsConstructor
    @ConditionalOnClass(OkHttpClient.class)
    @ConditionalOnProperty(value = "violet.http.lb-rest-template.enable")
    public static class LbRestTemplateAutoConfiguration {
        private final ApplicationContext context;

        /**
         * 支持负载均衡的 LbRestTemplate
         *
         * @param okHttpClient OkHttpClient
         * @return LbRestTemplate
         */
        @Bean
        @LoadBalanced
        @ConditionalOnMissingBean
        public LbRestTemplate lbRestTemplate(RestTemplateBuilder restTemplateBuilder, OkHttpClient okHttpClient) {
            restTemplateBuilder.requestFactory(() -> new OkHttp3ClientHttpRequestFactory(okHttpClient));
            LbRestTemplate restTemplate = restTemplateBuilder.build(LbRestTemplate.class);
            restTemplate.getInterceptors().add(context.getBean(RestTemplateHeaderInterceptor.class));
            configMessageConverters(context, restTemplate.getMessageConverters());
            return restTemplate;
        }
    }

    private static void configMessageConverters(ApplicationContext context, List<HttpMessageConverter<?>> converters) {
        converters.removeIf(x -> x instanceof StringHttpMessageConverter || x instanceof MappingJackson2HttpMessageConverter);
        converters.add(new StringHttpMessageConverter(Charsets.UTF_8));
        converters.add(new MappingJackson2HttpMessageConverter(context.getBean(ObjectMapper.class)));
    }

}