package org.violet.common.bootstrap.handler.adapter;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.google.auto.service.AutoService;
import org.violet.common.launch.entity.JsonResult;
import jakarta.validation.ConstraintViolation;
import jakarta.validation.ConstraintViolationException;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.violet.common.bootstrap.handler.ExceptionAdapter;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

@AutoService(ExceptionAdapter.class)
public class HttpParamsValidExceptionAdapter implements ExceptionAdapter {
    public HttpParamsValidExceptionAdapter() {
    }

    public String[] supportExceptions() {
        return new String[]{MethodArgumentNotValidException.class.getName(), BindException.class.getName(), ConstraintViolationException.class.getName(), MissingServletRequestParameterException.class.getName()};
    }

    public JsonResult adapter(Throwable e) {
        JsonResult result = JsonResult.ERROR(e.getMessage());
        if (e instanceof BindException) {
            BindException exception = (BindException) e;
            result.msg(this.getArgNotValidMessage(exception.getBindingResult()));
        } else if (e instanceof ConstraintViolationException) {
            ConstraintViolationException exception = (ConstraintViolationException) e;
            result.msg(getArgNotValidMessage(exception.getConstraintViolations()));
        } else if (e instanceof MissingServletRequestParameterException) {
            MissingServletRequestParameterException exception = (MissingServletRequestParameterException) e;
            result.msg(exception.getMessage());
        }

        return result;
    }

    public String getArgNotValidMessage(BindingResult bindingResult) {
        if (ObjectUtil.isNull(bindingResult)) {
            return "";
        } else {
            StringBuilder stringBuilder = StrUtil.builder();
            List<ObjectError> allErrorInfos = bindingResult.getAllErrors();
            Iterator var4 = allErrorInfos.iterator();

            while (var4.hasNext()) {
                ObjectError error = (ObjectError) var4.next();
                stringBuilder.append(",").append(error.getDefaultMessage());
            }

            return StrUtil.removePrefix(stringBuilder.toString(), ",");
        }
    }

    public static String getArgNotValidMessage(Set<ConstraintViolation<?>> constraintViolationSet) {
        if (ObjectUtil.isEmpty(constraintViolationSet)) {
            return "";
        } else {
            StringBuilder stringBuilder = StrUtil.builder();
            Iterator var2 = constraintViolationSet.iterator();

            while (var2.hasNext()) {
                ConstraintViolation<?> constraintViolation = (ConstraintViolation) var2.next();
                stringBuilder.append(",").append(constraintViolation.getMessage());
            }

            return StrUtil.removePrefix(stringBuilder.toString(), ",");
        }
    }
}