/*
 * Decompiled with CFR 0.152.
 */
package io.seata.rm.datasource.undo;

import io.seata.common.Constants;
import io.seata.common.exception.NotSupportYetException;
import io.seata.common.util.BlobUtils;
import io.seata.core.exception.TransactionException;
import io.seata.core.exception.TransactionExceptionCode;
import io.seata.rm.datasource.ConnectionContext;
import io.seata.rm.datasource.ConnectionProxy;
import io.seata.rm.datasource.DataSourceProxy;
import io.seata.rm.datasource.sql.struct.TableMeta;
import io.seata.rm.datasource.sql.struct.TableMetaCache;
import io.seata.rm.datasource.undo.AbstractUndoExecutor;
import io.seata.rm.datasource.undo.BranchUndoLog;
import io.seata.rm.datasource.undo.SQLUndoLog;
import io.seata.rm.datasource.undo.UndoExecutorFactory;
import io.seata.rm.datasource.undo.UndoLogParserFactory;
import java.sql.Blob;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.Statement;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class UndoLogManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(UndoLogManager.class);
    private static String UNDO_LOG_TABLE_NAME = "undo_log";
    private static String INSERT_UNDO_LOG_SQL = "INSERT INTO " + UNDO_LOG_TABLE_NAME + "\n\t(branch_id, xid, rollback_info, log_status, log_created, log_modified)\nVALUES (?, ?, ?, ?, now(), now())";
    private static String DELETE_UNDO_LOG_SQL = "DELETE FROM " + UNDO_LOG_TABLE_NAME + "\n\tWHERE branch_id = ? AND xid = ?";
    private static String SELECT_UNDO_LOG_SQL = "SELECT * FROM " + UNDO_LOG_TABLE_NAME + " WHERE  branch_id = ? AND xid = ? FOR UPDATE";

    private UndoLogManager() {
    }

    public static void flushUndoLogs(ConnectionProxy cp) throws SQLException {
        UndoLogManager.assertDbSupport(cp.getDbType());
        ConnectionContext connectionContext = cp.getContext();
        String xid = connectionContext.getXid();
        long branchID = connectionContext.getBranchId();
        BranchUndoLog branchUndoLog = new BranchUndoLog();
        branchUndoLog.setXid(xid);
        branchUndoLog.setBranchId(branchID);
        branchUndoLog.setSqlUndoLogs(connectionContext.getUndoItems());
        byte[] undoLogContent = UndoLogParserFactory.getInstance().encode(branchUndoLog);
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Flushing UNDO LOG: {}", (Object)new String(undoLogContent, Constants.DEFAULT_CHARSET));
        }
        UndoLogManager.insertUndoLogWithNormal(xid, branchID, undoLogContent, cp.getTargetConnection());
    }

    private static void assertDbSupport(String dbType) {
        if (!"mysql".equals(dbType)) {
            throw new NotSupportYetException("DbType[" + dbType + "] is not support yet!");
        }
    }

    public static void undo(DataSourceProxy dataSourceProxy, String xid, long branchId) throws TransactionException {
        UndoLogManager.assertDbSupport(dataSourceProxy.getDbType());
        Connection conn = null;
        ResultSet rs = null;
        Statement selectPST = null;
        while (true) {
            try {
                conn = dataSourceProxy.getPlainConnection();
                conn.setAutoCommit(false);
                selectPST = conn.prepareStatement(SELECT_UNDO_LOG_SQL);
                selectPST.setLong(1, branchId);
                selectPST.setString(2, xid);
                rs = selectPST.executeQuery();
                boolean exists = false;
                while (rs.next()) {
                    exists = true;
                    int state = rs.getInt("log_status");
                    if (!UndoLogManager.canUndo(state)) {
                        if (LOGGER.isInfoEnabled()) {
                            LOGGER.info("xid {} branch {}, ignore {} undo_log", new Object[]{xid, branchId, state});
                        }
                        return;
                    }
                    Blob b = rs.getBlob("rollback_info");
                    byte[] rollbackInfo = BlobUtils.blob2Bytes(b);
                    BranchUndoLog branchUndoLog = UndoLogParserFactory.getInstance().decode(rollbackInfo);
                    for (SQLUndoLog sqlUndoLog : branchUndoLog.getSqlUndoLogs()) {
                        TableMeta tableMeta = TableMetaCache.getTableMeta(dataSourceProxy, sqlUndoLog.getTableName());
                        sqlUndoLog.setTableMeta(tableMeta);
                        AbstractUndoExecutor undoExecutor = UndoExecutorFactory.getUndoExecutor(dataSourceProxy.getDbType(), sqlUndoLog);
                        undoExecutor.executeOn(conn);
                    }
                }
                if (exists) {
                    UndoLogManager.deleteUndoLog(xid, branchId, conn);
                    conn.commit();
                    if (LOGGER.isInfoEnabled()) {
                        LOGGER.info("xid {} branch {}, undo_log deleted with {}", new Object[]{xid, branchId, State.GlobalFinished.name()});
                    }
                } else {
                    UndoLogManager.insertUndoLogWithGlobalFinished(xid, branchId, conn);
                    conn.commit();
                    if (LOGGER.isInfoEnabled()) {
                        LOGGER.info("xid {} branch {}, undo_log added with {}", new Object[]{xid, branchId, State.GlobalFinished.name()});
                    }
                }
                return;
            }
            catch (SQLIntegrityConstraintViolationException e) {
                if (!LOGGER.isInfoEnabled()) continue;
                LOGGER.info("xid {} branch {}, undo_log inserted, retry rollback", (Object)xid, (Object)branchId);
                continue;
            }
            catch (Throwable e) {
                if (conn != null) {
                    try {
                        conn.rollback();
                    }
                    catch (SQLException rollbackEx) {
                        LOGGER.warn("Failed to close JDBC resource while undo ... ", (Throwable)rollbackEx);
                    }
                }
                throw new TransactionException(TransactionExceptionCode.BranchRollbackFailed_Retriable, String.format("%s/%s", branchId, xid), e);
            }
            finally {
                try {
                    if (rs != null) {
                        rs.close();
                    }
                    if (selectPST != null) {
                        selectPST.close();
                    }
                    if (conn == null) continue;
                    conn.close();
                }
                catch (SQLException closeEx) {
                    LOGGER.warn("Failed to close JDBC resource while undo ... ", (Throwable)closeEx);
                }
                continue;
            }
            break;
        }
    }

    public static void batchDeleteUndoLog(Set<String> xids, Set<Long> branchIds, Connection conn) throws SQLException {
        int xidSize = xids.size();
        int branchIdSize = branchIds.size();
        String batchDeleteSql = UndoLogManager.toBatchDeleteUndoLogSql(xidSize, branchIdSize);
        try (PreparedStatement deletePST = null;){
            deletePST = conn.prepareStatement(batchDeleteSql);
            int paramsIndex = 1;
            for (Long branchId : branchIds) {
                deletePST.setLong(paramsIndex++, branchId);
            }
            for (String xid : xids) {
                deletePST.setString(paramsIndex++, xid);
            }
            int deleteRows = deletePST.executeUpdate();
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("batch delete undo log size " + deleteRows);
            }
        }
    }

    protected static String toBatchDeleteUndoLogSql(int xidSize, int branchIdSize) {
        StringBuilder sqlBuilder = new StringBuilder(64);
        sqlBuilder.append("DELETE FROM ").append(UNDO_LOG_TABLE_NAME).append(" WHERE  branch_id IN ");
        UndoLogManager.appendInParam(branchIdSize, sqlBuilder);
        sqlBuilder.append(" AND xid IN ");
        UndoLogManager.appendInParam(xidSize, sqlBuilder);
        return sqlBuilder.toString();
    }

    protected static void appendInParam(int size, StringBuilder sqlBuilder) {
        sqlBuilder.append(" (");
        for (int i = 0; i < size; ++i) {
            sqlBuilder.append("?");
            if (i >= size - 1) continue;
            sqlBuilder.append(",");
        }
        sqlBuilder.append(") ");
    }

    public static void deleteUndoLog(String xid, long branchId, Connection conn) throws SQLException {
        try (PreparedStatement deletePST = null;){
            deletePST = conn.prepareStatement(DELETE_UNDO_LOG_SQL);
            deletePST.setLong(1, branchId);
            deletePST.setString(2, xid);
            deletePST.executeUpdate();
        }
    }

    private static void insertUndoLogWithNormal(String xid, long branchID, byte[] undoLogContent, Connection conn) throws SQLException {
        UndoLogManager.insertUndoLog(xid, branchID, undoLogContent, State.Normal, conn);
    }

    private static void insertUndoLogWithGlobalFinished(String xid, long branchID, Connection conn) throws SQLException {
        UndoLogManager.insertUndoLog(xid, branchID, "{}".getBytes(Constants.DEFAULT_CHARSET), State.GlobalFinished, conn);
    }

    private static void insertUndoLog(String xid, long branchID, byte[] undoLogContent, State state, Connection conn) throws SQLException {
        try (PreparedStatement pst = null;){
            pst = conn.prepareStatement(INSERT_UNDO_LOG_SQL);
            pst.setLong(1, branchID);
            pst.setString(2, xid);
            pst.setBlob(3, BlobUtils.bytes2Blob(undoLogContent));
            pst.setInt(4, state.getValue());
            pst.executeUpdate();
        }
    }

    private static boolean canUndo(int state) {
        return state == State.Normal.getValue();
    }

    private static enum State {
        Normal(0),
        GlobalFinished(1);

        private int value;

        private State(int value) {
            this.value = value;
        }

        public int getValue() {
            return this.value;
        }
    }
}

