package com.kdgcsoft.power.filestore.jackrabbit;

import java.net.MalformedURLException;

import javax.jcr.Repository;

import org.apache.jackrabbit.rmi.repository.URLRemoteRepository;

import com.kdgcsoft.power.filestore.FileStore;
import com.kdgcsoft.power.filestore.FileStoreException;

/**
 * RMI方式的远程JackRabbit仓库的实现类。
 * @author hling
 *
 */
public class RemoteJackRabbitFileStore extends BaseJackRabbitFileStore implements FileStore {
	
	/**
	 * 使用缺省账号，创建一个远程JackRabbit仓库的连接，实例名采用缺省名。
	 * 
	 * @param url
	 *            必须。远程仓库的URL，例如http://localhost:8082/rmi。
	 * @throws FileStoreException 存储库异常
	 **/            
	public RemoteJackRabbitFileStore(final String url) throws FileStoreException {
		this(url, DEFAULT_STORE_NAME);
	}
	
	/**
	 * 用指定的命名，创建一个远程JackRabbit仓库的连接
	 * 
	 * @param url
	 *            必须。远程仓库的URL，例如http://localhost:8082/rmi。
	 * @param storeName
	 *            必须。给仓库起个名字，后面需要根据这个名字来获取仓库实例。该名字也会成为仓库的根节点目录名。
	 *            文字中不能含有特殊字符如斜杠等。
	 * @throws FileStoreException 存储库异常
	 **/            
	public RemoteJackRabbitFileStore(final String url, final String storeName) throws FileStoreException {
		this(url, storeName, DEFAULT_USER, DEFAULT_PASS);
	}

	/**
	 * 用指定的命名和访问账号，创建一个远程JackRabbit仓库的连接。使用缺省的Store名"DefaultStore"
	 * 
	 * @param url
	 *            必须。远程仓库的URL，例如http://localhost:8082/rmi。
	 * @param userName 用于访问仓库的用户名。
	 * @param password 用于访问仓库的密码。注意用户名和密码的设置是在JackRabbit的配置文件中，这里设置的只能用于访问仓库，不能用于创建仓库账户。
	 * @throws FileStoreException 存储库异常
	 */
	public RemoteJackRabbitFileStore(final String url, final String userName, final String password) throws FileStoreException {
		this(url, DEFAULT_STORE_NAME, userName, password);
	}

	/**
	 * 用指定的命名和访问账号，创建一个远程JackRabbit仓库的连接
	 * 
	 * @param url
	 *            必须。远程仓库的URL，例如http://localhost:8082/rmi。
	 * @param storeName
	 *            给仓库实例起的名字，后面需要根据这个名字来获取仓库实例。该名字也会成为仓库的根节点目录名。 文字中不能含有特殊字符如斜杠等。
	 * @param userName 用于访问仓库的用户名。
	 * @param password 用于访问仓库的密码。注意用户名和密码的设置是在JackRabbit的配置文件中，这里设置的只能用于访问仓库，不能用于创建仓库账户。
	 * @throws FileStoreException 存储库异常
	 */
	public RemoteJackRabbitFileStore(final String url, final String storeName, final String userName, final String password) throws FileStoreException {
		Repository repository;
		try {
			repository = new URLRemoteRepository(url);
		} catch (MalformedURLException e) {
			logger.error("URL格式错误！{}", url, e);
			throw new FileStoreException("文件库的URL格式错误！", e);
		}
		
		init(repository, storeName, userName, password);
	}

	/* (non-Javadoc)
	 * @see com.kdgcsoft.power.filestore.FileStore#shutdown()
	 */
	@Override
	public void shutdown() {
		// Do Nothing
	}
	
}
