package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;
import java.util.Set;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.jodconverter.OfficeDocumentConverter;
import org.jodconverter.document.DocumentFormat;
import org.jodconverter.document.DocumentFormatRegistry;
import org.jodconverter.office.DefaultOfficeManagerBuilder;
import org.jodconverter.office.OfficeException;
import org.jodconverter.office.OfficeManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JodConverter(https://github.com/sbraconnier/jodconverter)的调用类。
 * @author hling
 *
 */
public class JodHelper {

	private static Logger logger = LoggerFactory.getLogger(JodHelper.class);

	protected static volatile OfficeManager officeManager = null;
	protected static volatile OfficeDocumentConverter converter;
	protected static volatile DocumentFormatRegistry formatRegistry;

	/**
	 * 在缺省位置启动OpenOffice/LibreOffice。缺省位置与操作系统有关。
	 * 如果是Windows系统，则位于系统盘的program files目录下。
	 * @throws OfficeException
	 */
	public static synchronized void startOfficeManager(final String openOfficeDir, final int[] ports) throws OfficeException {
		// 不能两次启动
		if (officeManager != null) {
			return;
		}
		
		// Start an office manager
		DefaultOfficeManagerBuilder builder = new DefaultOfficeManagerBuilder();
		builder = builder.setPortNumbers(ports);
		if (openOfficeDir != null) {
			builder = builder.setOfficeHome(openOfficeDir);
		}
		builder.setTaskQueueTimeout(60000);
		
		officeManager = builder.build();
		
		converter = new OfficeDocumentConverter(officeManager);
		formatRegistry = converter.getFormatRegistry();
		
		officeManager.start();
	}

	public static synchronized void stopOfficeManager() {
		if (officeManager != null) {
			try {
				officeManager.stop();
			} catch (OfficeException e) {
				logger.error("关闭LibreOffice/OpenOffice异常。您可能需要手动Kill进程！", e);
			}
		} else {
			logger.error("OpenOffice尚未启动，无需关闭");
		}
	}
	
	public static void main(String[] args) {
		try {
			startOfficeManager(null, new int[]{5678, 5679});			
//			converter.convert(new File("D:/111.docx"), new File("D:/111.png"));
			stopOfficeManager();
		} catch (OfficeException e) {
			e.printStackTrace();
		}
	}
	
	public static boolean canConvert(final File inputFile, final String outputExtension) {
		final String inputExtension = FilenameUtils.getExtension(inputFile
				.getName());
		final DocumentFormat inputFormat = formatRegistry
				.getFormatByExtension(inputExtension);
		
		if (inputFormat == null) {
			logger.error("不支持的输入文件格式:", inputFile.getName());
			return false;
		}
		
		// Get all supported output formats
		final Set<DocumentFormat> outputFormats = formatRegistry
				.getOutputFormats(inputFormat.getInputFamily());
		for (final DocumentFormat outputFormat : outputFormats) {
			// LibreOffice 4 fails natively on those one
			if (inputFormat.getExtension().equals("odg")
					&& outputFormat.getExtension().equals("svg")) {
				logger.info("-- skipping odg to svg test... ");
				continue;
			}
			if (StringUtils.equalsAny(outputFormat.getExtension(), "sxc",
					"sxw", "sxi")) {
				logger.info("-- skipping {} to {} test... ", inputExtension,
						outputFormat.getExtension());
				continue;
			}
			
			if (StringUtils.equals(outputFormat.getExtension(), outputExtension)){
				return true;
			}
		}
		
		logger.error("不支持的输出文件格式:", outputExtension);
		return false;
	}
	
	public static void convert(final File inputFile, final File outputFile, final String openOfficeDir, final int[] ports) throws OfficeException {
		if (officeManager == null) {
			logger.info("接收到转换任务，开始启动OpenOffice...");
			startOfficeManager(openOfficeDir, ports);
		}
		
		if (!officeManager.isRunning()) {
			logger.error("OpenOffice未处于运行状态，尝试启动...");
			officeManager.start();
		}
		
		converter.convert(inputFile, outputFile);		
	}
}
