package com.kdgcsoft.power.fileconverter;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.Locale;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.impl.JodHelper;
import com.kdgcsoft.power.fileconverter.storage.FreeStyleStorage;
import com.kdgcsoft.power.fileconverter.storage.TimeStampStorage;
import com.kdgcsoft.power.fileconverter.storage.UUIDStorage;
import com.kdgcsoft.power.fileconverter.util.FileExtensionUtil;


/**
 * 文档转换工具。调用外部程序，把office等所支持的文档格式转换为PDF等格式。 多线程执行，支持同步、异步调用。<br>
 * 转换后文件存放在工作目录下，可通过转换时返回的key作为参数，调用接口获取。
 * <br>
 * 源文件与key的映射记录可从日志文件中获取。建议在日志配置文件中给名为"ConvertHistory"的
 * Logger定义专门的输出文件位置，这样记录所有文件对应关系，便于维护查错。
 */
public class FileConverterService {

	private static final Logger logger = LoggerFactory.getLogger(FileConverterService.class);
	private static final Logger historyLogger = LoggerFactory.getLogger("ConvertHistory");
	
	private static final String[] CAN_CONVERT_EXTS = {"doc", "docx", "ppt", "pptx", "xls", "xlsx", "pdf"};

	private static ExecutorService service = null;

	private volatile static boolean inited = false;
	
	private static FileConverterSettings settings = null;
	
	/**
	 * 缺省使用基于时间戳的目录结构来存储转换后的文件
	 */
	private static IFileStorageHelper storageHelper = new TimeStampStorage();

	/**
	 * 初始化转换环境
	 * @param settings 转换设定
	 * @throws IOException
	 */
	synchronized public static void init(final FileConverterSettings settings) throws IOException {
		
		logger.info("启动文档转换功能...");
		
		if (inited) {
			logger.warn("已经对DocConvertService进行过初始化。本次初始化中止。");
			return;
		}
		
		if (settings.getConvertEngine() == null ) {
			logger.error("必须指定基本转换引擎，选择OpenOffice/LibreOffice、Jacob、wps其中之一");
			return;
		}
		
		File dir = settings.getWorkdir();
		// 转换成绝对路径，便于调试
		dir = new File(dir.getAbsolutePath());
		
		if (!dir.exists()) {
			try {
				// 建立文件夹
				FileUtils.forceMkdir(dir);
			} catch (IOException e) {
				logger.error("无法创建转换工作目录{}", dir.getAbsolutePath(), e);
				throw new IOException("无法创建文档转换工作目录" + dir.getAbsolutePath());
			}
		}

		logger.info("文档转换工作目录：{}", dir.getAbsolutePath());
		
		// 建立工作目录
		try {
			FileUtils.forceMkdir(new File(dir, "incoming"));
			FileUtils.forceMkdir(new File(dir, "storage"));
		} catch (IOException e) {
			logger.error("创建incoming和storage子目录失败。", e);
			throw new IOException("创建incoming和storage子目录失败。");
		}
		
		setStorageType(settings.getStorageType());
		
		service = Executors.newFixedThreadPool(settings.getMaxConvertThread());
		
		inited = true;
		FileConverterService.settings = settings;
		
		// 重新启动未完成的转换
//		redoUnfinished();
	}
	
	/**
	 * 设置本地文件存储管理方式，可使用基于UUID、时间戳或支持任意Key的文件存储结构。
	 * 默认是基于时间戳（年月日三层目录）结构来存储转换后的文件。
	 * 设置文件管理器的目的是为了在调试、运维时方便在服务器上直接查看转换后的文件。
	 * @param storType 本地文件存储管理方式
	 */
	synchronized private static void setStorageType(StorageType storType) {
		IFileStorageHelper storHelper;
		switch (storType) {
		case UUID:
			storHelper = new UUIDStorage();
			break;
		case TimeStamp:
			storHelper = new TimeStampStorage();
			break;
		case FreeStyle:
			storHelper = new FreeStyleStorage();
			break;
		default:
			storHelper = new TimeStampStorage();
		}
		
		storageHelper = storHelper;
	}

	/**
	 * 关闭外部转换进程。应在系统退出时调用。如果需要再次启动转换进程，要重新调用init()。
	 */
	synchronized public static void finish() {
		logger.info("关闭文档转换功能开始...");
		try {
			if (service != null) {
				service.shutdown();
				service.awaitTermination(300, TimeUnit.SECONDS);
			}
		} catch (InterruptedException e) {
			logger.error("关闭转换线程池时出现异常，关闭线程被取消", e);
			Thread.currentThread().interrupt();
		} finally {
			// TODO 确认：除了OpenOffice/LibreOffice，其他转换引擎是否也需要手动关闭？更灵活、松耦合的收尾退出方式？
			if (ConvertEngineType.openOffice.equals(settings.getConvertEngine())) {
				JodHelper.stopOfficeManager();
			}
			inited = false;

			logger.info("关闭文档转换功能结束");
		}
		
		
	}
	
	/**
	 * 重新执行未完成的文档转换。主要用于程序终止、机器断电等异常情况下的后处理。
	 * 该函数会检查工作目录下的incoming文件夹，当发现有符合要求的未完成转换的文件时(通过任务定义文件：[文件Key].[源文件扩展名].[OutputType].task)， 即将其重新投入队列进行转换。<br>
	 * 需要在调用{@link #init(FileConverterSettings)}之后调用本函数。调用过程是安全的，如果不存在异常终止的转换任务，则什么都不会做。
	 */
	synchronized public static void redoUnfinished() {
		logger.info("开始重新转换未完成的文件");

		if (!inited) {
			logger.error("工作目录未指定，不能开始Redo。应该先调用init初始化工作环境，且初始化参数必须与上次保持一致。");
			return;
		}

		Collection<File> list = FileUtils.listFiles(settings.getIncomingDir(),
				new String[]{FileConverterSettings.TASK_FILE_EXT}, false);
		for (File file : list) {
			// 文件名格式为 [文件Key].[源文件扩展名].[OutputType].task，例如： 201803281755070001.doc.pdf.task
			// taskName: [文件Key].[源文件扩展名].[OutputType]
			String taskName = FilenameUtils.removeExtension(file.getName());
			// originFileName: [文件Key].[源文件扩展名]
			String originFileName = FilenameUtils.getBaseName(taskName);
			// 同时有lock文件存在的，说明正在处理中，忽略
			File lock = new File(file.getParent(), FileExtensionUtil.addExtension(taskName, "lock"));
			if (lock.exists()) {
				continue;
			}

			try {
				// 判断是否是合法的Key
				String key = FilenameUtils.getBaseName(originFileName);
				storageHelper.validateKey(key);
				
				// 取得目标格式信息
				String format = FilenameUtils.getExtension(taskName);
				if (format == null || format.isEmpty()) {
					logger.warn("文件名{}中找不到转换目标格式，跳过。", file.getName());
					continue;
				}
				OutputType outType = OutputType.valueOf(format);

				logger.info("发现未完成的转换任务。Key：{}，源文件名：{}，转换目标：{}", key, originFileName, format);
				
				// 在incoming文件夹和storage文件夹中查找待转换的源文件
				File srcFile = new File(settings.getIncomingDir(), originFileName);
				if (!srcFile.exists() || !srcFile.isFile()) {
					srcFile = new File(getStorageDir(key), originFileName);
				}
				
				if (!srcFile.exists() || !srcFile.isFile()) {
					convert(srcFile, key, outType);
				} else {
					logger.error("找不到源文件{}，无法继续。", originFileName);
				}
			} catch (IllegalArgumentException e) {
				logger.warn("文件名中未包含可识别的文件Key：", file.getName());
			} catch (FileConverterException e) {
				logger.error("转换失败：" + file.getAbsolutePath(), e);
			}
		}

		logger.info("结束重新转换");
	}
	
	/**
	 * 是否支持把某文件转换成目标格式
	 * @param fileName 待转换的文件名。函数基于其扩展名进行判断。
	 * @param type 目标格式
	 * @return true:支持转换 false:不支持
	 */
	public static boolean canCanvert(String fileName, OutputType type) {
		if (!FilenameUtils.isExtension(fileName, CAN_CONVERT_EXTS)) {
			logger.error("不支持转换成目标类型。文件名：{}, 目标类型：{} ", fileName, type.toString());
			return false;
		} else {
			return true;
		}
	}

	/**
	 * 转换文档，可指定使用异步模式还是同步模式。
	 * 如果不支持转换到目标格式，则不进行转换，但也不抛出异常。
	 * 
	 * @param srcFile
	 *            源文档
	 * @param outType 输出类型枚举
	 * @param isAsync
	 *            是否异步
	 * @return 用于获取转换结果文件的Key
	 * @throws FileConverterException
	 */
	public static String convert(final File srcFile, final OutputType outType, final boolean isAsync)
			throws FileConverterException {
		return convert(srcFile, storageHelper.generateKey(), outType, isAsync);
	}

	/**
	 * 转换文件流到文档，可指定使用异步模式还是同步模式。
	 * 如果不支持转换到目标格式，则不进行转换，但也不抛出异常。
	 * 
	 * @param inputStream 源文件流
	 * @param srcFileName 源文件名。因为仅根据文件流无法识别源文件格式，所以转化器需要依赖文件名参数。只要扩展名正确即可，主文件名不使用。
	 * @param outType 输出类型枚举
	 * @param isAsync
	 *            是否异步
	 * @return 用于获取转换结果文件的Key
	 * @throws FileConverterException
	 */
	public static String convert(final InputStream inputStream, final String srcFileName, final OutputType outType, final boolean isAsync)
			throws FileConverterException {
		return convert(inputStream, srcFileName, storageHelper.generateKey(), outType, isAsync);
	}

	/**
	 * 使用同步方式转换文档。参数中直接指定文档的key(必须与当前StorageHelper使用的key格式一致），返回的是相同key。用于文档更新、REDO。
	 * 当外部调用者本身使用key管理文档时可采用此种转换方式，便于保持一致，避免系统中存在多种key。
	 * 也可用于文档更新（key不变，只重新转换）。
	 * 如果不支持转换到目标格式，则不进行转换，但也不抛出异常。
	 * 
	 * @param srcFile
	 *            源文档
	 * @param key
	 *            指定使用的文件Key，用于存储管理
	 * @return 用于获取转换结果文件的key
	 * @throws FileConverterException
	 */
	public static String convert(final File srcFile, final String key, final OutputType outType)
			throws FileConverterException {
		return convert(srcFile, key, outType, false);
	}

	/**
	 * 使用同步方式转换文档。参数中直接指定文档的key(必须与当前StorageHelper使用的key格式一致）。用于文档更新、REDO。
	 * 如果不支持转换到目标格式，则不进行转换，但也不抛出异常。
	 * 
	 * @param inputStream
	 *            源文档流
	 * @param srcFileName 源文件名。因为仅根据文件流无法识别源文件格式，所以转化器需要依赖文件名参数。只要扩展名正确即可，主文件名不使用。
	 * @param key 存储key
	 * @return 用于获取转换结果文件的key
	 * @throws FileConverterException
	 */
	public static String convert(final InputStream inputStream, final String srcFileName, final String key, final OutputType outType)
			throws FileConverterException {
		return convert(inputStream, srcFileName, key, outType, false);
	}
	
	/**
	 * 执行转换过程。此过程通过线程池执行转换任务，并根据是否异步的参数决定是启动任务后立即返回还是等待任务执行完毕再返回。
	 * 如果不支持转换到目标格式，则不进行转换，但也不抛出异常。
	 * 
	 * @param srcFile 源文档
	 * @param key 指定使用的文件Key。应用端第一次转换某文件时不要指定Key，但需要记录转换器返回的Key。
	 *               下次转换相同文件时则应传入该Key，便于转换器识别是否是曾经转换过的文件
	 * @param outType 要转换成的类型
	 * @param isAsync 是否异步
	 * @return 用于获取转换结果文件的key
	 * @throws FileConverterException
	 */
	public static String convert(final File srcFile, final String key, final OutputType outType, final boolean isAsync)
			throws FileConverterException {

		logger.info("准备转换文件{}到{}格式", srcFile.getAbsolutePath(), outType.toString());

		if (!inited) {
			throw new FileConverterException("未设置转换工作目录！请先调用init()函数设定工作目录");
		}

		// 检查源文件
		if (!srcFile.exists() || srcFile.isDirectory() || !srcFile.canRead()) {
			logger.error("无法读取待转换文件{}", srcFile);
			throw new FileConverterException("无法读取待转换文件" + srcFile);
		}
		
		// 检查是否是已经转换过的文件
		File convertedFile = getConvertedFile(key, outType);
		if (convertedFile != null) {
			logger.info("文档已被转换过，无需再次转换。key={}", key);
			return key;
		}
		
		if (!canCanvert(srcFile.getName(), outType)) {
			logger.info("不支持的格式转换。源文件名：{}，目标格式：{}", srcFile.getName(), outType.toString());
			return key;
		}
		
		String extName = FilenameUtils.getExtension(srcFile.getName())
				.toLowerCase(Locale.ENGLISH);

		// 统一使用key作为新文件名进行后续处理
		File incomingFile = new File(settings.getIncomingDir(), key + "." + extName);

		try {
			// 删除incoming下已有同名文件(key名称)，然后拷贝源文件到incoming下并改名为key名称。
			// 但如果源文件本身就在incoming目录下则不处理
			if (FileUtils.directoryContains(settings.getIncomingDir(), srcFile)) {
				// 源文件就在incoming目录下时，直接改名
				if (!FilenameUtils.equals(srcFile.getName(), incomingFile.getName())) {
					FileUtils.moveFile(srcFile, incomingFile);
				}
			} else {
				FileUtils.deleteQuietly(incomingFile);
				FileUtils.copyFile(srcFile, incomingFile);
			}
		} catch (IOException e) {
			logger.error("拷贝源文件失败", e);
			throw new FileConverterException("拷贝源文件失败：" + srcFile.getAbsolutePath());
		}

		logger.info("源文件:{} -> key:{}", srcFile.getAbsolutePath(), key);
		historyLogger.info("{}, {}", srcFile.getAbsolutePath(), key);

		File targetPath = getStorageDir(key);
		try {
			FileUtils.forceMkdir(targetPath);
		} catch (IOException e1) {
			throw new FileConverterException("创建目录失败", e1);
		}
		
		// 如果源文件与目标格式相同，则不进行转化，直接移动到存储目录并结束
		if (outType.toString().equals(extName)) {
			try {
				FileUtils.deleteQuietly(new File(targetPath, incomingFile.getName()));
				FileUtils.moveFileToDirectory(incomingFile, targetPath, true);
			} catch (IOException e) {
				logger.error("移动文件失败", e);
				throw new FileConverterException("转换失败！试图移动文件到存储区时出现异常：" + e.getMessage());
			}
			
			// 移动结束，直接返回
			return key;
		}
		
		// 添加转换任务（根据isAsync参数决定是否等待完成）
		addConvertTask(incomingFile, targetPath, outType, isAsync);
		return key;
	}
	
	/**
	 * 添加一个转换任务到转换队列，并根据参数决定是同步等待任务完成还是立即返回
	 * 
	 * @param srcFile 待转换的源文件
	 * @param targetDir 转换后存放的目标目录
	 * @param outType 要转换成的类型
	 * @param isAsync 是否异步执行。true：异步执行，函数立即返回  false：函数将等待转换任务完成再返回
	 * @return 同步执行：如果转换后是单个文件（不分页），则返回文件对象，否则返回分页文件所在的目录对象; <br>
	 *               异步执行：返回null
	 * @throws FileConverterException 转换时出现异常
	 */
	public static File addConvertTask(final File srcFile, final File targetDir, OutputType outType, boolean isAsync) throws FileConverterException {
		File result = null;
		
		Future<File> future = service.submit(new ConvertTask(srcFile, targetDir, outType, settings));
		
		if (isAsync) {
			// 异步调用的情况下，直接向调用者返回key
			logger.debug("异步调用：转换线程已启动。 srcFile={}", srcFile.getAbsolutePath());
		} else {
			// 同步调用，等待线程返回
			try {
				// TODO 根据文件尺寸动态调整超时时间？
				result = future.get(FileConverterSettings.CONVERT_TASK_TIMEOUT_MINUTES, TimeUnit.MINUTES);
				// 转换成功，返回内部ID
				if (result != null) {
					logger.debug("转换线程执行成功。srcFile={}, 转换结果：{}", srcFile.getName(), result);
				} else {
					logger.error("转换线程执行失败。srcFile={}", srcFile.getName());
				}
			} catch (InterruptedException e) {
				logger.error("转换线程被中断", e);
				Thread.currentThread().interrupt();
				throw new FileConverterException("转换中断:" + e.getMessage());
			} catch (ExecutionException e) {
				logger.error("转换异常", e.getCause());
				throw new FileConverterException("转换异常:" + e.getCause().getMessage());
			} catch (TimeoutException e) {
				logger.error("转换超时", e);
				throw new FileConverterException("转换超时:" + e.getMessage());
			}
		}
		
		return result;
	}
	
	/**
	 * 执行转换过程。此过程通过线程池执行转换任务，并根据是否异步的参数决定是启动任务后立即返回还是等待任务执行完毕再返回。
	 * 
	 * @param inputStream 源文档
	 * @param srcFileName 源文件名。因为仅根据文件流无法识别源文件格式，所以转化器需要依赖文件名参数。只要扩展名正确即可，主文件名不使用。
	 * @param key 指定使用的文件存储Key
	 * @param isAsync 是否异步
	 * @return 用于获取转换结果文件的key
	 * @throws FileConverterException
	 */
	public static String convert(final InputStream inputStream, final String srcFileName, final String key, final OutputType outType, final boolean isAsync)
			throws FileConverterException {
		
		File file = new File(settings.getIncomingDir(), key + "." + FilenameUtils.getExtension(srcFileName));
		try {
			// 直接用key作为文件名
			FileUtils.copyInputStreamToFile(inputStream, file);
		} catch (IOException e) {
			logger.error("获取数据流并写入文件时失败。", e);
			throw new FileConverterException("获取数据流并写入文件时失败:" + e.getMessage());
		}
		
		return convert(file, key, outType, isAsync);
	}

	/**
	 * 获取转换后的文件。如果文档正在转换，则函数会阻塞直到转换完成。如果指定格式的文档尚不存在，也不处于转换过程中，则返回null。<br>
	 * 如果要求的格式文件是单一文件，则返回文件对象；如果要求的格式文件是以分页形式存在的（例如png图片）则返回储存这些分页文件的子文件夹。<br>
	 * 调用者应该使用File.isFile()来判断返回的究竟是文件还是文件夹，再决定如何进行下一步处理。
	 * 当前转换配置下，只有请求png图片时会返回一个文件夹对象，其他格式返回的都是文件对象。
	 * 如果想要直接获取分片的文件，应使用另一个函数：{@link  #getConvertedPage(String, OutputType, int) }  
	 * @param key 文档key。在转换时获得。
	 * @param outType 要获取的文档类型（转换后的格式）
	 * @return 文档File对象。如果不存在或等待转换过程超时，返回null。
	 */
	final public static File getConvertedFile(final String key, final OutputType outType) {
		final File targetPath = getStorageDir(key);
		final File target = new File(targetPath, FileExtensionUtil.addExtension(key, outType.toString()));
		
		if (target.exists() && target.isFile()) {
			return target;
		} else {
			if (isConverting(key, outType)) {
				try {
					waitConvert(key, outType);
					if (target.exists() && target.isFile()) {
						return target;
					}
				} catch (TimeoutException e) {
					return null;
				}
			}
			
			// 如果存在文件夹对象，则返回文件夹
			File dir = new File(targetPath, FileExtensionUtil.addExtension(key, outType.toString() + FileConverterSettings.PAGED_FILE_DIR_SUFFIX));
			if (dir.exists() && dir.isDirectory()) {
				return dir;
			}
			
			return null;
		}
	}
	
	/**
	 * 获取转换后的完整文档数据流。如果文档正在转换，则函数会阻塞直到转换完成。<br>
	 * 当转换后文档是分页保存的情况下，需要指定页面编号，使用 {@link  #getConvertedPageAsStream(String, OutputType, int) }  
	 * @param key 文档key。在转换时获得。
	 * @param outType 要获取的文档类型（转换后的格式）
	 * @return 文档Stream对象。如果不存在，返回null。
	 */
	public static InputStream getConvertedFileAsStream(final String key, final OutputType outType) {
		File target = getConvertedFile(key, outType);
		if (target == null) {
			logger.error("请求的文件类型尚未进行转换。文件key={}，类型={}", key, outType.toString());
			return null;
		} else if (target.isFile()) {
			return getFileAsStream(target);
		} else {
			// 如果是文件夹，则返回第一页
			return getConvertedPageAsStream(key, outType, 0);
		}
	}
	
	/**
	 * 按页获取转换后的png图片文件。如果文档正在转换，则函数会阻塞直到转换完成。<br>
	 * @param key 文档key
	 * @param page 第几页。基于0
	 * @return 图片文件对象。如果不存在，返回null。
	 */
	public static File getConvertedPng(final String key, final int page) {
		return getConvertedPage(key, OutputType.png, page);
	}
	
	/**
	 * 按页获取转换后的png图片数据流。如果文档正在转换，则函数会阻塞直到转换完成。<br>
	 * @param key 文档key
	 * @param page 第几页。基于0
	 * @return 图片文件流。如果不存在，返回null。
	 */
	public static InputStream getConvertedPngAsStream(final String key, final int page) {
		File target = getConvertedPng(key, page);
		return getFileAsStream(target);
	}
	
	/**
	 * 按页获取转换后的文件。通常用于图片，但如果转换器支持分页，也可以用于HTML等文件。
	 * 如果文档正在转换，则函数会阻塞直到转换完成。<br>
	 * @param key 文档key
	 * @param outType 要获取的文件格式
	 * @param page 第几页。基于0
	 * @return 分页的文件对象。如果不存在，返回null。
	 */
	public static File getConvertedPage(final String key, OutputType outType, final int page) {
		if (StringUtils.isBlank(key)) {
			logger.warn("必须传入有效的key来获取文件");
			throw new IllegalArgumentException("必须传入有效的key来获取文件！当前key为null或空！");
		}
		
		String fileExt = outType.toString();

		if (page < 0) {
			logger.warn("传入参数的页数不合法");
			throw new IllegalArgumentException("页数不能小于0！");
		}
		
		logger.info("开始获取{}的第{}页{}文件", key, page, fileExt);
		
		try {
			waitConvert(key, outType);
		} catch (TimeoutException e) {
			// 等待超时
			return null;
		}

		File dir = getStorageDir(key);
		dir = new File(dir, key + "." + fileExt + FileConverterSettings.PAGED_FILE_DIR_SUFFIX);
		
		File file = new File(dir, key + "." + page + "." + fileExt);
		if (file.exists() && file.isFile()) {
			return file;
		} else {
			logger.error("找不到{}的第{}页{}文件，文件路径：{}", key, page, fileExt, file.getAbsolutePath());
			
			// 尝试查找单文件是否存在
			file = getConvertedFile(key, outType);
			if (file != null && file.exists() && file.isFile()) {
				logger.info("未找到{}格式的分页文件，但找到了单个文件，返回该文件。", outType.toString());
				return file;
			} else {
				return null;
			}
		}
	}
	
	/**
	 * 按页获取转换后的文件数据流。通常用于图片，但如果转换器支持分页，也可以用于HTML等文件。
	 * @param key 文档key
	 * @param outType 要获取的文件格式
	 * @param page 第几页。基于0
	 * @return 文件流。如果不存在，返回null。
	 */
	public static InputStream getConvertedPageAsStream(final String key, OutputType outType, final int page) {
		File target = getConvertedPage(key, outType, page);
		return getFileAsStream(target);
	}
	
	/**
	 * 读取文件并以流的形式返回
	 * @param target 要读取的文件
	 * @return 文件流
	 */
	private static InputStream getFileAsStream(File target) {
		if (target != null && target.exists()) {
			try {
				return new FileInputStream(target);
			} catch (FileNotFoundException e) {
				logger.error("无法打开文件{}", target.getAbsolutePath());
				return null;
			}
		} else if (target != null && !target.exists()){
			logger.warn("找不到文件{}，是否已被手工从磁盘上删除？", target.getAbsolutePath());
			return null;
		} else {
			logger.warn("传入的文件对象为null！");
			return null;
		}
	}
	
	/**
	 * 获取转换后的文件个数。通常用于png这样的分页保存的多文件。<br>
	 * 如果在查找时刚好正在进行该格式的文件转换，则会等待直到转换完成。
	 * 如果目标格式文件不存在，且不处于正在转换状态，则返回0。
	 * @param key 文档key
	 * @param outType 目标格式
	 * @return 目标格式的文件个数。如果不存在则返回0。
	 */
	public static int getConvertedFilesCount(final String key, final OutputType outType)  {
		storageHelper.validateKey(key);
		
		// 查找该格式文件
		File targetFile = getConvertedFile(key, outType);
		if (targetFile == null) {
			return 0;
		} else if (targetFile.isFile()) {
			return 1;
		} else {
			// 目标格式放在子文件夹下
			Collection<File> list = FileUtils.listFiles(targetFile, new IOFileFilter() {

				// 文件名模式为：[key].[数字].[格式]
				private String matchStr = "^" + key + "\\.\\d+\\." + outType.toString() + "$";
				
				@Override
				public boolean accept(File file) {
					String filename = file.getName().toLowerCase(Locale.ENGLISH);
					
					if (filename.matches(matchStr)){
						return true;
					} else {
						return false;
					}
				}

				@Override
				public boolean accept(File dir, String name) {
					return false;
				}
			}, null);
			
			logger.info("找到{}张图片", list.size());
			if (list.size() == 0) {
				logger.error("找不到该格式文件！文档Key:{}，格式:{}", key, outType.toString());
				return 0;
			} else {
				return list.size();
			}
		}
	}
	
	/**
	 * 判断某个转换任务是否正在进行
	 * @param key 转换文件的key
	 * @param outType 要转换的目标类型
	 * @return 是否正在进行转换，true: 正在转换
	 */
	public static boolean isConverting(final String key, final OutputType outType) {
		// 通过检查lock文件查询转换任务是否正在进行
		return findLockFile(key, outType) != null;
	}
	
	/**
	 * 查找某个转换任务的锁文件。如果存在锁文件，意味着该任务正在进行中。
	 * @param key 被转换的文件key
	 * @param outType 转换目标类型
	 * @return 如果锁文件存在，则返回锁文件对象，否则返回null
	 */
	public static File findLockFile(final String key, final OutputType outType) {
		// 通过检查lock文件查询转换任务是否正在进行
		Iterator<File> iter = FileUtils.iterateFiles(settings.getIncomingDir(), new IOFileFilter() {

			@Override
			public boolean accept(File file) {
				String filename = file.getName().toLowerCase(Locale.ENGLISH);
				// 查找是否存在以文件Key开头，以目标格式+.lock结尾的文件。如果有，则认为正在转换。
				if (filename.startsWith(key) && filename.endsWith(outType.toString() + FileConverterSettings.SUFFIX_LOCK)) {
					// 检查是否是因意外原因遗漏的锁文件。检查方法是判断文件创建时间距离现在是否超过了两倍的转换超时设置时间。
					long modifyTime = file.lastModified();
					long now = System.currentTimeMillis();
					if (now - modifyTime > FileConverterSettings.CONVERT_LOCK_EXPIRED_MINUTES) {
						logger.info("文件转换锁存在，但存在时间已经超过预设阀值，判定为失效锁。key={}, type={}", key, outType.toString());
						return false;
					} else {
						return true;
					}
				} else {
					return false;
				}
			}

			@Override
			public boolean accept(File dir, String name) {
				return false;
			}
		}, null);
		
		if (iter.hasNext()) {
			return iter.next();
		} else {
			return null;
		}
	}
	
	/**
	 * 如果转换任务存在，则等待转换任务完成，否则直接返回。<br>
	 * 具体方法是查看任务锁文件是否存在，如果存在则每隔100毫秒轮询一次，直到锁消失或等待超时（超过设定的转换超时时间）。如果不存在则直接返回。
	 * @param key 被转换的文件key
	 * @param outType 转换目标类型
	 * @throws TimeoutException 如果等待超时，抛出异常
	 */
	public static void waitConvert(final String key, final OutputType outType) throws TimeoutException {
		File lockFile = findLockFile(key, outType);
		if (lockFile != null) {
			logger.info("发现文件锁存在。开始等待文件转换结束。文件Key：{}, 目标类型：{}", key, outType.toString());
			long start = System.currentTimeMillis();
			long end = System.currentTimeMillis();
			try {
				while (lockFile.exists() && (end - start) < FileConverterSettings.CONVERT_TASK_TIMEOUT_MINUTES * 60 *1000) {
						Thread.sleep(100);
						end += 100;
				}
			} catch (InterruptedException e) {
				logger.error("等待文件转换结束时出现异常！", e);
				Thread.currentThread().interrupt();
			}
			logger.info("文件锁等待结束");
			if (lockFile.exists()) {
				String errMsg = "等待释放文件锁超时(" + lockFile.getAbsolutePath() + ")。可能有其他的转换任务超时且未完成，请稍后再试，或手工删除文件锁后再试。";
				logger.error(errMsg);
				throw new TimeoutException(errMsg);
			}
		}
	}
	
	/**
	 * 在转换存储库中删除指定key的文件，包括所有格式。
	 * @param key 文件key
	 */
	public static void deleteConvertedFileQuietly(final String key) {
		if (key == null) {
			logger.warn("必须指定一个key才能删除！");
			return;
		}
		
		logger.info("删除文件开始：{}", key);
		historyLogger.info("删除 - {}", key);
		
		File dir = getStorageDir(key);
		if (dir.exists() && dir.isDirectory() && dir.canWrite()) {
			try {
				Collection<File> files = FileUtils.listFiles(dir, new IOFileFilter() {

					@Override
					public boolean accept(File file) {
						// 删除所有以该key开始的文件
						if (file.getName().startsWith(key)) {
							return true;
						} else {
							return false;
						}
					}

					@Override
					public boolean accept(final File dir, final String name) {
						return false;
					}
				}, null);

				for(File file : files) {
					FileUtils.deleteQuietly(file);
				}
			} catch (Exception e) {
				logger.error("删除转换后文件时发生异常!", e);
			}
		} else {
			logger.error("删除转换后文件时发生异常：文件夹不存在或无操作权限。{}", dir.getAbsolutePath());
		}
	}

	/**
	 * 根据文件key得到其存储路径
	 * @param key 文件key
	 * @return 最终的存储位置
	 * @throws IllegalArgumentException 如果传入的key无法被当前的文件存储器解析，则抛出异常
	 */
	private static File getStorageDir(final String key) throws IllegalArgumentException {
		String realtivePath = storageHelper.getRelativePathByKey(key);
		File targetDir = new File(settings.getStorageDir(), realtivePath);
		return targetDir;
	}

}
