package com.kdgcsoft.power.fileconverter;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.impl.ChainConverter;
import com.kdgcsoft.power.fileconverter.impl.CopyOnlyConverter;
import com.kdgcsoft.power.fileconverter.impl.DummyConverter;
import com.kdgcsoft.power.fileconverter.impl.Excel2HtmlConverter;
import com.kdgcsoft.power.fileconverter.impl.JodOfficeConverter;
import com.kdgcsoft.power.fileconverter.impl.MsOfficeConverter;
import com.kdgcsoft.power.fileconverter.impl.Pdf2HtmlConverter;
import com.kdgcsoft.power.fileconverter.impl.Pdf2ImgConverter;
import com.kdgcsoft.power.fileconverter.impl.WpsOfficeConverter;

/**
 * 转换器工厂类，用于创建特定文件类型之间转换的转换器。
 * @author hling
 *
 */
public class ConverterFactory {
	
	private static final Logger logger = LoggerFactory.getLogger(ConverterFactory.class);
	
	public static IFileConverter createConverter(final String srcFileName, final OutputType outType, final FileConverterSettings settings) {
		
		// 测试模式下始终使用虚拟转换器
		if (settings.isTestMode()) {
			return new DummyConverter();
		}
		
		String srcExtName = FilenameUtils.getExtension(srcFileName).toLowerCase();
		
		if (srcExtName.equals(outType.toString())) {
			return new CopyOnlyConverter();
		}
		
		if (OutputType.png.equals(outType)) {
			if ("pdf".equalsIgnoreCase(srcExtName)) {
				return new Pdf2ImgConverter();
			} else {
				ChainConverter stepConverter = ChainConverter.createConverter(srcFileName, settings, OutputType.pdf, OutputType.png);
				return stepConverter;
			}
		} else if (OutputType.html.equals(outType)) {
			if ("xls".equals(srcExtName) || "xlsx".equals(srcExtName)) {
				return new Excel2HtmlConverter();
			} else if ("pdf".equals(srcExtName)) {
				return new Pdf2HtmlConverter();				
			} else {
				// 先转成pdf，再转成html
				return ChainConverter.createConverter(srcFileName, settings, OutputType.pdf, OutputType.html);
			}
		} else if (OutputType.pdf.equals(outType)){
			// 使用基础转换引擎
			ConvertEngineType engine = settings.getConvertEngine();
			if (ConvertEngineType.openOffice.equals(engine)) {
				return new JodOfficeConverter();
			} else if (ConvertEngineType.msoffice.equals(engine)) {
				return new MsOfficeConverter();
			} else if (ConvertEngineType.wps.equals(engine)) {
				return new WpsOfficeConverter();
			} else {
				logger.error("不支持的转换引擎：{}", engine.toString());
				return null;
			}
		} else {
			logger.error("不可能出现的错误：代码中增加了转换目标类型定义枚举，但尚未实现？");
			return null;
		}
	}
}
