package com.kdgcsoft.power.fileconverter.storage;
import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.atomic.AtomicInteger;

import com.kdgcsoft.power.fileconverter.IFileStorageHelper;

/**
 * 基于时间戳的目录结构来存储转换后的文件管理器。
 * 使用时间戳+4位序号作为文件Key，并基于时间戳创建年/月/日三级目录层级的本地磁盘文件存储库
 * 其中，4位序号在同一秒内递增，每秒内的序号都从1开始。即最多支持一秒内存储9999个文件。
 * @author hling
 *
 */
public class TimeStampStorage implements IFileStorageHelper {

	private static final String KEY_FORMAT = "yyyyMMddHHmmss";
	private static final int LOOP_SIZE = 10000;
	private static final String SEQ_FORMAT = "%04d";
	private static final int KEY_LENGTH = KEY_FORMAT.length() + 4;
	
	/**
	 * 原子计数器，在同一秒内递增
	 */
	private AtomicInteger id = new AtomicInteger(0);
	/**
	 * 记录上一次产生计数的秒值
	 */
	private long lastSecond = 0;
	
	@Override
	public String generateKey() {
		Date date = new Date();
		// 在同一秒内循环计数
		long curSecond = date.getTime() / 1000;
		if (curSecond != lastSecond) {
			id.set(0);
			lastSecond = curSecond;
		}
		SimpleDateFormat sdf = new SimpleDateFormat(KEY_FORMAT);
		return sdf.format(date) + String.format(SEQ_FORMAT, getNextSeq());
	}

	@Override
	public String getRelativePathByKey(final String key) {
		validateKey(key);
		StringBuffer buf = new StringBuffer(10);
		buf.append(key.subSequence(0, 4)).append(File.separator)
				.append(key.subSequence(4, 6)).append(File.separator)
				.append(key.subSequence(6, 8));
		return buf.toString();
	}

	@Override
	public void validateKey(final String key) throws IllegalArgumentException {
		if (key == null) {
			throw new IllegalArgumentException("Key不能为null!");
		}
		
		if (key.length() != KEY_LENGTH) {
			throw new IllegalArgumentException("Key应该是" + KEY_LENGTH + "位整数");
		}
	}

	/**
	 * 获取全局唯一的序列数，自动递增。当超过100000时会重新从1开始
	 * @return 下一个序列数
	 */
	private int getNextSeq() {
		if (id.get() >= LOOP_SIZE) {
			id.set(0);
		}
		return id.incrementAndGet();
	}
}
