package com.kdgcsoft.power.fileconverter.storage;
import java.io.File;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;

import com.kdgcsoft.power.fileconverter.IFileStorageHelper;

/**
 * 基于任意Key字符串实现的的本地文件管理方式。 <br>
 * 这种存储方式的使用场景是应用程序想以自己的方式管理文件Key(例如使用数据库序列），并且希望文档转换库的操作接口也使用这个Key。
 * 这种方式下文档转换库内部将依据Key字符串（必须是英文字母不区分大小写或数字）的最后两个字符建立两级目录（使用小写字母）。<br>
 * 第一级目录使用key的倒数第2个字符作为目录名，第二级目录使用最后1个字符。<br>
 * 如果key长度不满2，则前补"$"到2个字符再建立目录。比如key为"A"时，前补$成为"$A"，建立的目录结构为 "$/a"。
 * 最终在不考虑$存在的情况下每级目录为0-9、a-z最多36个目录，所有文件分配在36*36个目录下存储。
 * @author hling
 *
 */
public class FreeStyleStorage implements IFileStorageHelper {

	@Override
	public String generateKey() {
		return StringUtils.remove(UUID.randomUUID().toString(), '-');
	}

	/* 
	 * 使用任意格式（英文字母或数字）字符串Key的最后两个字符建立两级目录，不区分大小写。<br>
	 * 第一级目录使用key的倒数第2个字符作为目录名，第二级使用最后1个字符。<br>
	 * 如果key长度不满2，则前补字符$到2个字符。比如key为"A"时，前补$成为"$A"，建立的目录结构为 "$/a"。
	 * (non-Javadoc)
	 * @see com.kdgcsoft.power.doc.convert.IFileStorageHelper#getRelativePathByKey(java.lang.String)
	 */
	@Override
	public String getRelativePathByKey(final String key) {
		validateKey(key);
		
		String lowCasekey = key.toLowerCase();
		if (lowCasekey.length() == 1) {
			return "$" + File.separator + lowCasekey.charAt(lowCasekey.length() - 1);
		} else {
			return lowCasekey.charAt(lowCasekey.length() - 2) + File.separator + lowCasekey.charAt(lowCasekey.length() - 1);
		}
	}

	@Override
	public void validateKey(final String key) throws IllegalArgumentException {
		if (!StringUtils.isAlphanumeric(key)) {
			throw new IllegalArgumentException("Key必须是数字或英文字母，且至少有一位！当前Key为：" + key);
		}
	}

}
