package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;
import com.kdgcsoft.power.fileconverter.util.StreamGobbler;
import com.sun.star.lang.IllegalArgumentException;

/**
 * 基于<a href="https://github.com/coolwanglu/pdf2htmlEX">PDF2HtmlEX</a>封装的PDF->HTML转换器。<br>
 * 转换生成的HTML文件是单文件，需要浏览器支持HTML5。<br>
 * <a href="http://soft.rubypdf.com/software/pdf2htmlex-windows-version">Windows版下载地址</a> 
 * <a href="https://github.com/coolwanglu/pdf2htmlEX/wiki/Download"> 全部下载地址</a> <br>
 * PDF2HtmlEX是在外部运行的独立程序，本转换器仅仅调用它进行转换。
 * @author hling
 *
 */
public class Pdf2HtmlConverter implements IFileConverter {
	
	private static Logger logger = LoggerFactory.getLogger(Pdf2HtmlConverter.class);
	
	@Override
	public File convert(File srcFile, File destFile, final FileConverterSettings settings) throws Exception {
		String cmdPath = settings.getPdf2HtmlExePath();
		pdf2html(cmdPath, srcFile, destFile);
		return destFile;
	}

	/**
	 * 调用pdf2htmlEX将pdf文件转换为html文件
	 * 
	 * @param exeFilePath pdf2htmlEX可执行文件路径
	 * @param srcFile pdf源文件
	 * @param destFile 转换目标html文件
	 * @throws FileConverterException 当出现异常时返回异常对象。可以用getCause()查看原始异常信息。
	 */
	public static void pdf2html(String exeFilePath, File srcFile, File destFile) throws FileConverterException {
		File file = new File(exeFilePath);
		if (exeFilePath == null || !file.exists() || !file.isFile() || !file.canExecute()) {
			logger.error("Pdf2HtmlEx主程序路径设置不正确或无权限执行：{}", exeFilePath);
			throw new IllegalArgumentException("找不到Pdf2HtmlEx可执行程序或无权限执行！");
		}
		
		if (srcFile == null || !srcFile.exists() || !srcFile.isFile() 
				|| !"pdf".equals(FilenameUtils.getExtension(srcFile.getName()))) {
			logger.error("无法处理的源文件：{}" + (srcFile==null?"null":srcFile.getAbsolutePath()));
			throw new IllegalArgumentException("要转换的源文件不存在或不是PDF文件！");
		}
		Runtime rt = Runtime.getRuntime();
		StringBuilder command = new StringBuilder();
		command.append(exeFilePath).append(" ");
		
		// 生成文件存放位置,需要替换文件路径中的空格
		command.append(" --dest-dir ").append("\"" + destFile.getParentFile().getAbsolutePath() + "\"");
		command.append(" --zoom 1.4 ");
		command.append(" --hdpi 288 ");
		command.append(" --vdpi 288 ");
		
		command.append(" \"" + srcFile.getAbsolutePath() + "\"");
		command.append(" \"" + destFile.getName() + "\"");
		try {
			logger.info("PDF2HtmlEx Command：" + command.toString());
			Process p = rt.exec(command.toString());
			StreamGobbler errorGobbler = new StreamGobbler(logger, p.getErrorStream(), "");
			// 开启屏幕标准错误流
			errorGobbler.start();
			StreamGobbler outGobbler = new StreamGobbler(logger, p.getInputStream(), "");
			// 开启屏幕标准输出流
			outGobbler.start();
			int w = p.waitFor();
			int v = p.exitValue();
			if (w == 0 && v == 0) {
				return;
			} else {
				throw new FileConverterException("PDF2HtmlEx程序返回值不为0，可能出现错误，请查看日志分析原因");
			}
		} catch (Exception e) {
			logger.error("调用PDF2HtmlEx程序异常", e.getMessage());
			throw new FileConverterException("调用PDF2HtmlEx程序异常", e);
		}
	}
	
}
