package com.kdgcsoft.power.fileconverter.impl;

import java.io.File;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.jacob.activeX.ActiveXComponent;
import com.jacob.com.ComThread;
import com.jacob.com.Dispatch;
import com.jacob.com.Variant;
import com.kdgcsoft.power.fileconverter.FileConverterException;
import com.kdgcsoft.power.fileconverter.FileConverterSettings;
import com.kdgcsoft.power.fileconverter.IFileConverter;

/**
 * 基于Jacob的转换器，将office文档转为pdf文件
 *
 */
public class MsOfficeConverter implements IFileConverter {

	private static Logger logger = LoggerFactory.getLogger(MsOfficeConverter.class);
	
	private static final int wdFormatPDF = 17; //PDF 格式
	private static final int xlTypePDF = 0;  
    private static final int ppSaveAsPDF = 32; 
    
	@Override
	public File convert(final File srcFile, final File destFile, final FileConverterSettings settings) throws FileConverterException{
        if(!srcFile.exists()){  
        	logger.info("{}该文件不存在!", srcFile.getName());            
        	
        }else{
        	String suffix =  FilenameUtils.getExtension(srcFile.getName());
	        if(suffix.equals("pdf")){
	            logger.info("{}PDF not need to convert!", srcFile.getName());
	        }  
	        if(suffix.equals("doc")||suffix.equals("docx")||suffix.equals("txt")){  
	            word2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());  
	        }else if(suffix.equals("ppt")||suffix.equals("pptx")){  
	            ppt2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());  
	        }else if(suffix.equals("xls")||suffix.equals("xlsx")){  
	            excel2PDF(srcFile.getAbsolutePath(),destFile.getAbsolutePath());  
	        }else{
	            logger.error("{}文件格式不支持转换!", srcFile.getName());
	            throw new FileConverterException(srcFile.getName()+"文件格式不支持转换!");
	        }  
        }
        
        return destFile;

	}

	/**
	 * Word文档转为PDF
	 * @param inputFile
	 * @param pdfFile
	 * @throws FileConverterException
	 */
	public void word2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch doc = null;
		try {
			ComThread.InitMTA(true);
			// 打开word应用程序
			app = new ActiveXComponent("Word.Application");
			// 设置word不可见
			app.setProperty("Visible", false);
			// 获得word中所有打开的文档,返回Documents对象
			Dispatch docs = app.getProperty("Documents").toDispatch();
			// 调用Documents对象中Open方法打开文档，并返回打开的文档对象Document
			doc = Dispatch.call(docs, "Open", inputFile, false, false).toDispatch();
			// 调用Document对象的SaveAs方法，将文档保存为pdf格式
			deletePdfFile(pdfFile);
			// Dispatch.call(doc, "SaveAs", pdfFile, wdFormatPDF);
			//word保存为pdf格式宏，值为17 );
			 
			Dispatch.call(doc, "ExportAsFixedFormat", pdfFile, wdFormatPDF );// word保存为pdf格式宏，值为17
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+"文件转换失败!", e);
		}finally {
			if (doc != null) {
				// 关闭文档
                Dispatch.call(doc, "Close", true);  
                doc.safeRelease();
            }
            if (app != null) {  
            	// 关闭word应用程序
    			app.invoke("Quit", 0);  
    			app.safeRelease();
            }
            ComThread.Release(); //释放进程
            ComThread.quitMainSTA();
        }  
	}

	/**
	 * Excel文档转换为PDF
	 * @param inputFile Excel文件
	 * @param pdfFile 目标PDF文件
	 * @throws FileConverterException
	 */
	public void excel2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch excel=null;
		try {
			ComThread.InitMTA(true);
			app = new ActiveXComponent("Excel.Application");
			app.setProperty("Visible", false);
			app.setProperty("AutomationSecurity", new Variant(3)); //禁用宏
			Dispatch excels = app.getProperty("Workbooks").toDispatch();
			excel = Dispatch.call(excels, "Open", inputFile, false, true).toDispatch();
			deletePdfFile(pdfFile);
			Dispatch.call(excel, "ExportAsFixedFormat",xlTypePDF,pdfFile);
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+"文件转换失败!", e);
		}finally {
			if (excel != null) {  
                Dispatch.call(excel, "Close", false);
                excel.safeRelease();
            }
            if (app != null) {  
            	// 关闭word应用程序
    			app.invoke("Quit");  
    			app.safeRelease();
            }
            ComThread.Release(); //释放进程
            ComThread.quitMainSTA();
        } 
	}
	
	/**
	 * PowerPoint文档转换为PDF
	 * @param inputFile 源PPT文件
	 * @param pdfFile 目标PDF文件
	 * @throws FileConverterException
	 */
	public void ppt2PDF(String inputFile, String pdfFile) throws FileConverterException {
		ActiveXComponent app=null;
		Dispatch ppt=null;
		try {
			ComThread.InitMTA(true);
			app = new ActiveXComponent("PowerPoint.Application");
			Dispatch ppts = app.getProperty("Presentations").toDispatch();

			ppt = Dispatch.call(ppts, "Open", inputFile, true,// ReadOnly
					true,// Untitled指定文件是否有标题
					false// WithWindow指定文件是否可见
					).toDispatch();
			deletePdfFile(pdfFile);
			Dispatch.call(ppt, "SaveAs", pdfFile, ppSaveAsPDF);
			return;
		} catch (Exception e) {
	         throw new FileConverterException(inputFile+"文件转换失败!", e);
		}finally {
			if (ppt != null) {  
                Dispatch.call(ppt, "Close");
                ppt.safeRelease();
            }
            if (app != null) {  
            	// 关闭word应用程序
    			app.invoke("Quit");  
    			app.safeRelease();
            }
            ComThread.Release(); //释放进程
            ComThread.quitMainSTA();
            
        } 
	}
	/**
	 * 根据路径检查文件是否存在,存在则删除,MSoffice转PDF 如果文件存在会报错,所以转换前检查目标文件是否存在,存在则删除
	 */
	public void deletePdfFile(String pdfFile) {
		File tofile = new File(pdfFile);
		if (tofile.exists()) {
		   logger.info("{}文件已存在,删除后重新生成!", pdfFile);   
           tofile.delete();
		}
		
	}
	public static void main(String[] args) throws FileConverterException {
		MsOfficeConverter conver =new MsOfficeConverter();
		conver.excel2PDF("D:/666.xls", "D:/666.pdf");
	}
}
