package com.kdgcsoft.power.fileconverter;

import java.io.File;
import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.excel2html.HtmlTemplate;

public class FileConverterSettings {
	
	private static final Logger logger = LoggerFactory.getLogger(FileConverterService.class);
	
	/**
	 * 标志一个任务正在进行的锁文件扩展名
	 */
	public static final String SUFFIX_LOCK = ".lock";
	
	/**
	 * 记录转换任务信息的临时文件的扩展名。<br>
	 * 例：当存在一个doc转pdf的任务时，会生成一个空的临时文件：[key].doc.pdf.task，文件名包含了任务的信息。
	 * 任务成功完成后，该文件会被自动删除。
	 */
	public static final String TASK_FILE_EXT = "task";
	
	/**
	 * 储存分页文件的子目录后缀。最终为：[key].[OutputType].dir。例: 201804101949360001.png.dir
	 */
	public static final String PAGED_FILE_DIR_SUFFIX = ".dir";
	
	/**
	 * 转换任务的超时时间，单位：分钟
	 */
	public static final int CONVERT_TASK_TIMEOUT_MINUTES = 5;
	
	/**
	 * 判定转换锁的失效时间。当锁文件创建时间距离现在超过这个值时，则认为转换锁已经无效，可以再次进行转换。单位：分钟
	 * 这样避免当因非正常退出程序时遗留转换文件锁没有被删除，从而导致无法重新转换的问题。
	 */
	public static final int CONVERT_LOCK_EXPIRED_MINUTES = 2 * CONVERT_TASK_TIMEOUT_MINUTES * 60 * 1000;
	
	/**
	 * 主工作目录
	 */
	private File workdir = null;
	/**
	 * 临时存放转换源文件、任务记录文件、转换锁文件的工作目录
	 */
	private File incomingDir = null;
	/**
	 * 存储转换后文件的根目录
	 */
	private File storageDir = null;
	/**
	 * 是否测试模式。当为true时将使用内置虚拟转换器模拟所有转换过程，创建0字节的目标文件，实际不做转换。
	 * 从而能够在单元测试时不依赖任何外部程序环境地测试整体逻辑。
	 * 缺省为false。
	 */
	private boolean isTestMode = false;
	/**
	 * 可同时进行的并发转换任务
	 */
	private int maxThreadNumber = 5;
	/**
	 * 基础转换引擎
	 */
	private ConvertEngineType convertEngine = ConvertEngineType.msoffice;
	/**
	 * 文件存储目录的存储结构。缺省为基于时间戳的存储目录结构。
	 */
	private StorageType storType = StorageType.TimeStamp;	
	/**
	 * OpenOffice的安装目录
	 */
	private String openOfficePath = null;
	/**
	 * OpenOffice的监听端口
	 */
	private int[] openOfficePorts = new int[]{45111, 45112};
	/**
	 * PDF2HtmlEx可执行程序的路径，包含可执行文件名。例如："C:\\pdf2htmlEx\\pdf2htmlEX.exe"
	 */
	private String pdf2HtmlExePath = "D:\\pdf2htmlEX\\pdf2htmlEX.exe";
	/**
	 * 是否在转换文档存储库中保存源文件。如果保存的话，会多占用磁盘空间，但方便运维查错。建议在调试初期启用。
	 * 缺省为true：保存源文件。
	 */
	private boolean alsoStorageOriginalFile = true;
	/**
	 * Excel2Html转换时是否显示行列坐标。默认为不显示。
	 */
	private boolean excel2html_showHeaders = false;
	/**
	 * Excel2Html转换时是否在HTML中隐藏Excel中不可见的行列。默认为隐藏。
	 */
	private boolean excel2html_showHiddenContent = false;
	/**
	 * Excel2Html转换时，如果只有一个Sheet，则是否隐藏该Sheet的Tab标签。默认为不隐藏，即总是显示Tab标签。
	 */
	private boolean excel2html_hideTabIfOnlyOne = false;
	/**
	 * 定制Excel2Html输出的Html模板字符串。
	 */
	private String excel2html_htmlTemplateStr = null;
	/**
	 * 定制Excel2Html输出的Html模板对象。为了提高性能，在设置模板字符串后转换成对象保存，重复使用。
	 */
	private HtmlTemplate excel2html_htmlTemplate = null;

	public FileConverterSettings(){};
	
	/**
	 * 获取文档转换工作目录
	 * @return 工作目录
	 */
	public File getWorkdir() {
		return workdir;
	}

	/**
	 * 设置文档转换的工作目录
	 * @param dir 工作目录
	 * @throws IOException
	 */
	public void setWorkdir(final File dir) throws IOException {
		this.workdir = dir;
		this.incomingDir = new File(workdir, "incoming");
		this.storageDir = new File(workdir, "storage");
	}
	
	/**
	 * 获取工作目录下用于存储临时文件的目录。该目录用于暂时存放转换源文件、转换任务记录、任务锁
	 * @return 临时目录
	 */
	public File getIncomingDir() {
		return this.incomingDir;
	}

	/**
	 * 获取工作目录下用于存储转换后文件的根目录
	 * @return 存储转换后文件的根目录
	 */
	public File getStorageDir() {
		return this.storageDir;
	}

	/**
	 * 获取最大并发转换数
	 * @return 最大并发转换数
	 */
	public int getMaxConvertThread() {
		return this.maxThreadNumber;
	}

	/**
	 * 设置最大并发转换数。默认为5。
	 * @param threadNumber 最大并发数
	 */
	public void setMaxConvertThread(final int threadNumber) {
		this.maxThreadNumber = threadNumber;
	}

	/**
	 * 获取当前的基础转换引擎。
	 * @return 当前转换引擎
	 */
	public ConvertEngineType getConvertEngine() {
		return this.convertEngine;
	}

	/**
	 * 设置基础转换引擎。默认为jacob
	 * @param convertEngine
	 */
	public void setConvertEngine(final ConvertEngineType convertEngine) {
		this.convertEngine = convertEngine;
	}

	/**
	 * 获取存储目录结构组织方式
	 * @return 录结构组织方式
	 */
	public StorageType getStorageType() {
		return this.storType;
	}

	/**
	 * 设置存储目录结构组织方式。默认为按时间戳组织目录结构
	 * @param storType
	 */
	public void setStorageType(final StorageType storType) {
		this.storType = storType;
	}
	
	/**
	 * 获得当前设置的LibreOffice/OpenOffice的安装目录。如果未设置，则返回null，转换器将使用默认路径查找LibreOffice/OpenOffice。
	 * @return LibreOffice/OpenOffice安装路径
	 */
	public String getOpenOfficePath() {
		return this.openOfficePath;
	}

	/**
	 * 当使用OpenOffice/LibreOffice作为缺省转换引擎时，如果没有安装在默认路径下，则需要指定安装路径。
	 * 如果安装在默认路径下，不需要调用本函数。
	 * @param path OpenOffice/LibreOffice的安装目录
	 */
	public void setOpenOfficePath(final String path) {
		File file = new File(path);
		if (file.exists() && file.isDirectory()) {
			this.openOfficePath = path;
		} else {
			throw new IllegalArgumentException("OpenOffice安装目录设置错误，找不到目录：" + path);
		}
	}

	/**
	 * 获取所设置的OpenOffice/LibreOffice监听端口。默认为45111和45112。
	 * @return 监听端口数组
	 */
	public int[] getOpenOfficePorts() {
		return this.openOfficePorts.clone();
	}

	/**
	 * 当使用OpenOffice/LibreOffice作为缺省转换引擎时，可以指定其默认的监听端口（用于支持并发）。
	 * 不设置的话，默认端口只有2个，端口号为45111和45112。并发时可能会因为等待端口连接超时而导致转换任务失败。
	 * 根据测试，在Windows中每启用一个端口大约占据50M操作系统内存（不是JVM内存！）
	 * 注意：请确保端口没有被其他程序占用。如果被占用，会引起运行错误。
	 * @param listenPorts 监听的端口号数组
	 */
	public void setOpenOfficePorts(final int[] listenPorts) {
		this.openOfficePorts = listenPorts.clone();
	}

	/**
	 * 获取PDF2HtmlEx可执行程序的全路径，包含文件名。
	 * @return 可执行文件的路径
	 */
	public String getPdf2HtmlExePath() {
		return this.pdf2HtmlExePath;
	}

	/**
	 * 当希望使用PDF2HtmlEx作为PDF转HTML引擎时，需要指定PDF2HtmlEx可执行文件的位置，例如："C:\\pdf2htmlEx\\pdf2htmlEX.exe"。<br>
	 * 如果没有设置这个路径，转换器将使用通过{@link #setConvertEngine(ConvertEngineType)}设置的基础转换引擎尝试转换PDF为HTML（有可能不支持）。
	 * 默认路径为：D:\\pdf2htmlEX\\pdf2htmlEX.exe
	 * @param path PDF2HtmlEx可执行文件的路径
	 */
	public void setPdf2HtmlExePath(final String path) {
		File file = new File(path);
		if (file.exists() && file.isFile() && file.canExecute()) {
			this.pdf2HtmlExePath = path;
		} else {
			logger.error("PDF2HtmlEx可执行程序路径设置错误：{}", path);
			throw new IllegalArgumentException("PDF2HtmlEx可执行程序路径设置错误：" + path);
		}
	}
	
	/**
	 * Excel2Html转换时是否显示行列坐标。默认为不显示。
	 * @return 是否显示行列坐标
	 */
	public boolean isExcel2HtmlShowHeaders() {
		return excel2html_showHeaders;
	}

	/**
	 * Excel2Html转换时是否显示行列坐标。默认为不显示
	 * @param excel2html_showHeaders 是否显示行列坐标
	 */
	public void setExcel2HtmlShowHeaders(final boolean excel2html_showHeaders) {
		this.excel2html_showHeaders = excel2html_showHeaders;
	}

	/**
	 * Excel2Html转换时是否在HTML中隐藏Excel中隐藏的行列。默认为隐藏
	 * @return 是否在HTML中隐藏Excel中隐藏的行列
	 */
	public boolean isExcel2htmlShowHiddenContent() {
		return excel2html_showHiddenContent;
	}

	/**
	 * Excel2Html转换时是否在HTML中隐藏Excel中的隐藏内容。默认为隐藏。
	 * @see com.kdgcsoft.power.excel2html.Excel2Html#setShowHiddenContent(boolean)
	 * @param excel2html_showHiddenContent
	 */
	public void setExcel2HtmlShowHiddenContent(final boolean excel2html_showHiddenContent) {
		this.excel2html_showHiddenContent = excel2html_showHiddenContent;
	}

	/**
	 * Excel2Html转换时，如果只有一个Sheet，则是否隐藏该Sheet的Tab标签。默认为不隐藏，即总是显示Tab。
	 * @return 是否隐藏唯一Sheet的Tab标签
	 */
	public boolean isExcel2HtmlHideTabIfOnlyOne() {
		return excel2html_hideTabIfOnlyOne;
	}

	/**
	 * Excel2Html转换时，如果只有一个Sheet，则是否隐藏该Sheet的Tab标签。默认为不隐藏，即总是显示Tab。
	 * @see com.kdgcsoft.power.excel2html.Excel2Html#setHideSheetTabIfOnlyOne(boolean)
	 * @param excel2html_hideTabIfOnlyOne 是否隐藏唯一Sheet的Tab标签
	 */
	public void setExcel2HtmlHideTabIfOnlyOne(final boolean excel2html_hideTabIfOnlyOne) {
		this.excel2html_hideTabIfOnlyOne = excel2html_hideTabIfOnlyOne;
	}

	/**
	 * 当前Excel2Html输出的Html模板字符串。如果没有设置，则返回null。此时Excel2Html会使用自带模板。
	 * @return 当前设置的Html模板字符串
	 */
	public String getExcel2HtmlTemplateStr() {
		return excel2html_htmlTemplateStr;
	}

	/**
	 * 定制Excel2Html输出的Html模板字符串。一般情况下Excel2Html自带模板就可以了。
	 * @see com.kdgcsoft.power.excel2html.Excel2Html#setHtmlTemplateString(String)
	 * @param excel2html_htmlTemplate 自定义Html模板。
	 */
	public void setExcel2HtmlTemplateStr(final String excel2html_htmlTemplate) {
		if (excel2html_htmlTemplate != null && !"".equals(excel2html_htmlTemplate)) {
			this.excel2html_htmlTemplateStr = excel2html_htmlTemplate;
			this.excel2html_htmlTemplate = new HtmlTemplate(excel2html_htmlTemplate);
		}
	}
	
	/**
	 * 当前Excel2Html输出的Html模板对象。如果没有设置，则返回null。此时Excel2Html会使用自带模板。
	 * @return Html模板对象
	 */
	public HtmlTemplate getExcel2HtmlTemplate() {
		return excel2html_htmlTemplate;
	}
	
	/**
	 * 返回当前是否在转换后的文档存储库中保存源文件的开关状态。缺省为true。
	 * @return true：保存源文件,  false：不保存
	 */
	public boolean isAlsoStorageOriginalFile() {
		return this.alsoStorageOriginalFile;
	}

	/**
	 * 设置是否在转换后的文档存储库中保存源文件。缺省为true。
	 * @param alsoStorageOriginalFile 是否保存源文件，true：保存，false：不保存 
	 */
	public void setAlsoStorageOriginalFile(final boolean alsoStorageOriginalFile) {
		this.alsoStorageOriginalFile = alsoStorageOriginalFile;
	}
	
	/**
	 * 设置为测试模式。此模式下将无视设置的转换引擎，总是使用虚拟转换器{@link com.kdgcsoft.power.fileconverter.impl.DummyConverter} 进行转换。
	 * @return 是否为测试模式。true:测试模式, false:非测试模式。默认是非测试模式。
	 */
	public boolean isTestMode() {
		return isTestMode;
	}

	/**
	 * 设置为测试模式。此模式下将无视设置的转换引擎，总是使用虚拟转换器{@link com.kdgcsoft.power.fileconverter.impl.DummyConverter} 进行转换。<br>
	 * 默认为非测试模式。
	 */
	public void setTestMode(final boolean isTestMode) {
		if (isTestMode) {
			logger.warn("----------------当前转换模式为测试模式，只创建0字节目标文件，不做真正转换！-------------------");
			this.isTestMode = isTestMode;
		}
	}

}
