package com.kdgcsoft.power.fileconverter;

import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.concurrent.Callable;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kdgcsoft.power.fileconverter.util.FileExtensionUtil;

/**
 * 封装转换任务的类，用于多线程执行。
 * 
 * @author hling
 * 
 */
class ConvertTask implements Callable<File> {
	
	private static final String TEMP_FILE_PREFIX = "$temp$";

	private static final Logger logger = LoggerFactory.getLogger(ConvertTask.class);

	private File incomingFile;
	private File targetDir;	
	private OutputType outType;
	
	private FileConverterSettings settings;
	
	private final static Object lock = new Object();
	
	/**
	 * 构造一个转换任务
	 * @param srcFile 待转换的源文件
	 * @param targetDir 转换后文件保存的目录（不包含文件名）
	 * @param outType 转换目标类型
	 * @param settings 转换设置
	 */
	public ConvertTask(final File srcFile, final File targetDir, final OutputType outType, final FileConverterSettings settings) {
		this.incomingFile = srcFile;
		this.targetDir = targetDir;
		this.outType = outType;
		this.settings = settings;
	}
	
	@Override
	public File call() throws Exception {
		File infoFile = new File(settings.getIncomingDir(), FileExtensionUtil.addExtension(
				incomingFile.getName(), outType.toString() + "." + FileConverterSettings.TASK_FILE_EXT));
		// 创建记录文件，文件名中包含转换格式信息，用于转换失败时重做。
		try {
			FileUtils.touch(infoFile);
		} catch (IOException e) {
			logger.warn("创建任务定义文件异常，在转换失败时将无法恢复转换任务");
		}
		
		logger.info("开始转换文件：{}，目标格式：{}", incomingFile.getAbsolutePath(), outType.toString());
		File result = null;
		try {
			long start = System.currentTimeMillis();
			result = doConvert();
			long end = System.currentTimeMillis();
			logger.info("转换{}为{}格式完毕！转换结果：{}，耗时：{}秒", incomingFile.getName(), outType.toString(), result, (end-start)/1000);

			// 转换成功，删除任务定义文件。如果转换过程抛出了异常，任务定义文件会遗留下来。
			// 目前转换过程返回的布尔值永远是true。
			FileUtils.deleteQuietly(infoFile);

			return result; 
		} catch (Exception e) {
			logger.error("转换失败！", e);
			return null;
		}
	}

	/**
	 * 执行真正的转换：先加锁，再转换，然后释放锁。
	 * 
	 * @return 转换器返回的文件对象，可能是文件或文件夹（如果转换结果是多个分页文件）。
	 * @throws Exception
	 */
	private File doConvert() throws Exception {

		IFileConverter converter = ConverterFactory.createConverter(incomingFile.getName(), outType, settings);
		
		String format = outType.toString();
		File destFile = new File(targetDir, FileExtensionUtil.replaceExtension(incomingFile.getName(), format));
		
		// 创建锁文件，指示文件正在处理中。锁文件有文件ID、转换目标类型、".lock"扩展名三部分组成
		File lockFile = tryLock(incomingFile, destFile, outType);
		if (lockFile == null) {
			// 目标文件在加锁过程中被另一个线程转换完成，不需要继续处理
			return destFile;
		}

		// 函数返回的文件对象
		File finalResult = null;
		File tempConvertedFile = null;
		try {
			File tempFile = new File(destFile.getParent(), TEMP_FILE_PREFIX + destFile.getName());
			FileUtils.deleteQuietly(tempFile);//删除目标文件,重新转换
			tempConvertedFile = converter.convert(incomingFile, tempFile, settings);
			finalResult = renameTempFileOrDir(tempConvertedFile);
		} catch (Throwable e) { // 注意这里必须是Throwable，确保捕获任何错误，从而确保删除锁和临时文件
			logger.error("转换异常", e);
			FileUtils.deleteQuietly(lockFile);
			// 删除转换失败的中间文件（一般都是不完整文件）
			if (tempConvertedFile != null) {
				if (tempConvertedFile.isDirectory()) {
					FileUtils.deleteDirectory(tempConvertedFile);
				} else {
					FileUtils.deleteQuietly(tempConvertedFile);
				}
			}
			
			throw e;
		}
		
		try {
			// 转换成功时，需要移动或删除incoming下面的源文件。保留源文件的目的是为了便于运维、查错或手动转换。
			// 在级联转换时初始任务（即父任务）的incomingFile会被第一个子任务移走，故初始任务结束时此文件已不存在，需要加判断
			if (settings.isAlsoStorageOriginalFile() && incomingFile.exists()) {
				// 把源文件移动到目标目录下。如果目标目录下已经存在该文件且大小相同则什么都不做。
				File srcFileBackup = new File(targetDir, incomingFile.getName());
				if (srcFileBackup.exists()) {
					if (FileUtils.sizeOf(srcFileBackup) != FileUtils.sizeOf(incomingFile)) {
						logger.warn("文件存储区已经存在源文件，但大小不一样！将使用新的源文件覆盖");
						FileUtils.deleteQuietly(new File(targetDir, incomingFile.getName()));
						FileUtils.moveFileToDirectory(incomingFile, targetDir, true);
					} else if (!srcFileBackup.getAbsolutePath().equals(incomingFile.getAbsolutePath())){
						FileUtils.deleteQuietly(incomingFile);
					}
				} else {
					FileUtils.moveFileToDirectory(incomingFile, targetDir, true);
				}
			} else {
				FileUtils.deleteQuietly(incomingFile);
			}
		} catch (IOException e) {
			logger.error("移动源文件到目标文件夹失败", e);
		} finally {
			// 删除锁文件
			FileUtils.deleteQuietly(lockFile);
		}

		return finalResult;
	}

	/**
	 * 检查转换器转换成功后返回的文件对象，如果是临时文件，则进行重命名，成为正式文件
	 * @param tempConvertedFile 转换器返回的文件对象，可能是文件或文件夹
	 * @return 经过重命名的正式文件或文件夹对象
	 * @throws IOException 文件读写错误
	 */
	private File renameTempFileOrDir(File tempConvertedFile) throws IOException {
		File finalResult = null;
		
		if (tempConvertedFile.isFile()) {
			logger.info("转换后临时文件：{}", tempConvertedFile.getAbsolutePath());
			// 生成成功后，把临时文件改名
			if (tempConvertedFile.getName().startsWith(TEMP_FILE_PREFIX)) {
				// 去掉前缀
				finalResult = new File(tempConvertedFile.getParentFile(), tempConvertedFile.getName().replace(TEMP_FILE_PREFIX, ""));
				FileUtils.deleteQuietly(finalResult);
				FileUtils.moveFile(tempConvertedFile, finalResult);	
				logger.info("转换后文件：{}", finalResult.getAbsolutePath());
			} else {
				finalResult = tempConvertedFile;
			}
		} else {
			logger.info("转换后文件夹：{}", tempConvertedFile.getAbsolutePath());
			if (tempConvertedFile.getName().startsWith(TEMP_FILE_PREFIX)) {
				// 生成成功后，把临时文件夹改名
				finalResult = new File(tempConvertedFile.getParentFile(), tempConvertedFile.getName().replace(TEMP_FILE_PREFIX, ""));
				if (finalResult.exists()) {
					if (finalResult.isDirectory()) {
						FileUtils.deleteDirectory(finalResult);
					} else {
						FileUtils.deleteQuietly(finalResult);
					}
				}
				FileUtils.moveDirectory(tempConvertedFile, finalResult);
			} else {
				finalResult = tempConvertedFile;
			}
			
			// 把临时文件夹下面的临时文件名去掉临时文件修饰
			Iterator<File> iter = FileUtils.iterateFiles(finalResult, new IOFileFilter() {

				@Override
				public boolean accept(File file) {
					if (file.getName().startsWith(TEMP_FILE_PREFIX)) {
						return true;
					} else {
						return false;
					}
				}

				@Override
				public boolean accept(File dir, String name) {
					return false;
				}
				
			}, null);
			
			while(iter.hasNext()) {
				File temp = iter.next();
				FileUtils.moveFile(temp, new File(temp.getParentFile(), temp.getName().replace(TEMP_FILE_PREFIX, "")));
			}
		}
		return finalResult;
	}
	
	/**
	 * 对当前转换加锁。任何时候，源文件Key、源文件扩展名、转换目标扩展名相同的锁只有一个，即同一时刻只有一个转换任务进行。
	 * 
	 * @param srcFile 源文件
	 * @param destFile 目标文件
	 * @param outType 目标扩展名
	 * @return 如果创建锁成功，返回锁文件对象，如果不需要锁（目标文件已经存在，不需要继续转换），则返回null.
	 * @throws FileConverterException 等待其他线程已经创建的锁释放时超时（超过转换任务的全局超时设置）
	 * @throws IOException 创建文件失败
	 */
	private File tryLock(final File srcFile, final File destFile, final OutputType outType) throws FileConverterException, IOException {
		File lockFile = new File(settings.getIncomingDir(),
										FileExtensionUtil.addExtension(srcFile.getName(), outType.toString() + FileConverterSettings.SUFFIX_LOCK));

		boolean lockSafe = waitLockRelease(lockFile);
		
		if (!lockSafe) {
			throw new FileConverterException("转换任务正在执行，请耐心等待！");
		}

		// 文件锁解除后，如果发现目标文件已经存在，则说明是刚刚才转换成功的，此时不再转换，直接返回true
		// 目前的设计思想是应用程序应自行记录或调用接口判断是否目标文件已经存在并决定是否进行转换，故一旦进行转换，即视为应用程序希望进行转换，不管是否已经转换过。
		// 仅当多线程转换同一个文件同一个目标类型时，因应用程序无法判断是否已经转换，此时转换任务会如下进行去重优化。
		if (destFile.exists() && destFile.isFile()) {
			logger.info("转换目标文件刚刚被另一个线程转换成功，本次转换中止并视为成功。{}", destFile.getAbsolutePath());
			return null;
		}
		
		// 转换目标也可能是以文件夹形式存在的子目录
		File destDir = new File(destFile.getParentFile(), destFile.getName() + FileConverterSettings.PAGED_FILE_DIR_SUFFIX);
		if (destDir.exists() && destDir.isDirectory()) {
			logger.info("转换目标文件刚刚被另一个线程转换成功，本次转换中止并视为成功。{}", destFile.getAbsolutePath());
			return null;
		}
		
		try {
			// 创建新锁，确保锁的生成时间是当前时间
			boolean createdByMe = false;
			synchronized (lock) {
				if (!lockFile.exists()) {
					FileUtils.touch(lockFile);
					createdByMe = true;
					logger.debug("创建文件锁成功：{}", lockFile.getAbsolutePath());
				}
			}

			if (!createdByMe) {
				// 被其他线程抢先进行该任务，继续等待重试
				return tryLock(srcFile, destFile, outType);
			} else {
				return lockFile;
			}
		} catch (IOException e) {
			logger.error("创建文件锁失败：" + lockFile.getAbsolutePath(), e);
			throw e;
		}
		
	}

	/**
	 * 等待锁文件释放。如果锁文件是很久以前的，则认为已经失效并删除锁。
	 * 如果等待时间超过阀值，则抛出DocConvertException。
	 * @param lockFile
	 * @return 锁不存在或等待释放成功则返回true，锁存在且未能等到释放返回false
	 */
	private boolean waitLockRelease(File lockFile) {
		if (!lockFile.exists() || !lockFile.isFile()) {
			return true;
		}
		
		long modifyTime = lockFile.lastModified();
		long now = System.currentTimeMillis();
		if (now - modifyTime > FileConverterSettings.CONVERT_LOCK_EXPIRED_MINUTES) {
			logger.warn("文件转换锁{}的存在时间已经超过预设阀值，判定为失效，将被删除。", lockFile);
			FileUtils.deleteQuietly(lockFile);
		} else {
			// 锁文件已经存在，且是新鲜的，说明有转换进程正在进行中，此时需要等待。
			long start = System.currentTimeMillis();
			long end = System.currentTimeMillis();
			logger.warn("文件锁已经存在，可能有转换任务正在进行。开始等待... 文件锁：{}", lockFile.getAbsolutePath());
			
			try {
				while (lockFile.exists() && (end - start) < FileConverterSettings.CONVERT_TASK_TIMEOUT_MINUTES * 60 *1000) {
					Thread.sleep(1000);
					end += 1000;
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
			logger.info("文件锁等待结束");
			if (lockFile.exists()) {
				String errMsg = "等待释放文件锁超时(" + lockFile.getAbsolutePath() + ")。可能转换过程较慢，请稍后再试，或手工删除文件锁后再试。当前转换任务退出。";
				logger.error(errMsg);
				return false;
			}
		}
		
		return true;
	}
	
}