package com.kdgcsoft.dtp.plugin.reader.databasereader.service.impl;

import com.kdgcsoft.dtp.plugin.common.databasecommon.exception.JdbcQueryException;
import com.kdgcsoft.dtp.plugin.common.databasecommon.util.DBUtil;
import com.kdgcsoft.dtp.plugin.common.databasecommon.util.DatasourceProxy;
import com.kdgcsoft.dtp.plugin.reader.databasereader.entity.QueryEntity;
import com.kdgcsoft.dtp.plugin.reader.databasereader.entity.ReaderResult;
import com.kdgcsoft.dtp.plugin.reader.databasereader.handle.QueryExceptionHandler;
import com.kdgcsoft.dtp.plugin.reader.databasereader.handle.impl.QueryExceptionHandlerImpl;
import com.kdgcsoft.dtp.plugin.reader.databasereader.service.AbstractBlockWriteStreamWriter;
import com.kdgcsoft.dtp.plugin.reader.databasereader.service.DataReader;
import com.kdgcsoft.dtp.plugin.reader.databasereader.service.ResultSetHandler;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.io.IOException;
import java.sql.*;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

public class StringDataBaseReader implements DataReader<String[], AbstractBlockWriteStreamWriter> {


    //SQL批处理数量
    private static final int SQL_BATCH_COUNT = 800;
    private QueryExceptionHandler exceptionHandler = new QueryExceptionHandlerImpl();

    @Override
    public ReaderResult query(QueryEntity queryEntity, AbstractBlockWriteStreamWriter writer,
                              ResultSetHandler<String[]> handler) throws SQLException {

        Connection connection = null;
        PreparedStatement stmt = null;
        ResultSet resultSet = null;

        validateQueryEntity(queryEntity);
        try {
            DataSource dataSource = queryEntity.getDataSource();
            connection = DatasourceProxy.getConnection(dataSource);
            String queryString = queryEntity.getQueryString();
            stmt = connection.prepareStatement(queryString);

            if (connection.getMetaData().getDatabaseProductName().equals("MYSQL")) {
                stmt.setFetchSize(Integer.MIN_VALUE);
            } else {
                stmt.setFetchSize(SQL_BATCH_COUNT);
            }
            resultSet = stmt.executeQuery();
            return writeStream(resultSet, writer, handler);
        } catch (JdbcQueryException e) {
            exceptionHandler.handler(e);
        } finally {
            DBUtil.closeDBResources(resultSet, stmt, connection);
        }
        return new ReaderResult(0, false);
    }

    @Override
    public Future<ReaderResult> asyncQuery(QueryEntity queryEntity, AbstractBlockWriteStreamWriter write,
                                           ResultSetHandler<String[]> handler) {
        ExecutorService executor = Executors.newSingleThreadExecutor();
        return executor.submit(() -> query(queryEntity, write, handler));
    }


    private void validateQueryEntity(QueryEntity queryEntity) {
        if (queryEntity == null) {
            throw new RuntimeException("QueryEntity 查询对象不能为空。");
        }
        DataSource source = queryEntity.getDataSource();
        if (Objects.isNull(source)) {
            throw new RuntimeException("数据库插件插件中数据源不能为空");
        }
        String tableName = queryEntity.getTableName();
        String queryString = queryEntity.getQueryString();
        if (StringUtils.isEmpty(tableName) && StringUtils.isEmpty(queryString)) {
            throw new RuntimeException("数据库查询插件数据库表或查询sql必须有一个不能为空");
        }

    }

    private ReaderResult writeStream(ResultSet resultSet, AbstractBlockWriteStreamWriter writer,
                                     ResultSetHandler<String[]> handler) {
        long total = 0;
        try {
            writer.writeHead(getHeaders(resultSet));
            while (resultSet.next()) {
                final String[] values = handler.handle(resultSet);
                writer.writeBody(values);
                total++;
            }
            writer.finish();
        } catch (JdbcQueryException | SQLException e) {
            throw new JdbcQueryException(e,total);
        }
        return new ReaderResult(total, true);
    }


    private String[] getHeaders(ResultSet resultSet) throws SQLException {
        ResultSetMetaData metaData = resultSet.getMetaData();
        String[] headers = new String[metaData.getColumnCount()];
        for (int i = 0; i < metaData.getColumnCount(); i++) {
            headers[i] = metaData.getColumnName(i+1);
        }
        return headers;
    }


    public void setExceptionHandler(QueryExceptionHandler exceptionHandler) {
        this.exceptionHandler = exceptionHandler;
    }

}
