/*
 * Decompiled with CFR 0.152.
 */
package com.github.signaflo.math.optim;

import com.github.signaflo.math.Real;
import com.github.signaflo.math.function.AbstractFunction;
import com.github.signaflo.math.function.Function;
import com.github.signaflo.math.function.SlopeFunction;
import com.github.signaflo.math.optim.CubicInterpolation;
import com.github.signaflo.math.optim.NaNStepLengthException;
import com.github.signaflo.math.optim.QuadraticInterpolation;

final class StrongWolfeLineSearch {
    private static final int MAX_UPDATE_ITERATIONS = 40;
    private static final double DELTA_MAX = 4.0;
    private static final double DELTA_MIN = 0.5833333333333334;
    private static final double EPSILON = Math.ulp(1.0);
    private final AbstractFunction phi;
    private final double c1;
    private final double c2;
    private final double f0;
    private final double slope0;
    private final double alphaMax;
    private final Function psi;
    private final SlopeFunction dPsi;
    private double alphaLower;
    private double psiAlphaLower;
    private double dPsiAlphaLower;
    private double alphaUpper;
    private double psiAlphaUpper;
    private double dPsiAlphaUpper;
    private double alphaT;
    private double psiAlphaT;
    private double dPsiAlphaT;
    private final double tolerance;

    private StrongWolfeLineSearch(Builder builder) {
        this.phi = builder.phi;
        this.c1 = builder.c1;
        this.c2 = builder.c2;
        this.f0 = builder.f0;
        this.slope0 = builder.slope0;
        this.alphaMax = builder.alphaMax;
        this.alphaT = builder.alpha0;
        this.psi = alpha -> this.phi.at(alpha) - this.f0 - this.c1 * this.slope0 * alpha;
        this.dPsi = (alpha, f) -> this.phi.slopeAt(alpha, f) - this.c1 * this.slope0;
        this.psiAlphaT = this.psi.at(this.alphaT);
        this.dPsiAlphaT = this.dPsi.at(this.alphaT, this.psiAlphaT + this.f0 + this.c1 * this.slope0 * this.alphaT);
        this.alphaLower = 0.0;
        this.psiAlphaLower = 0.0;
        this.dPsiAlphaLower = this.slope0 * (1.0 - this.c1);
        this.tolerance = 1.0E-8;
    }

    static Builder newBuilder(AbstractFunction f, double f0, double slope0) {
        return new Builder(f, f0, slope0);
    }

    double search() {
        if (this.psiAlphaT <= this.tolerance && Math.abs(this.dPsiAlphaT + this.c1 * this.slope0) - this.c2 * Math.abs(this.slope0) < this.tolerance) {
            return this.alphaT;
        }
        Real.Interval initialInterval = this.getInitialInterval();
        if (initialInterval.endpointsEqual(EPSILON)) {
            return initialInterval.lowerDbl();
        }
        return this.zoom();
    }

    private double zoom() {
        double oldIntervalLength = Math.abs(this.alphaUpper - this.alphaLower);
        int trials = 0;
        for (int k = 1; k < 40; ++k) {
            while (Double.isInfinite(this.psiAlphaUpper) && k < 40) {
                this.alphaUpper = 0.5 * this.alphaUpper;
                this.psiAlphaUpper = this.psi.at(this.alphaUpper);
                this.dPsiAlphaUpper = this.dPsi.at(this.alphaUpper, this.psiAlphaUpper + this.f0 + this.c1 * this.slope0 * this.alphaUpper);
                ++k;
            }
            double newIntervalLength = Math.abs(this.alphaUpper - this.alphaLower);
            if (Math.abs((newIntervalLength - oldIntervalLength) / oldIntervalLength) < 0.667 && trials > 1) {
                this.alphaT = Math.abs(this.alphaLower + this.alphaUpper) / 2.0;
                trials = 1;
                oldIntervalLength = newIntervalLength;
            } else {
                this.alphaT = this.getTrialValue(this.alphaLower, this.alphaUpper, this.psiAlphaLower, this.psiAlphaUpper, this.dPsiAlphaLower, this.dPsiAlphaUpper);
                ++trials;
            }
            if (Double.isNaN(this.alphaT)) {
                throw new NaNStepLengthException("The step length in Strong Wolfe line search was NaN");
            }
            this.psiAlphaT = this.psi.at(this.alphaT);
            this.dPsiAlphaT = this.dPsi.at(this.alphaT, this.psiAlphaT + this.f0 + this.c1 * this.slope0 * this.alphaT);
            if (this.psiAlphaT <= 0.0 && Math.abs(this.dPsiAlphaT - this.c1 * this.slope0) - this.c2 * Math.abs(this.slope0) < 0.0) {
                return this.alphaT;
            }
            this.updateInterval(this.alphaLower, this.alphaT, this.alphaUpper, this.psiAlphaLower, this.psiAlphaT, this.dPsiAlphaT);
            if (this.alphaLower != this.alphaUpper) continue;
            return this.alphaLower;
        }
        return this.alphaT;
    }

    private void updateInterval(double alphaLower, double alphaK, double alphaUpper, double psiAlphaLower, double psiAlphaK, double dPsiAlphaK) {
        if (psiAlphaLower > this.psiAlphaUpper || psiAlphaLower > 0.0 || this.dPsiAlphaLower * (alphaUpper - alphaLower) >= 0.0) {
            throw new RuntimeException("The assumptions of Theorem 2.1 (More-Thuente 1994) are not met.");
        }
        if (psiAlphaK > psiAlphaLower) {
            this.alphaUpper = alphaK;
            this.psiAlphaUpper = psiAlphaK;
            this.dPsiAlphaUpper = dPsiAlphaK;
        } else if (dPsiAlphaK * (alphaLower - alphaK) > 0.0) {
            this.alphaLower = alphaK;
            this.psiAlphaLower = psiAlphaK;
            this.dPsiAlphaLower = dPsiAlphaK;
        } else if (dPsiAlphaK * (alphaLower - alphaK) < 0.0) {
            this.alphaUpper = alphaLower;
            this.psiAlphaUpper = psiAlphaLower;
            this.dPsiAlphaUpper = this.dPsiAlphaLower;
            this.alphaLower = alphaK;
            this.psiAlphaLower = psiAlphaK;
            this.dPsiAlphaLower = dPsiAlphaK;
        } else {
            this.alphaT = this.alphaLower = this.alphaUpper;
        }
    }

    private Real.Interval getInitialInterval() {
        for (int k = 0; k < 1000; ++k) {
            if (this.psiAlphaT > this.psiAlphaLower) {
                this.alphaUpper = this.alphaT;
                this.psiAlphaUpper = this.psiAlphaT;
                this.dPsiAlphaUpper = this.dPsiAlphaT;
                return new Real.Interval(this.alphaLower, this.alphaT);
            }
            if (this.dPsiAlphaT * (this.alphaLower - this.alphaT) > 0.0) {
                double priorAlphaLower = this.alphaLower;
                this.alphaLower = this.alphaT;
                this.psiAlphaLower = this.psiAlphaT;
                this.dPsiAlphaLower = this.dPsiAlphaT;
                if (this.psiAlphaT <= 0.0 && this.dPsiAlphaT < 0.0) {
                    this.alphaT = Math.min(this.alphaT + 4.0 * (this.alphaT - priorAlphaLower), this.alphaMax);
                    this.psiAlphaT = this.psi.at(this.alphaT);
                    this.dPsiAlphaT = this.dPsi.at(this.alphaT, this.psiAlphaT + this.f0 + this.c1 * this.slope0 * this.alphaT);
                }
                if (this.alphaT != this.alphaMax) continue;
                return new Real.Interval(this.alphaMax, this.alphaMax);
            }
            if (this.dPsiAlphaT * (this.alphaLower - this.alphaT) < 0.0) {
                this.alphaUpper = this.alphaLower;
                this.psiAlphaUpper = this.psiAlphaLower;
                this.dPsiAlphaUpper = this.dPsiAlphaLower;
                this.alphaLower = this.alphaT;
                this.psiAlphaLower = this.psiAlphaT;
                this.dPsiAlphaLower = this.dPsiAlphaT;
                return new Real.Interval(this.alphaT, this.alphaLower);
            }
            return new Real.Interval(this.alphaT, this.alphaT);
        }
        throw new RuntimeException("Couldn't find an interval containing a point satisfying the Strong Wolfe conditions.");
    }

    private double getTrialValue(double alphaL, double alphaT, double fAlphaL, double fAlphaT, double dAlphaL, double dAlphaT) {
        double alphaC = CubicInterpolation.minimum(alphaL, alphaT, fAlphaL, fAlphaT, dAlphaL, dAlphaT);
        double alphaQ = QuadraticInterpolation.minimum(alphaL, alphaT, fAlphaL, fAlphaT, dAlphaL);
        return Math.abs(alphaC - alphaL) < Math.abs(alphaQ - alphaL) ? alphaC : 0.5 * (alphaQ + alphaC);
    }

    static class Builder {
        private final AbstractFunction phi;
        private final double f0;
        private final double slope0;
        private double c1 = 0.001;
        private double c2 = 0.5;
        private double alphaMax = 1000.0;
        private double alpha0 = 1.0;

        Builder(AbstractFunction phi, double f0, double slope0) {
            this.phi = phi;
            this.f0 = f0;
            this.slope0 = slope0;
        }

        final Builder c1(double c1) {
            this.c1 = c1;
            return this;
        }

        final Builder c2(double c2) {
            this.c2 = c2;
            return this;
        }

        final Builder alphaMax(double alphaMax) {
            this.alphaMax = alphaMax;
            return this;
        }

        final Builder alpha0(double alpha0) {
            this.alpha0 = alpha0;
            return this;
        }

        public final StrongWolfeLineSearch build() {
            return new StrongWolfeLineSearch(this);
        }
    }
}

