/*
 * Decompiled with CFR 0.152.
 */
package com.crabshue.commons.file.nio;

import com.crabshue.commons.exceptions.ApplicationException;
import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.exceptions.context.ErrorContext;
import com.crabshue.commons.exceptions.context.ErrorType;
import com.crabshue.commons.file.nio.exceptions.FileErrorContext;
import com.crabshue.commons.file.nio.exceptions.FileErrorType;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemUtils {
    private static Logger logger = LoggerFactory.getLogger(FileSystemUtils.class);

    protected FileSystemUtils() {
    }

    public static Path provideFolder(Path folderPath) {
        Validate.notNull((Object)folderPath);
        if (Files.notExists(folderPath, new LinkOption[0])) {
            try {
                Files.createDirectories(folderPath, new FileAttribute[0]);
                logger.info("Folder created [{}]", (Object)folderPath);
            }
            catch (IOException e) {
                throw new SystemException((ErrorType)FileErrorType.ERROR_CREATING_FOLDER, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath);
            }
        }
        return folderPath;
    }

    public static Path provideSubFolder(Path parentPath, String subfolder) {
        Validate.notNull((Object)parentPath);
        Validate.notNull((Object)subfolder);
        Path ret = parentPath.resolve(subfolder);
        return FileSystemUtils.provideFolder(ret);
    }

    public static Path retrieveFileInFolder(String filename, Path folderPath) {
        Validate.notNull((Object)folderPath);
        Validate.notNull((Object)filename);
        Path ret = folderPath.resolve(filename);
        if (Files.notExists(ret, new LinkOption[0]) || !Files.isRegularFile(ret, new LinkOption[0])) {
            throw new ApplicationException((ErrorType)FileErrorType.NO_FILE_FOUND, "No file found in the folder with given name").addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath).addContextValue((ErrorContext)FileErrorContext.FILENAME, (Object)filename);
        }
        logger.info("File [{}] found in folder [{}]", (Object)ret, (Object)folderPath);
        return ret;
    }

    public static Path copyFile(Path filePath, Path outputFilePath) {
        Path ret;
        try {
            ret = Files.copy(filePath, outputFilePath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_COPY_FILE, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)filePath).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)outputFilePath);
        }
        return ret;
    }

    public static Path copyFileToFolder(Path filePath, Path targetFolderPath) {
        Path ret;
        Validate.notNull((Object)targetFolderPath);
        Validate.notNull((Object)filePath);
        if (!Files.isDirectory(targetFolderPath, new LinkOption[0])) {
            throw new IllegalArgumentException("target is not a directory " + targetFolderPath);
        }
        if (!Files.isRegularFile(filePath, new LinkOption[0])) {
            throw new IllegalArgumentException("Given file is not a file " + filePath);
        }
        Path targetPath = targetFolderPath.resolve(filePath.getFileName());
        try {
            ret = Files.copy(filePath, targetPath, new CopyOption[0]);
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_COPY_FILE, "Cannot copy file", (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)filePath).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)targetPath);
        }
        logger.info("Copied file [{}] to [{}}", (Object)filePath, (Object)ret);
        return ret;
    }

    public static Path copyFolder(Path sourceFolderPath, Path targetFolderPath) {
        Validate.notNull((Object)sourceFolderPath);
        Validate.notNull((Object)targetFolderPath);
        if (!Files.isDirectory(sourceFolderPath, new LinkOption[0])) {
            throw new IllegalArgumentException("at least one of the parameters is not a directory");
        }
        try {
            Files.walkFileTree(sourceFolderPath, new DuplicateFileVisitor(sourceFolderPath, targetFolderPath));
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_COPY_FILE, "Cannot copy folder", (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)sourceFolderPath).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)targetFolderPath);
        }
        logger.info("Copied folder [{}] to [{}}", (Object)sourceFolderPath, (Object)targetFolderPath);
        return targetFolderPath;
    }

    public static boolean isFolderEmpty(Path folderPath) {
        Validate.notNull((Object)folderPath);
        try {
            return Files.list(folderPath).count() <= 0L;
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_READING_FILE, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath);
        }
    }

    public static Path retrieveOrCreateFile(Path folderPath, String filename) {
        Validate.notNull((Object)folderPath);
        Validate.notNull((Object)filename);
        Path ret = folderPath.resolve(filename);
        return FileSystemUtils.retrieveOrCreateFile(ret);
    }

    public static Path retrieveOrCreateFile(Path filePath) {
        Validate.notNull((Object)filePath);
        if (Files.notExists(filePath, new LinkOption[0])) {
            try {
                Path parentPath = filePath.getParent();
                if (parentPath != null) {
                    FileSystemUtils.provideFolder(parentPath);
                }
                Files.createFile(filePath, new FileAttribute[0]);
                logger.info("Created file [{}]", (Object)filePath);
            }
            catch (IOException e) {
                throw new SystemException((ErrorType)FileErrorType.ERROR_WRITING_FILE, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)filePath);
            }
        } else {
            logger.info("Retrieved file [{}]", (Object)filePath);
        }
        return filePath;
    }

    public static boolean isFileEmpty(Path filePath) {
        try {
            return Files.size(filePath) == 0L;
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_READING_FILE, "Cannot read file", (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FILE, (Object)filePath);
        }
    }

    public static void cleanDirectory(Path folderPath) {
        if (Files.notExists(folderPath, new LinkOption[0])) {
            logger.info("Folder [{}] does not exist. No need to clean.", (Object)folderPath);
            return;
        }
        try {
            DirectoryStream<Path> paths = Files.newDirectoryStream(folderPath);
            paths.forEach(path -> {
                try {
                    if (Files.isRegularFile(path, new LinkOption[0])) {
                        Files.delete(path);
                    } else {
                        FileSystemUtils.cleanDirectory(path);
                        Files.deleteIfExists(path);
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
            });
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_CLEANING_FOLDER, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath);
        }
    }

    public static Collection<Path> listFiles(Path folderPath) {
        Validate.notNull((Object)folderPath);
        ArrayList<Path> ret = new ArrayList<Path>();
        try {
            Files.newDirectoryStream(folderPath).forEach(item -> {
                if (Files.isRegularFile(item, new LinkOption[0])) {
                    ret.add((Path)item);
                } else {
                    ret.addAll(FileSystemUtils.listFiles(item));
                }
            });
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_READING_FOLDER, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath);
        }
        return ret;
    }

    public static Collection<Path> listFiles(Path folderPath, String glob) {
        Validate.notNull((Object)folderPath);
        Validate.notNull((Object)glob);
        ArrayList<Path> ret = new ArrayList<Path>();
        try {
            Files.newDirectoryStream(folderPath, glob).forEach(ret::add);
        }
        catch (IOException e) {
            throw new SystemException((ErrorType)FileErrorType.ERROR_READING_FOLDER, (Throwable)e).addContextValue((ErrorContext)FileErrorContext.FOLDER, (Object)folderPath);
        }
        return ret;
    }

    private static class DuplicateFileVisitor
    implements FileVisitor<Path> {
        private final Path sourceFolderPath;
        private final Path targetFolderPath;

        public DuplicateFileVisitor(Path sourceFolderPath, Path targetFolderPath) {
            this.sourceFolderPath = sourceFolderPath;
            this.targetFolderPath = targetFolderPath;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            Path pathToFolder = this.sourceFolderPath.relativize(dir);
            Path outputFolderPath = this.targetFolderPath.resolve(pathToFolder);
            FileSystemUtils.provideFolder(outputFolderPath);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            Path pathToFile = this.sourceFolderPath.relativize(file);
            Path outputFilePath = this.targetFolderPath.resolve(pathToFile);
            FileSystemUtils.copyFile(file, outputFilePath);
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
            return FileVisitResult.TERMINATE;
        }

        @Override
        public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
            return FileVisitResult.CONTINUE;
        }
    }
}

