package org.flywaydb.core.internal.database.kingbase;

import org.flywaydb.core.api.configuration.Configuration;
import org.flywaydb.core.internal.database.base.Database;
import org.flywaydb.core.internal.database.base.Table;
import org.flywaydb.core.internal.jdbc.JdbcConnectionFactory;
import org.flywaydb.core.internal.jdbc.StatementInterceptor;
import org.flywaydb.core.internal.util.StringUtils;

import java.sql.Connection;
import java.sql.SQLException;

/**
 * @author xu.wenchang
 * @version 1.0 2023/10/17
 */
public class KingbaseDatabase extends Database<KingbaseConnection> {
    public KingbaseDatabase(Configuration configuration, JdbcConnectionFactory jdbcConnectionFactory,
                            StatementInterceptor statementInterceptor) {
        super(configuration, jdbcConnectionFactory, statementInterceptor);
    }
    
    @Override
    protected KingbaseConnection doGetConnection(Connection connection) {
        return new KingbaseConnection(this, connection);
    }
    
    @Override
    public void ensureSupported() {
        ensureDatabaseIsRecentEnough("9.0");
        
        ensureDatabaseNotOlderThanOtherwiseRecommendUpgradeToFlywayEdition("10", org.flywaydb.core.internal.license.Edition.ENTERPRISE);
        
        recommendFlywayUpgradeIfNecessaryForMajorVersion("14");
    }
    
    @Override
    public String getRawCreateScript(Table table, boolean baseline) {
        String tablespace = configuration.getTablespace() == null ? "" : " TABLESPACE \"" + configuration.getTablespace() + "\"";
        
        return "CREATE TABLE " + table + " (\n" + "    \"installed_rank\" INT NOT NULL,\n" + "    \"version\" VARCHAR(50),\n" + "    \"description\" VARCHAR(200) NOT NULL,\n" + "    \"type\" VARCHAR(20) NOT NULL,\n" + "    \"script\" VARCHAR(1000) NOT NULL,\n" + "    \"checksum\" INTEGER,\n" + "    \"installed_by\" VARCHAR(100) NOT NULL,\n" + "    \"installed_on\" TIMESTAMP NOT NULL DEFAULT now(),\n" + "    \"execution_time\" INTEGER NOT NULL,\n" + "    \"success\" BOOLEAN NOT NULL\n" + ")" + tablespace + ";\n" + (baseline ? getBaselineStatement(table) + ";\n" : "") + "ALTER TABLE " + table + " ADD CONSTRAINT \"" + table.getName() + "_pk\" PRIMARY KEY (\"installed_rank\");\n" + "CREATE INDEX \"" + table.getName() + "_s_idx\" ON " + table + " (\"success\");";
    }
    
    @Override
    protected String doGetCurrentUser() throws SQLException {
        return getMainConnection().getJdbcTemplate()
                                  .queryForString("SELECT current_user");
    }
    
    @Override
    public boolean supportsDdlTransactions() {
        return true;
    }
    
    @Override
    public boolean supportsChangingCurrentSchema() {
        return true;
    }
    
    @Override
    public String getBooleanTrue() {
        return "TRUE";
    }
    
    @Override
    public String getBooleanFalse() {
        return "FALSE";
    }
    
    @Override
    public String doQuote(String identifier) {
        return getOpenQuote() + StringUtils.replaceAll(identifier, getCloseQuote(), getEscapedQuote()) + getCloseQuote();
    }
    
    @Override
    public String getEscapedQuote() {
        return "\"\"";
    }
    
    @Override
    public boolean catalogIsSchema() {
        return false;
    }
    
    @Override
    public boolean useSingleConnection() {
        return true;
    }
    
    /**
     * This exists to fix this issue: https://github.com/flyway/flyway/issues/2638
     * See https://www.pgpool.net/docs/latest/en/html/runtime-config-load-balancing.html
     */
    @Override
    public String getSelectStatement(Table table) {
        return "/*NO LOAD BALANCE*/\n" + "SELECT " + quote("installed_rank") + "," + quote("version") + "," + quote("description") + "," + quote("type") + "," + quote("script") + "," + quote("checksum") + "," + quote("installed_on") + "," + quote("installed_by") + "," + quote("execution_time") + "," + quote("success") + " FROM " + table + " WHERE " + quote("installed_rank") + " > ?" + " ORDER BY " + quote("installed_rank");
    }
}